# Hookify Patterns and Examples

## Regex Quick Reference

| Pattern | Matches |
|---------|---------|
| `rm\s+-rf` | rm -rf, rm  -rf |
| `console\.log\(` | console.log( |
| `(eval\|exec)\(` | eval( or exec( |
| `chmod\s+777` | chmod 777 |
| `API_KEY\s*=` | API_KEY=, API_KEY = |

**Escape special characters:** `.` → `\.`, `(` → `\(`, `[` → `\[`

**Metacharacters:** `\s` (whitespace), `\d` (digit), `\w` (word char), `.` (any), `+` (1+), `*` (0+), `?` (0-1), `|` (OR)

## Common Pitfalls

**Too broad:** `pattern: log` matches "log", "login", "dialog", "catalog"
Better: `console\.log\(|logger\.`

**Too specific:** `pattern: rm -rf /tmp` only matches exact path
Better: `rm\s+-rf`

**YAML quoting:** Unquoted patterns work as-is. Quoted strings need double backslashes.

## Testing Patterns

```bash
python3 -c "import re; print(re.search(r'your_pattern', 'test text'))"
```

## Example Rules

### Block dangerous rm

```markdown
---
name: block-dangerous-rm
enabled: true
event: bash
action: block
pattern: rm\s+-rf\s+/
---
Blocked: recursive deletion of absolute path.
```

### Warn console.log in TypeScript

```markdown
---
name: warn-console-log
enabled: true
event: file
conditions:
  - field: file_path
    operator: regex_match
    pattern: \.tsx?$
  - field: new_text
    operator: regex_match
    pattern: console\.log\(
---
Debug logging detected in TypeScript. Remove before committing.
```

### Completion checklist

```markdown
---
name: require-verification
enabled: true
event: stop
pattern: .*
---
Before stopping, verify:
- [ ] Tests were run
- [ ] Build succeeded
- [ ] Documentation updated
```

### Sensitive file warning

```markdown
---
name: warn-env-edits
enabled: true
event: file
conditions:
  - field: file_path
    operator: regex_match
    pattern: \.env$
  - field: new_text
    operator: contains
    pattern: API_KEY
---
Adding API key to .env file. Ensure this file is in .gitignore.
```
