import argparse
import json
import os
from datetime import datetime
from jinja2 import Template
from xhtml2pdf import pisa

# 语言映射表：把 .py 映射为 Python，.go 映射为 Go
LANG_MAP = {
    '.py': 'Python', '.go': 'Go', '.rs': 'Rust', 
    '.js': 'JavaScript', '.ts': 'TypeScript', 
    '.java': 'Java', '.c': 'C', '.cpp': 'C++',
    '.html': 'HTML', '.css': 'CSS', '.md': 'Markdown',
    '.sql': 'SQL', '.sh': 'Shell'
}

class LanguageAnalyzer:
    def __init__(self, root_path, ignore_folders):
        self.root_path = os.path.abspath(root_path)
        self.ignore_set = set(ignore_folders.split(',')) if ignore_folders else set()
        
    def count_lines(self, filepath):
        count = 0
        try:
            with open(filepath, 'r', encoding='utf-8', errors='ignore') as f:
                for line in f:
                    if line.strip(): count += 1
        except: return 0
        return count

    def scan(self):
        # 数据结构：{'Python': {'files': 10, 'lines': 500}, 'Go': ...}
        stats = {}
        total_lines = 0

        for root, dirs, files in os.walk(self.root_path):
            dirs[:] = [d for d in dirs if d not in self.ignore_set]

            for file in files:
                ext = os.path.splitext(file)[1].lower()
                if ext in LANG_MAP:
                    lang_name = LANG_MAP[ext]
                    lines = self.count_lines(os.path.join(root, file))
                    
                    if lang_name not in stats:
                        stats[lang_name] = {'files': 0, 'lines': 0}
                    
                    stats[lang_name]['files'] += 1
                    stats[lang_name]['lines'] += lines
                    total_lines += lines
        
        # 转换为列表并按行数排序
        sorted_stats = [
            {"language": k, "files": v['files'], "lines": v['lines']} 
            for k, v in stats.items()
        ]
        sorted_stats.sort(key=lambda x: x['lines'], reverse=True)
        
        return sorted_stats, total_lines

def generate_pdf_report(analyzer_results, target_dir):
    lang_stats, total_lines = analyzer_results
    
    # 获取 assets 绝对路径
    base_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
    template_path = os.path.join(base_dir, 'assets', 'summary_report.html')
    output_path = os.path.join(target_dir, f"language_summary_{datetime.now().strftime('%Y%m%d')}.pdf")

    try:
        with open(template_path, 'r', encoding='utf-8') as f:
            template = Template(f.read())
            
        html_content = template.render(
            project_path=target_dir,
            date=datetime.now().strftime("%Y-%m-%d"),
            total_lines=total_lines,
            lang_stats=lang_stats
        )
        
        with open(output_path, "wb") as pdf_file:
            pisa_status = pisa.CreatePDF(html_content, dest=pdf_file)
            
        if pisa_status.err: return {"status": "error", "message": "PDF error"}
            
        return {
            "status": "success",
            "report_path": output_path,
            "summary": {item['language']: item['lines'] for item in lang_stats}
        }
    except Exception as e:
        return {"status": "error", "message": str(e)}

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--path", required=True)
    parser.add_argument("--ignore", default="")
    args = parser.parse_args()

    analyzer = LanguageAnalyzer(args.path, args.ignore)
    results = analyzer.scan()
    print(json.dumps(generate_pdf_report(results, analyzer.root_path), indent=2))