# フィードバック対応ガイド

## 概要

レビューコメントへの効果的な対応方法を解説します。建設的な議論、修正の優先順位付け、コミュニケーションのベストプラクティスについて学びます。

## 目次

1. [コメントへの返信](#コメントへの返信)
2. [建設的な議論](#建設的な議論)
3. [修正の優先順位](#修正の優先順位)
4. [意見の相違への対処](#意見の相違への対処)
5. [Re-review依頼](#re-review依頼)

---

## コメントへの返信

### 返信の基本原則

#### 1. すべてのコメントに返信する

```markdown
## ❌ Bad: 無視する
レビュワー: 「この部分、エッジケースの考慮が必要では？」
作成者: （返信なし、修正もなし）

## ✅ Good: 必ず返信
レビュワー: 「この部分、エッジケースの考慮が必要では？」
作成者: 「ご指摘ありがとうございます！
空配列の場合の処理を追加しました。
コミット: abc123」
```

#### 2. 感謝の気持ちを示す

```markdown
## ✅ 返信例

### 問題指摘に対して
「ご指摘ありがとうございます！確かにその通りですね。
XXXの場合に問題が発生する可能性があります。
YYYのように修正しました。」

### 改善提案に対して
「素晴らしい提案ありがとうございます！
その方法の方が明らかに読みやすいですね。
そのように修正しました。」

### 賞賛に対して
「ありがとうございます！
XXXのパターンを使うと綺麗に書けることを学びました。」
```

#### 3. 修正内容を明確にする

```markdown
## ✅ Good: 何をしたか明確に

レビュワー: 「この関数は複雑すぎます」
作成者: 「ご指摘の通りです。以下のように分割しました:

- `validateOrder()`: 検証ロジック
- `calculateTotal()`: 計算ロジック
- `saveOrder()`: 永続化ロジック

コミット: abc123

各関数のテストも追加しました。
コミット: def456」
```

### 返信のタイミング

```markdown
## 優先度別対応時間

### 🔴 即座（30分以内）
- 質問（仕様確認など）
- クリティカルな問題の指摘
- ブロッカー

### 🟡 当日中
- バグ指摘
- セキュリティ問題
- 設計の問題

### 🟢 翌日まで
- 軽微な改善提案
- コードスタイル
- ドキュメント

## ❌ Bad: まとめて返信
3日後にすべてのコメントに一気に返信
→ レビュワーのコンテキストが失われる

## ✅ Good: 段階的に返信
質問 → 即座に回答
問題 → 修正後に報告
提案 → 検討後に返信
```

---

## 建設的な議論

### 質問への回答

```markdown
## レビュワーの質問
「なぜこの実装を選択したのですか？」

## ❌ Bad: 防御的
「これがベストな方法です」
「他の方法は試しました」
→ 議論を閉じてしまう

## ✅ Good: オープンに説明
「この実装を選択した理由は以下の通りです:

1. パフォーマンス: XXXと比較して2倍高速
2. 可読性: YYYのパターンより理解しやすい
3. 保守性: 将来の拡張が容易

ただし、ZZZの懸念もあることは認識しています。
他に良い方法があればご提案いただけると嬉しいです！」
→ 議論を促進
```

### 技術的議論

```markdown
## レビュワーの提案
「この部分、Strategyパターンを使った方が良いのでは？」

## ❌ Bad: 拒否
「今のままで十分です」
「複雑になりすぎます」

## ✅ Good: 検討して判断
「Strategyパターンのご提案ありがとうございます！

検討した結果:

### メリット
- 拡張性が向上
- テストしやすくなる

### デメリット
- 現時点では拡張予定なし
- ファイル数が増える

### 判断
今回は現在の実装で進めさせてください。
将来的に拡張が必要になったら、その時点で
Strategyパターンへリファクタリングします。

Issue #123 として記録しました。

いかがでしょうか？」
```

### 代替案の提示

```typescript
// レビュワーのコメント
// 「この実装はパフォーマンスが心配です」

// ❌ Bad: 修正のみ
// （コメントなしで修正してpush）

// ✅ Good: 説明と選択肢を提示
/*
返信:

ご懸念の通り、パフォーマンスを検証しました。
1000件のデータで以下の結果でした:

現在の実装: 150ms
提案1（Map使用）: 50ms
提案2（Set使用）: 45ms

提案2の方が若干速いですが、Mapの方が
データの取得も必要なため、提案1を採用したいと思います。

ベンチマークコード:
https://gist.github.com/...

いかがでしょうか？
*/

// 修正版（Map使用）
function filterUsers(users: User[], validIds: string[]): User[] {
  const validIdMap = new Map(validIds.map(id => [id, true]));
  return users.filter(user => validIdMap.has(user.id));
}
```

---

## 修正の優先順位

### 優先度の分類

```markdown
## P0: 即座に修正（必須）
- [ ] セキュリティ脆弱性
- [ ] データ損失の危険
- [ ] クリティカルなバグ
- [ ] アーキテクチャ違反

## P1: 今回のPRで修正
- [ ] バグ
- [ ] パフォーマンス問題
- [ ] テスト不足
- [ ] エラーハンドリング不足

## P2: 今回または次回で修正
- [ ] リファクタリング提案
- [ ] 可読性改善
- [ ] ドキュメント改善

## P3: 将来的に検討
- [ ] 大規模なリファクタリング
- [ ] アーキテクチャ変更
- [ ] パフォーマンス最適化（現時点で問題なし）
```

### 優先順位の返信例

```markdown
## レビューコメントが10個ある場合

### 返信例

ご確認ありがとうございます！
いただいたコメントを以下のように対応します:

### 即座に修正（今日中）
- [x] #1: SQL Injectionの脆弱性 → 修正完了（abc123）
- [x] #2: null参照のバグ → 修正完了（def456）
- [x] #3: テストケース追加 → 追加完了（ghi789）

### 今回のPRで修正（明日まで）
- [ ] #4: エラーハンドリング改善
- [ ] #5: 関数の分割
- [ ] #6: ドキュメント追加

### 次回のPRで対応
- [ ] #7: 大規模リファクタリング → Issue #456 作成
- [ ] #8: パフォーマンス最適化 → Issue #457 作成

### 確認事項
- #9: 仕様について確認させてください
- #10: XXXの意図はYYYで正しいでしょうか？

上記の対応方針で問題ないでしょうか？
```

---

## 意見の相違への対処

### パターン1: 技術的意見の相違

```markdown
## 状況
レビュワー: 「クラス名は動詞ではなく名詞にすべき」
作成者: 「この場合は動詞の方が適切だと思う」

## ❌ Bad: 押し通す
「私の方が正しいです」
「これで問題ありません」

## ✅ Good: データと論拠を示す
「ご指摘ありがとうございます。

この命名について、以下の理由で動詞を選択しました:

1. Google Style Guide では、サービスクラスは
   動詞でも可とされています
   参考: https://...

2. 既存コードベースの慣習
   - AuthenticateUser
   - ValidateEmail
   - ProcessPayment

3. クラスの役割
   このクラスは単一のアクションを実行するため、
   動詞の方が意図が明確です。

ただし、プロジェクトで統一すべきであれば、
すべてを名詞に変更することも検討します。

チームの方針はいかがでしょうか？」
```

### パターン2: スコープの相違

```markdown
## 状況
レビュワー: 「この機能も一緒に実装してください」
作成者: 「スコープが大きくなりすぎる」

## ✅ Good: 範囲を明確にする
「ご提案ありがとうございます。

今回のPRのスコープは以下の通りです:
- ユーザー登録機能
- メール認証

ご提案の機能（パスワードリセット）は
重要ですが、以下の理由で別PRにしたいです:

1. PR サイズ: 現在500行 → +300行で800行に
2. テスト: 追加で15テストケース必要
3. レビュー負荷: レビュー時間が2倍に

別PRにすることで:
- 今回: ユーザー登録を早くリリース
- 次回: パスワードリセットに集中

Issue #789 を作成しました。
次のスプリントで対応予定です。

この方針でよろしいでしょうか？」
```

### パターン3: アーキテクチャの相違

```markdown
## 状況
レビュワー: 「MVVMパターンに従うべき」
作成者: 「この画面はシンプルなのでMVCで十分」

## ✅ Good: チームで議論
「アーキテクチャについて、ご意見をいただき
ありがとうございます。

この判断について、チームで議論したいと思います。

### 論点
1. この画面の複雑度
   - 表示のみ（入力なし）
   - ビジネスロジックなし
   - 10行のコード

2. 一貫性 vs 柔軟性
   - すべてMVVM: 一貫性◎、柔軟性×
   - 場合によってMVC: 一貫性△、柔軟性◎

### 提案
チーム会議で以下を決定しませんか？
- シンプルな画面の基準
- アーキテクチャパターンの適用ルール

それまでこのPRは保留にします。

@team お時間いただけますでしょうか？」
```

### エスカレーション

```markdown
## いつエスカレートするか

### エスカレートすべき状況
1. 技術的意見が平行線
2. アーキテクチャの方針が不明確
3. プロジェクト全体に影響する決定

### エスカレート先
- テックリード
- アーキテクト
- チーム会議

### エスカレーション例

「@tech-lead

レビューで以下の点について意見が分かれています:
- 問題: XXX
- レビュワーの意見: YYY
- 私の意見: ZZZ

プロジェクト全体に影響する決定なので、
ご判断をいただけますでしょうか？

関連PR: #123
関連Issue: #456」
```

---

## Re-review依頼

### いつRe-review依頼するか

```markdown
## Re-review依頼のタイミング

### ✅ すぐにRe-review依頼
- すべてのコメントに対応済み
- すべての修正が完了
- CIが通っている

### ❌ 待つべき状況
- まだ対応中のコメントがある
- CIが失敗している
- 追加の変更を予定している

## 依頼方法

### ✅ Good: 変更内容を要約
「@reviewer

レビューありがとうございました！
すべてのコメントに対応しました。

### 主な変更点
1. SQL Injection対策（コミット: abc123）
   - プリペアドステートメント使用
   - 入力値検証追加

2. エラーハンドリング改善（コミット: def456）
   - カスタムエラークラス追加
   - エラーメッセージ詳細化

3. テスト追加（コミット: ghi789）
   - エッジケース: 5件
   - エラーケース: 3件
   - カバレッジ: 75% → 87%

4. ドキュメント更新（コミット: jkl012）
   - APIドキュメント
   - README

再度レビューをお願いできますでしょうか？」
```

### 部分的なRe-review

```markdown
## 一部のみ再レビューが必要な場合

「@reviewer

コメント#1, #2, #3について対応しました。

### 対応済み
- #1: セキュリティ問題 → 修正（abc123）
- #2: バグ → 修正（def456）
- #3: テスト → 追加（ghi789）

### 保留中
- #4: 大規模リファクタリング → Issue #456
- #5: パフォーマンス最適化 → Issue #457

#1-#3の修正について、ご確認いただけますでしょうか？
#4-#5は次のPRで対応予定です。」
```

---

## ベストプラクティス

### Do's

```markdown
✅ すべてのコメントに返信する
✅ 感謝の気持ちを示す
✅ 修正内容を明確にする
✅ 質問には具体的に答える
✅ 代替案を提示する
✅ データと根拠を示す
✅ 意見の相違は建設的に議論
✅ エスカレートすべき時はエスカレート
✅ Re-review依頼は変更内容を要約
```

### Don'ts

```markdown
❌ コメントを無視しない
❌ 防御的にならない
❌ 「これがベスト」と断言しない
❌ 議論を避けない
❌ 感情的にならない
❌ レビュワーを批判しない
❌ スコープを無限に広げない
❌ 修正なしでRe-review依頼しない
```

---

## 返信テンプレート

### 修正完了の返信

```markdown
ご指摘ありがとうございます！
修正しました。

**修正内容:**
[修正の詳細]

**コミット:** abc123

ご確認をお願いします。
```

### 質問への回答

```markdown
ご質問ありがとうございます！

[質問への回答]

追加で確認事項があれば、お知らせください。
```

### 提案の採用

```markdown
素晴らしい提案ありがとうございます！
そのように修正しました。

**コミット:** abc123

この方法の方が明らかに[理由]ですね。
勉強になりました！
```

### 提案の不採用

```markdown
ご提案ありがとうございます。

検討しましたが、以下の理由で今回は
現在の実装を維持したいと思います:

[理由1]
[理由2]

ただし、将来的には有用な改善だと思いますので、
Issue #XXX として記録しました。

この判断でよろしいでしょうか？
```

---

## まとめ

フィードバックへの適切な対応は、コードの品質を向上させ、チームの関係性を良好に保ちます。

### 重要ポイント

1. **すべてに返信する**
2. **感謝の気持ち**
3. **建設的な議論**
4. **優先順位付け**
5. **適切なエスカレーション**

### 次のステップ

- [コミュニケーション](11-communication.md)
- [自動化](12-automation.md)
