# レビュー実施ガイド

## 概要

効果的なコードレビューを実施するための実践的なガイドです。レビュープロセス、効率的なレビュー方法、タイムマネジメントについて解説します。

## 目次

1. [レビュープロセス](#レビュープロセス)
2. [効率的なレビュー方法](#効率的なレビュー方法)
3. [タイムマネジメント](#タイムマネジメント)
4. [コメントの書き方](#コメントの書き方)
5. [承認基準](#承認基準)
6. [よくある間違い](#よくある間違い)

---

## レビュープロセス

### レビューの6ステップ

#### 1. PRの理解（5分）

```markdown
## 確認項目

### PR説明を読む
- [ ] 何を変更するのか
- [ ] なぜ変更するのか
- [ ] 関連するIssue番号

### スコープを把握
- [ ] 変更ファイル数
- [ ] 変更行数
- [ ] 影響範囲

### 大きすぎるPRの場合
→ 分割を依頼するか、段階的レビュー
```

#### 2. ビルド・テストの確認（2分）

```bash
# CI/CDステータス確認
✅ Build: Success
✅ Tests: 95/95 passed
✅ Lint: No issues
✅ Coverage: 85% (+2%)

# ❌ CI失敗時
# レビューする前に修正を依頼
「CIが失敗しているようです。修正後に再度レビュー依頼をお願いします。」
```

#### 3. 全体構造の把握（10分）

```markdown
## 確認ポイント

### アーキテクチャ
- [ ] 適切なレイヤーに配置されているか
- [ ] 依存関係の方向は正しいか
- [ ] 新しいパターンを導入していないか

### ファイル構成
- [ ] 論理的な場所にファイルが配置されているか
- [ ] 命名規則に従っているか
- [ ] 不要なファイルがないか
```

#### 4. 詳細レビュー（20-30分）

7つの観点でレビュー:

1. **機能性**: 要件を満たしているか
2. **設計**: アーキテクチャに従っているか
3. **可読性**: 理解しやすいか
4. **テスト**: 十分なテストがあるか
5. **セキュリティ**: 脆弱性はないか
6. **パフォーマンス**: 効率的か
7. **保守性**: 将来の変更が容易か

#### 5. コメント作成（10分）

```markdown
## コメントの構造

### [必須] クリティカルな問題
- セキュリティ脆弱性
- バグ
- パフォーマンス問題

### [推奨] 改善提案
- リファクタリング
- 可読性向上
- ベストプラクティス

### [質問] 確認事項
- 意図の確認
- 仕様の確認

### [賞賛] 良い点
- 優れた実装
- 学びになる点
```

#### 6. 総合判断（5分）

```markdown
## 承認基準

### ✅ Approve
- すべての必須項目がOK
- 推奨項目は次のPRでもOK
- 全体的に品質が高い

### 🔄 Request Changes
- 必須項目に問題あり
- セキュリティ/バグの懸念
- アーキテクチャ違反

### 💬 Comment
- 質問のみ
- 軽微な提案のみ
- 情報共有
```

---

## 効率的なレビュー方法

### PRサイズによる戦略

#### Small PR (< 200 lines)

```markdown
時間: 10-15分

## アプローチ
1. 全ファイルを一度に読む
2. 細部まで確認
3. すべての行をチェック

## メリット
- 包括的なレビュー可能
- 見落としが少ない
- 迅速なフィードバック
```

#### Medium PR (200-500 lines)

```markdown
時間: 30-45分

## アプローチ
1. まず全体構造を把握
2. 重要なファイルから順に
3. テストコードは後回し

## 優先順位
1. ビジネスロジック
2. API/Controller
3. テストコード
4. ドキュメント
```

#### Large PR (> 500 lines)

```markdown
時間: 1-2時間（複数セッションに分割）

## アプローチ
1. PR作成者に分割を依頼
2. または段階的レビュー

## 段階的レビュー
### セッション1: アーキテクチャ（30分）
- 全体設計の確認
- ファイル構成の確認

### セッション2: コアロジック（30分）
- 重要な変更の詳細確認

### セッション3: その他（30分）
- テスト、ドキュメント
- 細部の確認
```

### 読む順序

```typescript
// ✅ Good: この順序で読む

// 1. テストから読む（仕様理解）
describe('UserService', () => {
  test('creates user with valid data', () => {
    // これで何をするコードか理解できる
  });
});

// 2. インターフェース/型定義
interface User {
  id: string;
  email: string;
  name: string;
}

// 3. public API
class UserService {
  createUser(data: CreateUserDTO): Promise<User> {
    // public なメソッドから読む
  }

  // 4. private メソッド
  private validateEmail(email: string): void {
    // 詳細は後で
  }
}
```

### ツールの活用

```bash
# GitHub CLI
gh pr view 123
gh pr diff 123
gh pr checks 123

# ローカルでレビュー
git fetch origin pull/123/head:pr-123
git checkout pr-123
npm install
npm test
npm run build

# エディタで開く
code .
```

---

## タイムマネジメント

### レビュー時間の目安

| PRサイズ | 行数 | 時間 | セッション |
|---------|------|------|-----------|
| XS | < 50 | 5分 | 1 |
| S | 50-200 | 15分 | 1 |
| M | 200-500 | 30-45分 | 1-2 |
| L | 500-1000 | 1-2時間 | 2-3 |
| XL | > 1000 | 分割依頼 | - |

### レビュータイミング

```markdown
## 優先度

### 🔴 高優先度（即座）
- Hotfix（本番障害対応）
- ブロッカー（他の開発者の進行を妨げる）
- セキュリティ修正

### 🟡 中優先度（当日中）
- 通常の機能追加
- バグ修正
- リファクタリング

### 🟢 低優先度（翌日以降OK）
- ドキュメント更新
- 軽微な改善
- コードスタイル修正
```

### 集中時間の確保

```markdown
## レビュー時間の確保方法

### ❌ Bad: 割り込みでレビュー
- Slackでレビュー依頼が来たらすぐ対応
- コーディング中に中断してレビュー
→ 効率が悪く、見落としが増える

### ✅ Good: 固定時間にレビュー
- 午前10:00-10:30: レビュー時間
- 午後14:00-14:30: レビュー時間
- 終業前17:00-17:30: レビュー時間

### メリット
- 集中してレビューできる
- 見落としが減る
- 計画的に時間を使える
```

---

## コメントの書き方

### コメントの種類

#### [必須] 修正が必要

```markdown
[必須] メモリリークの可能性

この実装だと、ViewControllerがDataLoaderを強参照し、
DataLoaderのクロージャがViewControllerを強参照するため、
循環参照が発生してメモリリークします。

修正案:
\`\`\`swift
dataLoader?.onComplete = { [weak self] in
    self?.updateUI()
}
\`\`\`

参考: https://docs.swift.org/...
```

#### [推奨] 改善提案

```markdown
[推奨] この関数は分割できます

現在60行ありますが、以下のように分割すると
テストしやすく、理解しやすくなります:

\`\`\`typescript
// Before
function processOrder(order: Order) {
  // 60行のロジック
}

// After
function processOrder(order: Order) {
  validateOrder(order);
  calculateTotal(order);
  saveOrder(order);
  sendNotification(order);
}
\`\`\`

ただし、urgent なら次のPRでもOKです。
```

#### [質問] 確認

```markdown
[質問] この仕様で良いですか？

ユーザーが18歳未満の場合、エラーを返していますが、
仕様書では「警告を表示して続行可能」となっています。

どちらが正しいでしょうか？
```

#### [賞賛] 良い点

```markdown
[賞賛] エラーハンドリングが素晴らしい！

各エラーケースがカスタムエラークラスで表現されていて、
エラーメッセージもユーザーフレンドリーですね。

このパターンは他の場所でも使えそうです 👍
```

### コメントのトーン

```markdown
## ❌ Bad: 攻撃的・上から目線

「なんでこんなコード書いたの？」
「これ間違ってるよ」
「こんなの常識でしょ」
「前も言ったけど」

## ✅ Good: 建設的・協力的

「この部分について確認させてください」
「こうするとより良くなると思います」
「一緒に考えてみましょう」
「こういう方法もあります」

## 具体例

### ❌ Bad
「このコード遅すぎる」

### ✅ Good
「パフォーマンスについて確認させてください。
大量データの場合、このO(n²)のアルゴリズムは
遅くなる可能性があります。

Setを使ったO(n)の実装も検討してみてはどうでしょうか？

\`\`\`typescript
// 現在の実装（O(n²)）
items.filter(item => validIds.includes(item.id))

// 提案（O(n)）
const validIdSet = new Set(validIds);
items.filter(item => validIdSet.has(item.id))
\`\`\`

ベンチマーク結果が必要であれば、測定しますよ！」
```

### コード例を含める

```markdown
## ❌ Bad: 説明のみ

「Optional bindingを使ってください」

## ✅ Good: コード例付き

「Optional bindingを使うことで、クラッシュを防げます:

\`\`\`swift
// 現在
let name = user!.name  // userがnilだとクラッシュ

// 提案
guard let user = user else { return }
let name = user.name  // 安全
\`\`\`
」
```

---

## 承認基準

### Approve すべき場合

```markdown
✅ 以下すべてを満たす場合

1. 機能性
   - 要件を満たしている
   - バグがない
   - エッジケースを考慮

2. 品質
   - テストがある（カバレッジ維持）
   - セキュリティ問題なし
   - パフォーマンス問題なし

3. 設計
   - アーキテクチャに従っている
   - 保守性が確保されている

4. スタイル
   - コーディング規約に従っている
   - Lintが通る

5. ドキュメント
   - 必要なドキュメントがある
```

### Request Changes すべき場合

```markdown
🔄 以下いずれかに該当する場合

1. 重大な問題
   - セキュリティ脆弱性
   - データ損失の危険
   - パフォーマンス劣化

2. バグ
   - 明らかなロジックエラー
   - エッジケースの考慮不足

3. アーキテクチャ違反
   - レイヤー構造の違反
   - 依存関係の問題

4. テスト不足
   - テストがない
   - カバレッジが大幅に低下
```

### Comment のみの場合

```markdown
💬 以下の場合

1. 質問のみ
   - 仕様確認
   - 意図確認

2. 軽微な提案
   - 変数名の提案
   - コメント追加の提案
   - 承認をブロックしない改善案

3. 情報共有
   - 参考情報の提供
   - 学びの共有
```

---

## よくある間違い

### 1. 完璧主義

```markdown
❌ Bad
「変数名がベストじゃない」
「この部分はもっと綺麗に書ける」
「100%完璧にしてから承認」
→ レビューが遅くなる、モチベーション低下

✅ Good
「80%の品質で承認」
「軽微な改善は次のPRで」
「完璧より進捗」
```

### 2. スコープクリープ

```markdown
❌ Bad
「この機能を追加してください」
「ついでにこれも修正してください」
「せっかくだからリファクタリングも」
→ PRが肥大化、レビュー困難

✅ Good
「この改善は別PRで対応しましょう」
「Issue作成して次回対応でどうでしょう」
「今回はこのスコープに集中しましょう」
```

### 3. コンテキスト不足

```markdown
❌ Bad
「ここ間違ってます」
「これダメです」
→ 何がどう問題か不明

✅ Good
「XXXの場合、YYYになる可能性があります」
「仕様書ではZZZとなっていますが、実装がそれと異なっているようです」
→ 理由と根拠を明確に
```

### 4. 遅いレビュー

```markdown
❌ Bad
PR作成から3日後にレビュー
→ 作成者が別の作業に移っている
→ コンテキストスイッチのコスト

✅ Good
当日中、遅くとも翌日までに初回レビュー
→ 作成者の記憶が新しい
→ 迅速なフィードバックループ
```

---

## レビューチェックリスト

### レビュワー用クイックチェックリスト

```markdown
## 開始前
- [ ] PR説明を読んだ
- [ ] CIが通っている
- [ ] 変更規模を確認した

## 機能性
- [ ] 要件を満たしている
- [ ] エッジケースを考慮
- [ ] エラーハンドリングが適切

## 設計
- [ ] アーキテクチャに従っている
- [ ] SOLID原則に従っている
- [ ] 適切な抽象化

## 可読性
- [ ] 命名が明確
- [ ] コメントが適切
- [ ] 複雑度が適切

## テスト
- [ ] テストがある
- [ ] カバレッジが維持されている
- [ ] テストケースが適切

## セキュリティ
- [ ] 入力検証がある
- [ ] 認証・認可が適切
- [ ] 機密情報の扱いが適切

## パフォーマンス
- [ ] N+1問題がない
- [ ] 適切なアルゴリズム
- [ ] メモリリークがない

## ドキュメント
- [ ] コードコメントがある
- [ ] APIドキュメント更新済み
- [ ] README更新済み（必要に応じて）

## 完了前
- [ ] すべてのコメントを投稿した
- [ ] 承認/変更要求/コメントを選択した
- [ ] 理由を明確に記載した
```

---

## まとめ

効果的なレビューは、品質とスピードのバランスです。

### 重要ポイント

1. **構造的なアプローチ**
2. **時間管理の徹底**
3. **建設的なフィードバック**
4. **適切な承認基準**
5. **迅速なレビュー**

### 次のステップ

- [フィードバック対応](10-feedback-response.md)
- [コミュニケーション](11-communication.md)
- [自動化](12-automation.md)
