# コミュニケーションガイド

## 概要

コードレビューにおける効果的なコミュニケーション方法を解説します。建設的なフィードバック、適切なトーン、チーム文化の構築について学びます。

## 目次

1. [建設的なフィードバック](#建設的なフィードバック)
2. [コメントテンプレート](#コメントテンプレート)
3. [トーンとマナー](#トーンとマナー)
4. [文化構築](#文化構築)
5. [リモートでのレビュー](#リモートでのレビュー)

---

## 建設的なフィードバック

### フィードバックの原則

#### 1. 具体的に指摘する

```markdown
## ❌ Bad: 曖昧
「このコードは良くない」
「もっと綺麗に書いて」
「パフォーマンスが心配」

## ✅ Good: 具体的
「この部分は O(n²) の計算量になっています。
1000件以上のデータでは遅くなる可能性があります。」

「変数名 `tmp` では意図が不明確です。
`filteredUsers` のような具体的な名前はいかがでしょうか？」

「この関数は現在80行ありますが、以下のように
3つの関数に分割すると理解しやすくなります...」
```

#### 2. 理由を説明する

```markdown
## ❌ Bad: 指示のみ
「ここは async/await を使ってください」

## ✅ Good: 理由付き
「ここは async/await を使うことをお勧めします。

**理由:**
1. エラーハンドリングが try/catch で統一できる
2. ネストが浅くなり可読性が向上する
3. デバッグが容易になる

**例:**
\`\`\`typescript
// Before
fetchUser()
  .then(user => fetchPosts(user.id))
  .then(posts => console.log(posts))
  .catch(err => console.error(err));

// After
try {
  const user = await fetchUser();
  const posts = await fetchPosts(user.id);
  console.log(posts);
} catch (err) {
  console.error(err);
}
\`\`\`
」
```

#### 3. 解決策を提案する

```markdown
## ❌ Bad: 問題の指摘のみ
「この実装だとメモリリークします」

## ✅ Good: 解決策も提示
「この実装だとメモリリークの可能性があります。

**問題:**
ViewControllerとDataLoaderの間に循環参照が
発生しています。

**解決策:**
weak self を使用してください:

\`\`\`swift
dataLoader?.onComplete = { [weak self] in
    self?.updateUI()
}
\`\`\`

**参考:**
https://docs.swift.org/swift-book/LanguageGuide/AutomaticReferenceCounting.html
」
```

#### 4. 優先度を明示する

```markdown
## ✅ 優先度ラベルを使用

### [必須] セキュリティ問題
「このコードはSQLインジェクションの脆弱性があります。
必ず修正してください。」

### [推奨] リファクタリング
「この関数は分割することをお勧めしますが、
急ぎであれば次のPRでも構いません。」

### [質問] 仕様確認
「この動作は意図的でしょうか？
仕様書では別の動作になっているようです。」

### [提案] 代替案
「こういう実装方法もあります。
参考までにお知らせします。」

### [学習] 情報共有
「最近のベストプラクティスでは、
こういうパターンが推奨されています。」

### [賞賛] 良い点
「このエラーハンドリング、とても丁寧ですね！」
```

---

## コメントテンプレート

### バグ指摘

```markdown
[必須] バグ: [簡潔な説明]

**問題:**
[バグの詳細説明]

**再現手順:**
1. XXXする
2. YYYする
3. ZZZが発生

**期待される動作:**
[本来あるべき動作]

**修正案:**
\`\`\`[言語]
[修正後のコード]
\`\`\`

**テストケース:**
このケースもテストに追加してください:
- [テストケース1]
- [テストケース2]
```

### セキュリティ問題

```markdown
[必須] セキュリティ: [脆弱性の種類]

**脆弱性:**
[詳細な説明]

**攻撃シナリオ:**
攻撃者が以下のように悪用できます:
[具体的な攻撃例]

**影響範囲:**
- [影響1]
- [影響2]

**修正方法:**
\`\`\`[言語]
[セキュアなコード例]
\`\`\`

**参考:**
- OWASP: [URL]
- CWE-XXX: [URL]
```

### パフォーマンス問題

```markdown
[必須/推奨] パフォーマンス: [問題の要約]

**現状:**
- 計算量: O(n²)
- 想定データ量: 10,000件
- 予想実行時間: ~10秒

**問題:**
[具体的な問題点]

**改善案:**
\`\`\`[言語]
[最適化されたコード]
\`\`\`

**改善後:**
- 計算量: O(n)
- 予想実行時間: ~100ms

**ベンチマーク:**
[ベンチマーク結果があれば]
```

### リファクタリング提案

```markdown
[推奨] リファクタリング: [提案内容]

**現状の課題:**
- [課題1]
- [課題2]

**提案:**
[具体的な提案内容]

**メリット:**
- ✅ [メリット1]
- ✅ [メリット2]
- ✅ [メリット3]

**実装例:**
\`\`\`[言語]
[リファクタリング後のコード]
\`\`\`

**注意:**
急ぎであれば次のPRでも構いません。
Issue として記録しておくことをお勧めします。
```

### 設計の問題

```markdown
[必須/推奨] 設計: [問題点]

**現状:**
[現在の設計]

**懸念点:**
1. [懸念1]
2. [懸念2]

**代替案:**
[より良い設計の提案]

**比較:**
| 観点 | 現在の設計 | 提案する設計 |
|------|-----------|-------------|
| 拡張性 | △ | ◎ |
| テスト容易性 | △ | ◎ |
| 複雑度 | シンプル | やや複雑 |

**判断:**
[どちらを選ぶべきかの推奨]

**議論:**
チームで議論したい内容があれば、
ミーティングを設定しましょうか？
```

### 賞賛

```markdown
[賞賛] [良い点の要約]

[具体的にどこが良いか]

この実装から学びました:
- [学んだこと1]
- [学んだこと2]

他の場所でも使えそうですね！ 👍
```

---

## トーンとマナー

### 適切なトーン

```markdown
## ポジティブなトーン

### ❌ Bad: 否定的
「これは間違っています」
「こんな書き方はダメです」
「なんでこうしたんですか？」

### ✅ Good: 前向き
「こうするとより良くなります」
「この方法も検討してみてください」
「この意図について教えていただけますか？」

## 協力的なトーン

### ❌ Bad: 命令的
「XXXに変更してください」
「YYYを削除してください」

### ✅ Good: 提案的
「XXXに変更してはいかがでしょうか？」
「YYYは削除しても良いかもしれません」
「一緒に考えてみましょう」
```

### 質問の仕方

```markdown
## ❌ Bad: 批判的な質問
「なんでこんなコード書いたの？」
「これで動くと思ったの？」
「前も言ったけど、理解してる？」

## ✅ Good: 好奇心を持った質問
「この実装を選択した背景を教えていただけますか？」
「この動作は意図的でしょうか？」
「以前議論した内容との関係について確認させてください」

## ✅ より良い質問
「この部分について確認させてください。
XXXの場合、YYYになると思うのですが、
ZZZを考慮されていますか？」
```

### 文化的配慮

```markdown
## 多様なバックグラウンド

### ✅ 明確で簡潔な英語
「Please consider using a Set instead of an Array
for better performance.」

### ❌ スラングや慣用句
「This code is a piece of cake to break.」
→ 非ネイティブには理解困難

### ✅ 文化的に中立
「This implementation has some issues.」

### ❌ 文化的に偏った表現
「This is common sense.」
→ 文化によって「常識」は異なる

## タイムゾーンへの配慮

### ✅ 時間の明示
「明日13:00 JST にミーティングしましょう」
「2024-03-15 までに対応をお願いします」

### ❌ 曖昧な時間表現
「明日の午後」
「今週中」
```

---

## 文化構築

### レビュー文化の原則

```markdown
## 基本原則

### 1. コードを批判し、人を批判しない
❌ 「あなたのコードは読みにくい」
✅ 「このコードは読みにくくなっています」

### 2. 学習の機会と捉える
レビュワー → 作成者: 新しい知識を教える
作成者 → レビュワー: 新しい視点を学ぶ

### 3. 完璧を求めない
80% の品質で OK
100% を目指すと時間がかかりすぎる

### 4. 早く、頻繁にレビュー
大きなPRより小さなPR
週1回より毎日

### 5. 自動化できるものは自動化
スタイル → Linter
型チェック → TypeScript
テスト → CI

### 6. ポジティブなフィードバックも
良い点も積極的にコメント
改善点だけではモチベーション低下
```

### チームでの取り組み

```markdown
## レビューガイドライン作成

### 含めるべき内容
1. レビューの目的
2. レビュー観点
3. コメントの書き方
4. 承認基準
5. タイムライン

### 例: チームガイドライン
\`\`\`markdown
# コードレビューガイドライン

## 目的
- 品質向上
- 知識共有
- チーム学習

## レビュー観点
1. 機能性: 要件を満たすか
2. テスト: 十分なテストがあるか
3. セキュリティ: 脆弱性はないか
... (7つの観点)

## タイムライン
- 初回レビュー: 24時間以内
- Re-review: 4時間以内
- ブロッカー: 1時間以内

## コメント形式
[必須], [推奨], [質問], [賞賛] を使用
\`\`\`

## 定期的なふりかえり

### 月次レビューミーティング
- 良かったレビュー事例の共有
- 改善点の議論
- ガイドラインの更新

### メトリクス確認
- 平均レビュー時間
- PRサイズの推移
- コメント数の推移
- 承認までの時間
```

### 新メンバーへの教育

```markdown
## オンボーディングプログラム

### Week 1: 観察
- 他の人のレビューを見る
- コメントのパターンを学ぶ
- 質問を推奨

### Week 2: ペアレビュー
- 経験者と一緒にレビュー
- フィードバックをもらう
- 少しずつコメント

### Week 3: 独立レビュー
- 小さなPRから開始
- メンターがダブルチェック
- フィードバック

### Week 4+: 通常レビュー
- 独立してレビュー
- 困ったら相談OK
```

---

## リモートでのレビュー

### 非同期コミュニケーション

```markdown
## ✅ 非同期レビューのベストプラクティス

### 1. コンテキストを十分に提供
PR説明に以下を含める:
- 背景と目的
- 変更内容
- テスト方法
- スクリーンショット（UI変更時）

### 2. 詳細なコメント
対面で補足説明できないため:
- 理由を明確に
- コード例を含める
- 参考リンクを追加

### 3. 絵文字を活用
トーンを伝えるために:
- ✅ 承認
- 🤔 疑問
- 💡 提案
- 👍 賞賛
- ⚠️ 警告
- 🐛 バグ

### 4. タイムゾーンを考慮
- レビュー可能時間を共有
- 緊急度を明示
- 必要に応じて同期ミーティング
```

### 同期コミュニケーション

```markdown
## いつ同期ミーティングするか

### ✅ ミーティングすべき状況
1. アーキテクチャの大きな決定
2. 意見が平行線
3. 複雑な技術的議論
4. 大規模リファクタリング

### ミーティング形式

#### ペアレビュー（15-30分）
- 画面共有でコードを見ながら
- リアルタイムで質問・回答
- その場で修正

#### 設計ディスカッション（30-60分）
- ホワイトボードで図を描く
- 代替案を議論
- 合意形成

#### レトロスペクティブ（60分）
- レビューp ロセスの改善
- 学びの共有
- ガイドライン更新
```

---

## ケーススタディ

### 良いレビューの例

```markdown
## PR: ユーザー認証機能追加

### レビュワーのコメント

[賞賛] JWT実装が素晴らしい！

トークン生成のロジックがとても綺麗ですね。
エラーハンドリングも丁寧で、セキュリティも考慮されています。

特に以下の点が良いです:
- リフレッシュトークンの実装
- トークンの有効期限管理
- 適切な例外処理

---

[質問] トークン保存先について

現在、トークンをLocalStorageに保存していますが、
セキュリティの観点から HttpOnly Cookie の使用も
検討されましたか？

XSS攻撃のリスクを考えると、Cookie の方が安全です。

参考: https://owasp.org/...

---

[推奨] テストケースの追加

以下のエッジケースもテストに追加することを
お勧めします:

1. トークン有効期限切れ
2. 不正なトークン
3. トークンなし

\`\`\`typescript
test('throws error when token is expired', async () => {
  const expiredToken = generateExpiredToken();
  await expect(verifyToken(expiredToken))
    .rejects
    .toThrow(TokenExpiredError);
});
\`\`\`

急ぎであれば次のPRでも構いません。

---

[必須] パスワード保存について

現在、パスワードが平文で保存されています。
これはセキュリティ上の重大な問題です。

bcrypt でハッシュ化してください:

\`\`\`typescript
import bcrypt from 'bcrypt';

const hashedPassword = await bcrypt.hash(password, 10);
user.password = hashedPassword;
\`\`\`

---

### 総評

全体的に素晴らしい実装です！ 🎉

パスワードハッシュ化を修正したらApprove します。
他の点は次のPRで改善しましょう。

お疲れ様でした！
```

---

## まとめ

良いコミュニケーションは、効果的なコードレビューの基盤です。

### 重要ポイント

1. **具体的で建設的なフィードバック**
2. **ポジティブなトーン**
3. **文化的配慮**
4. **チーム文化の構築**
5. **非同期コミュニケーションの最適化**

### 次のステップ

- [自動化ガイド](12-automation.md)
- [ベストプラクティス完全ガイド](best-practices-complete.md)
