---
name: documentation
description: |
  技術ドキュメント、README、API仕様書、技術書、記事の作成ガイド。

  【最重要】実測データ・検証結果の扱い方
  ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
  絶対に守るべきルール:
  1. 実際に測定・検証していないデータを「実測」として書かない
  2. 架空の数値を具体的に記載しない
  3. 「〜したところ」「検証した結果」など、実証したかのような表現を使わない
  4. Before/Afterに具体的な数値（メモリ使用量、処理時間等）を書かない
  5. 公式ドキュメント・論文・信頼できる情報源のみを引用

  読者は「著者が実際に検証した」と受け取ります。
  検証していないのに検証したように見える表現は、読者の信頼を裏切る行為です。
skill_type: project
visibility: gitignored
---

# 技術ドキュメント作成ガイド - 誠実性第一

## 【絶対厳守】実測データ・検証結果の扱い

### なぜ厳守すべきか

読者は以下のように受け取ります：

```markdown
❌ 「メモリ使用量: 1,240MB → 85MB（93%削減）」
→ 読者の認識: 「著者が実際に測定した結果だ」

❌ 「実測したところ、78%高速化しました」
→ 読者の認識: 「著者が検証して確認した事実だ」

❌ 「このプロジェクトで効果を実証しました」
→ 読者の認識: 「実際のプロジェクトで使って成功した」
```

**問題点:**
- 実際には検証していないのに、検証したと受け取られる
- 読者が「著者は嘘をついている」と感じる
- 信頼性が完全に失われる
- 技術者としての評判を損なう

## 最重要原則: 誠実性と正確性

### ❌ 絶対にやってはいけないこと

#### パターン1: 架空の具体的数値

```markdown
❌ 悪い例:
## パフォーマンス改善の実測結果

実測データ:
- Before: 処理時間 5.2秒、メモリ使用量 1,240MB
- After: 処理時間 1.1秒、メモリ使用量 85MB
- 改善率: 79%高速化、93%メモリ削減
```

**なぜダメか:**
- 具体的な数値（5.2秒、1,240MB等）を書くと「実測した」と受け取られる
- 実際には測定していないのに、測定したように見える
- これは**虚偽の情報**

#### パターン2: 「〜したところ」「検証した結果」

```markdown
❌ 悪い例:
実際のプロジェクトで検証したところ、パフォーマンスが大幅に向上しました。
測定した結果、メモリ使用量が90%削減されました。
本番環境で試したところ、レスポンスタイムが半分になりました。
```

**なぜダメか:**
- 「検証したところ」「測定した結果」は「著者が実際に行った」と読める
- 実際にはやっていないことを、やったように書いている
- これは**読者への欺瞞**

#### パターン3: 曖昧だが誤解を招く表現

```markdown
❌ 悪い例:
このテクニックを使うと、メモリ使用量が大幅に減ります。
パフォーマンスが劇的に向上します。
処理速度が数倍になります。
```

**なぜダメか:**
- 「大幅に」「劇的に」「数倍」は具体性がないが、効果を保証しているように見える
- 実際の効果は環境・データ・実装により大きく異なる
- 読者が期待した効果が出ないと「嘘だった」と感じる

#### パターン4: Before/Afterに具体的数値

```markdown
❌ 悪い例:

### Before
```typescript
const users = await fetchAll(); // 100,000件
// 処理時間: 8.3秒
// メモリ使用量: 1,240MB
```

### After
```typescript
const users = await fetchStream(); // 100,000件
// 処理時間: 2.1秒（74%改善）
// メモリ使用量: 85MB（93%削減）
```
```

**なぜダメか:**
- コメントで具体的な数値を書くと「著者が測定した」と受け取られる
- Before/Afterの比較に具体的数値があると、検証済みに見える
- これは**検証していない情報の提示**

### ✅ 正しい書き方

#### 1. 理論・原理を説明

```markdown
## パフォーマンス改善の原理

Stream処理は、データ全体をメモリにロードせず、
チャンク単位で処理するため、メモリ使用量を大幅に削減できます。

### なぜ効率的か

- データ全体をメモリに保持しない
- チャンク単位（通常64KB）で処理
- GC（ガベージコレクション）の負荷が減る
```

#### 2. 公式ドキュメント・ベンチマークを引用

```markdown
## Fastifyのパフォーマンス

公式ベンチマーク（https://fastify.dev/benchmarks/）によると:
- Fastify: 78,513 req/sec
- Express: 14,200 req/sec

※ 環境や条件により結果は異なります
```

#### 3. 一般的な傾向として記載

```markdown
## メモリ最適化の効果

大容量データ（数GB）をStream処理に変更すると:
- メモリ使用量が大幅に削減される傾向
- ピークメモリが数十MB〜数百MBに抑えられることが多い
- 処理速度は若干向上またはほぼ同等

※ 実際の効果は、データ構造やアプリケーションによって異なります
```

#### 4. 架空の例として明記

```markdown
## 最適化の例（架空のシナリオ）

以下は、10万件のユーザーデータを処理する**仮想的な例**です:

```typescript
// 全てメモリにロード（仮想例）
// → メモリ使用量が増加する可能性
const users = await prisma.user.findMany(); // 全件取得
```

**注意:** これは説明のための架空の例であり、実測値ではありません。
実際のパフォーマンスは環境・データ量・実装により大きく異なります。
```

## ドキュメント作成の原則

### 1. 事実と推測を明確に区別

```markdown
✅ 良い例:
「Node.jsはシングルスレッドで動作します（公式ドキュメント）」
「一般的に、非同期I/Oは高い同時実行性を実現できます」

❌ 悪い例:
「Node.jsは必ずExpressより高速です」（状況による）
「この方法で必ず50%高速化します」（検証していない）
```

### 2. 情報源を明記

```markdown
✅ 良い例:
- Node.js公式ドキュメント（https://nodejs.org/docs/）
- MDN Web Docs
- 公式ベンチマーク
- 著名な技術ブログ（作者・日付明記）

❌ 悪い例:
「実測したところ」（実際には測定していない）
「プロジェクトで検証した」（検証していない）
```

### 3. 「一般的に」「多くの場合」を使う

```markdown
✅ 良い例:
「一般的に、LRUキャッシュはメモリ使用量を抑えられます」
「多くの場合、Stream処理は大容量データに適しています」
「通常、インデックスはクエリを高速化します」

❌ 悪い例:
「LRUキャッシュで必ずメモリが95%削減されます」
「Stream処理は常に高速です」
```

### 4. コード例は動作する実装

```typescript
// ✅ 良い例: 実際に動作するコード
import express from 'express';

const app = express();

app.get('/api/users', async (req, res) => {
  const users = await getUsers();
  res.json(users);
});

app.listen(3000);
```

```typescript
// ❌ 悪い例: 動作しない架空のコード
import magicFramework from 'magic'; // 存在しないライブラリ

magicFramework.autoOptimize(); // 架空のAPI
// これで自動的に10倍高速化！
```

## 技術書・記事の構成

### 基本構成

```markdown
# タイトル

## 概要
- 何を学ぶか
- なぜ重要か
- 前提知識

## 原理・理論
- なぜそうなるか
- どのような仕組みか
- 公式ドキュメントの引用

## 実装例
- 実際に動作するコード
- Before/Afterの比較
- ベストプラクティス

## よくある間違い
- アンチパターン
- なぜ避けるべきか

## まとめ
- 重要なポイント
- 次のステップ
```

### Before/Afterの正しい書き方

```markdown
## 最適化の例

### Before: 改善の余地がある実装

```typescript
// グローバルキャッシュが無限に増える
const cache = new Map();

app.get('/users/:id', async (req, res) => {
  if (!cache.has(req.params.id)) {
    const user = await fetchUser(req.params.id);
    cache.set(req.params.id, user);
  }
  res.json(cache.get(req.params.id));
});
```

**問題点:**
- エントリが無限に増加
- メモリリークの原因
- 古いデータが残り続ける

### After: LRUキャッシュで改善

```typescript
import LRU from 'lru-cache';

const cache = new LRU({
  max: 1000,
  ttl: 1000 * 60 * 10, // 10分
});

app.get('/users/:id', async (req, res) => {
  let user = cache.get(req.params.id);
  if (!user) {
    user = await fetchUser(req.params.id);
    cache.set(req.params.id, user);
  }
  res.json(user);
});
```

**改善点:**
- 最大エントリ数を制限
- 古いエントリは自動削除
- TTLでデータ鮮度を保証

**期待される効果:**
- メモリ使用量の安定化
- 長期間稼働しても安全
```

## データ・統計の扱い方

### ✅ 許容される表現

1. **公式データの引用**
```markdown
Node.js公式ベンチマーク（https://...）によると:
- V8エンジンのJITコンパイラにより、動的型付け言語としては高速
```

2. **一般的な傾向**
```markdown
一般的に、以下のような傾向があります:
- 非同期I/Oは高い同時実行性を実現
- ただし、CPU集約的な処理には不向き
```

3. **条件付きの記述**
```markdown
大容量データ（数GB以上）の場合:
- Stream処理が推奨される
- メモリ使用量を大幅に削減できる可能性
```

### ❌ 避けるべき表現

```markdown
❌ 「実測したところ、78%高速化しました」
   → 実際には測定していない

❌ 「メモリ使用量: 1,240MB → 85MB」
   → 具体的な数値は検証が必要

❌ 「このプロジェクトで実証済み」
   → 実際のプロジェクトではない
```

## チェックリスト

執筆前に必ず確認:

- [ ] 実測していないデータを「実測」として記載していないか
- [ ] 架空の数値を事実のように書いていないか
- [ ] 情報源（公式ドキュメント等）を明記しているか
- [ ] コード例は実際に動作するか
- [ ] 「一般的に」「多くの場合」など適切な表現を使っているか
- [ ] Before/Afterの比較は原理・理論に基づいているか
- [ ] 読者に誤解を与える表現がないか

## 参考リンク

- [Node.js公式ドキュメント](https://nodejs.org/docs/)
- [MDN Web Docs](https://developer.mozilla.org/)
- [TypeScript公式](https://www.typescriptlang.org/docs/)

## まとめ

技術ドキュメント作成の鉄則:
1. **誠実性**: 検証していないことを事実のように書かない
2. **正確性**: 公式ドキュメント・理論に基づく
3. **明確性**: 事実と推測を区別
4. **実用性**: 実際に動作するコード例

読者の信頼を得るためには、正確で誠実な情報提供が最も重要です。
