# Maintenance Scripts 完全ガイド

## 目次
1. [メンテナンススクリプトの基礎](#メンテナンススクリプトの基礎)
2. [クリーンアップスクリプト](#クリーンアップスクリプト)
3. [バックアップとリストア](#バックアップとリストア)
4. [ログ管理とローテーション](#ログ管理とローテーション)
5. [データベースメンテナンス](#データベースメンテナンス)
6. [モニタリングとアラート](#モニタリングとアラート)
7. [セキュリティスキャン](#セキュリティスキャン)
8. [パフォーマンス最適化](#パフォーマンス最適化)

---

## メンテナンススクリプトの基礎

### メンテナンスウィンドウの管理

```bash
#!/usr/bin/env bash
# maintenance-window.sh - メンテナンスウィンドウ管理

set -euo pipefail

readonly MAINTENANCE_FILE="/var/run/maintenance.lock"
readonly STATUS_PAGE_API="${STATUS_PAGE_API:-}"

# メンテナンスモードの開始
start_maintenance() {
    local reason="${1:-Scheduled maintenance}"
    local duration="${2:-1h}"

    echo "Starting maintenance mode"

    # メンテナンスフラグの作成
    cat > "${MAINTENANCE_FILE}" <<EOF
{
    "started": "$(date -u +%Y-%m-%dT%H:%M:%SZ)",
    "reason": "${reason}",
    "duration": "${duration}"
}
EOF

    # ステータスページの更新
    update_status_page "maintenance" "${reason}"

    # Nginxメンテナンスページの有効化
    if [[ -f /etc/nginx/sites-available/maintenance ]]; then
        ln -sf /etc/nginx/sites-available/maintenance /etc/nginx/sites-enabled/
        nginx -t && nginx -s reload
    fi

    # 通知
    notify_users "maintenance_start" "${reason}"
}

# メンテナンスモードの終了
end_maintenance() {
    echo "Ending maintenance mode"

    # メンテナンスフラグの削除
    rm -f "${MAINTENANCE_FILE}"

    # ステータスページの更新
    update_status_page "operational" "All systems operational"

    # Nginxメンテナンスページの無効化
    rm -f /etc/nginx/sites-enabled/maintenance
    nginx -t && nginx -s reload

    # 通知
    notify_users "maintenance_end" "Maintenance completed"
}

# ステータスページの更新
update_status_page() {
    local status="${1}"
    local message="${2}"

    if [[ -z "${STATUS_PAGE_API}" ]]; then
        return 0
    fi

    # Statuspage.io API の例
    curl -X PATCH "${STATUS_PAGE_API}/incidents" \
        -H "Authorization: OAuth ${STATUS_PAGE_TOKEN}" \
        -H "Content-Type: application/json" \
        -d "{
            \"incident\": {
                \"status\": \"${status}\",
                \"body\": \"${message}\"
            }
        }"
}

# ユーザーへの通知
notify_users() {
    local event="${1}"
    local message="${2}"

    # メール通知
    if [[ -f /etc/maintenance/subscribers.txt ]]; then
        while IFS= read -r email; do
            send_email "${email}" "Maintenance Notification" "${message}"
        done < /etc/maintenance/subscribers.txt
    fi

    # Slack通知
    send_slack_notification "${message}"
}

# メインテナンスの実行
run_maintenance() {
    local maintenance_tasks="${1:-}"

    start_maintenance "Scheduled system maintenance" "2h"

    # メンテナンスタスクの実行
    if [[ -n "${maintenance_tasks}" ]]; then
        eval "${maintenance_tasks}" || {
            echo "Maintenance tasks failed" >&2
            end_maintenance
            return 1
        }
    fi

    end_maintenance
}
```

### ヘルスチェックスクリプト

```bash
#!/usr/bin/env bash
# health-check.sh - システムヘルスチェック

set -euo pipefail

# サービスのヘルスチェック
check_service_health() {
    local service_name="${1}"

    if systemctl is-active --quiet "${service_name}"; then
        echo "✓ ${service_name} is running"
        return 0
    else
        echo "✗ ${service_name} is not running" >&2
        return 1
    fi
}

# ポートのヘルスチェック
check_port() {
    local host="${1}"
    local port="${2}"

    if timeout 5 bash -c "cat < /dev/null > /dev/tcp/${host}/${port}" 2>/dev/null; then
        echo "✓ Port ${port} on ${host} is reachable"
        return 0
    else
        echo "✗ Port ${port} on ${host} is not reachable" >&2
        return 1
    fi
}

# HTTPエンドポイントのヘルスチェック
check_http_endpoint() {
    local url="${1}"
    local expected_status="${2:-200}"

    local status_code
    status_code="$(curl -s -o /dev/null -w '%{http_code}' "${url}")"

    if [[ "${status_code}" == "${expected_status}" ]]; then
        echo "✓ ${url} returned ${status_code}"
        return 0
    else
        echo "✗ ${url} returned ${status_code} (expected ${expected_status})" >&2
        return 1
    fi
}

# データベース接続チェック
check_database() {
    local db_type="${1}"
    local connection_string="${2}"

    case "${db_type}" in
        postgres)
            if psql "${connection_string}" -c "SELECT 1" &>/dev/null; then
                echo "✓ PostgreSQL connection successful"
                return 0
            else
                echo "✗ PostgreSQL connection failed" >&2
                return 1
            fi
            ;;
        mysql)
            if mysql "${connection_string}" -e "SELECT 1" &>/dev/null; then
                echo "✓ MySQL connection successful"
                return 0
            else
                echo "✗ MySQL connection failed" >&2
                return 1
            fi
            ;;
        redis)
            if redis-cli -u "${connection_string}" PING | grep -q PONG; then
                echo "✓ Redis connection successful"
                return 0
            else
                echo "✗ Redis connection failed" >&2
                return 1
            fi
            ;;
    esac
}

# ディスク容量チェック
check_disk_space() {
    local threshold="${1:-90}"
    local mount_point="${2:-/}"

    local usage
    usage="$(df -h "${mount_point}" | tail -1 | awk '{print int($5)}')"

    if ((usage < threshold)); then
        echo "✓ Disk usage is ${usage}% (threshold: ${threshold}%)"
        return 0
    else
        echo "✗ Disk usage is ${usage}% (threshold: ${threshold}%)" >&2
        return 1
    fi
}

# 総合ヘルスチェック
comprehensive_health_check() {
    local failed_checks=0

    echo "=== System Health Check ==="
    echo ""

    # サービスチェック
    echo "Services:"
    check_service_health "nginx" || ((failed_checks++))
    check_service_health "postgresql" || ((failed_checks++))
    check_service_health "redis" || ((failed_checks++))
    echo ""

    # ポートチェック
    echo "Ports:"
    check_port "localhost" "80" || ((failed_checks++))
    check_port "localhost" "443" || ((failed_checks++))
    check_port "localhost" "5432" || ((failed_checks++))
    echo ""

    # HTTPチェック
    echo "HTTP Endpoints:"
    check_http_endpoint "http://localhost/health" || ((failed_checks++))
    check_http_endpoint "http://localhost/api/status" || ((failed_checks++))
    echo ""

    # データベースチェック
    echo "Database Connections:"
    check_database "postgres" "postgresql://localhost/mydb" || ((failed_checks++))
    check_database "redis" "redis://localhost:6379" || ((failed_checks++))
    echo ""

    # リソースチェック
    echo "Resources:"
    check_disk_space 90 "/" || ((failed_checks++))
    echo ""

    echo "==========================="
    if ((failed_checks > 0)); then
        echo "Health check failed: ${failed_checks} issue(s) found"
        return 1
    else
        echo "All health checks passed"
        return 0
    fi
}

comprehensive_health_check
```

---

## クリーンアップスクリプト

### ディスククリーンアップ

```bash
#!/usr/bin/env bash
# disk-cleanup.sh - ディスククリーンアップ

set -euo pipefail

readonly DRY_RUN="${DRY_RUN:-false}"

# 古いログファイルの削除
cleanup_old_logs() {
    local log_dir="${1:-/var/log}"
    local days_old="${2:-30}"

    echo "Cleaning up logs older than ${days_old} days in ${log_dir}"

    local find_cmd=(
        find "${log_dir}"
        -type f
        \( -name "*.log" -o -name "*.log.*" \)
        -mtime "+${days_old}"
    )

    if [[ "${DRY_RUN}" == "true" ]]; then
        "${find_cmd[@]}" -print
    else
        "${find_cmd[@]}" -delete -print
    fi
}

# 一時ファイルのクリーンアップ
cleanup_temp_files() {
    local temp_dirs=("/tmp" "/var/tmp")
    local days_old="${1:-7}"

    for temp_dir in "${temp_dirs[@]}"; do
        echo "Cleaning up ${temp_dir}"

        if [[ "${DRY_RUN}" == "true" ]]; then
            find "${temp_dir}" -type f -atime "+${days_old}" -print
        else
            find "${temp_dir}" -type f -atime "+${days_old}" -delete
        fi
    done
}

# Dockerイメージのクリーンアップ
cleanup_docker() {
    if ! command -v docker &>/dev/null; then
        return 0
    fi

    echo "Cleaning up Docker resources"

    if [[ "${DRY_RUN}" == "true" ]]; then
        docker system df
        return 0
    fi

    # 使用されていないイメージの削除
    docker image prune -a --filter "until=168h" -f

    # 停止中のコンテナの削除
    docker container prune -f

    # 使用されていないボリュームの削除
    docker volume prune -f

    # 使用されていないネットワークの削除
    docker network prune -f

    # ビルドキャッシュのクリーンアップ
    docker builder prune -a -f
}

# npm/yarnキャッシュのクリーンアップ
cleanup_package_managers() {
    echo "Cleaning up package manager caches"

    if command -v npm &>/dev/null; then
        echo "Cleaning npm cache"
        npm cache clean --force
    fi

    if command -v yarn &>/dev/null; then
        echo "Cleaning yarn cache"
        yarn cache clean
    fi

    if command -v pip &>/dev/null; then
        echo "Cleaning pip cache"
        pip cache purge
    fi
}

# システムパッケージのクリーンアップ
cleanup_system_packages() {
    echo "Cleaning up system packages"

    if command -v apt-get &>/dev/null; then
        apt-get autoremove -y
        apt-get autoclean -y
        apt-get clean -y
    elif command -v yum &>/dev/null; then
        yum autoremove -y
        yum clean all
    fi
}

# ジャーナルログのクリーンアップ
cleanup_journal() {
    local max_size="${1:-100M}"

    if command -v journalctl &>/dev/null; then
        echo "Cleaning up journal logs (keeping ${max_size})"
        journalctl --vacuum-size="${max_size}"
    fi
}

# メインクリーンアップ
main_cleanup() {
    echo "=== Disk Cleanup Started ==="
    echo "Dry run: ${DRY_RUN}"
    echo ""

    # ディスク使用量（クリーンアップ前）
    echo "Disk usage before cleanup:"
    df -h /
    echo ""

    cleanup_old_logs "/var/log" 30
    cleanup_temp_files 7
    cleanup_docker
    cleanup_package_managers
    cleanup_system_packages
    cleanup_journal "100M"

    # ディスク使用量（クリーンアップ後）
    echo ""
    echo "Disk usage after cleanup:"
    df -h /
    echo ""

    echo "=== Disk Cleanup Completed ==="
}

main_cleanup
```

### アプリケーションデータのクリーンアップ

```bash
#!/usr/bin/env bash
# app-data-cleanup.sh - アプリケーションデータクリーンアップ

set -euo pipefail

# 古いセッションの削除
cleanup_sessions() {
    local session_dir="${1:-/var/lib/sessions}"
    local hours_old="${2:-24}"

    echo "Cleaning up sessions older than ${hours_old} hours"

    find "${session_dir}" \
        -type f \
        -name "sess_*" \
        -mmin "+$((hours_old * 60))" \
        -delete \
        -print
}

# キャッシュのクリーンアップ
cleanup_cache() {
    local cache_dir="${1:-/var/cache/app}"

    echo "Cleaning up application cache"

    # Redisキャッシュのクリア
    if command -v redis-cli &>/dev/null; then
        redis-cli --scan --pattern "cache:*" | while read -r key; do
            local ttl
            ttl="$(redis-cli TTL "${key}")"

            # TTLが-1（期限なし）のキーを削除
            if [[ "${ttl}" == "-1" ]]; then
                redis-cli DEL "${key}"
                echo "Deleted cache key: ${key}"
            fi
        done
    fi

    # ファイルシステムキャッシュのクリア
    if [[ -d "${cache_dir}" ]]; then
        find "${cache_dir}" -type f -mtime +7 -delete
    fi
}

# データベースの古いレコード削除
cleanup_database_records() {
    local db_name="${1}"
    local retention_days="${2:-90}"

    echo "Cleaning up database records older than ${retention_days} days"

    # PostgreSQLの例
    psql "${db_name}" <<EOF
-- 古い監査ログの削除
DELETE FROM audit_logs
WHERE created_at < NOW() - INTERVAL '${retention_days} days';

-- 古い通知の削除
DELETE FROM notifications
WHERE created_at < NOW() - INTERVAL '${retention_days} days'
  AND read = TRUE;

-- 古いジョブ履歴の削除
DELETE FROM job_history
WHERE completed_at < NOW() - INTERVAL '${retention_days} days'
  AND status IN ('completed', 'failed');

-- VACUUMの実行
VACUUM ANALYZE;
EOF
}

# アップロードファイルのクリーンアップ
cleanup_uploads() {
    local upload_dir="${1:-/var/www/uploads}"
    local days_old="${2:-180}"

    echo "Cleaning up uploads older than ${days_old} days"

    # データベースに記録されていないファイルを検索
    find "${upload_dir}" -type f -mtime "+${days_old}" -print | while read -r file; do
        local filename
        filename="$(basename "${file}")"

        # データベースで参照されているかチェック
        local count
        count="$(psql mydb -t -c "
            SELECT COUNT(*)
            FROM files
            WHERE filename = '${filename}'
        " | tr -d ' ')"

        if [[ "${count}" == "0" ]]; then
            echo "Deleting unreferenced file: ${file}"
            rm "${file}"
        fi
    done
}
```

---

## バックアップとリストア

### 包括的なバックアップスクリプト

```bash
#!/usr/bin/env bash
# comprehensive-backup.sh - 包括的バックアップ

set -euo pipefail

readonly BACKUP_ROOT="${BACKUP_ROOT:-/var/backups}"
readonly TIMESTAMP="$(date +%Y%m%d_%H%M%S)"
readonly BACKUP_DIR="${BACKUP_ROOT}/${TIMESTAMP}"
readonly RETENTION_DAYS="${RETENTION_DAYS:-30}"

# バックアップディレクトリの作成
setup_backup_dir() {
    mkdir -p "${BACKUP_DIR}"/{database,files,config}
}

# データベースバックアップ
backup_databases() {
    echo "Backing up databases"

    # PostgreSQL
    if command -v pg_dumpall &>/dev/null; then
        pg_dumpall -U postgres | gzip > "${BACKUP_DIR}/database/postgres.sql.gz"
    fi

    # MySQL
    if command -v mysqldump &>/dev/null; then
        mysqldump --all-databases | gzip > "${BACKUP_DIR}/database/mysql.sql.gz"
    fi

    # MongoDB
    if command -v mongodump &>/dev/null; then
        mongodump --out "${BACKUP_DIR}/database/mongodb"
        tar -czf "${BACKUP_DIR}/database/mongodb.tar.gz" -C "${BACKUP_DIR}/database" mongodb
        rm -rf "${BACKUP_DIR}/database/mongodb"
    fi
}

# ファイルバックアップ
backup_files() {
    echo "Backing up files"

    local directories=(
        "/var/www"
        "/home"
        "/etc"
    )

    for dir in "${directories[@]}"; do
        if [[ -d "${dir}" ]]; then
            local backup_name
            backup_name="$(echo "${dir}" | tr '/' '_' | sed 's/^_//')"

            tar -czf "${BACKUP_DIR}/files/${backup_name}.tar.gz" \
                --exclude='node_modules' \
                --exclude='.git' \
                --exclude='*.log' \
                "${dir}"
        fi
    done
}

# 設定ファイルバックアップ
backup_configs() {
    echo "Backing up configurations"

    local configs=(
        "/etc/nginx"
        "/etc/systemd"
        "/etc/cron.d"
    )

    for config_dir in "${configs[@]}"; do
        if [[ -d "${config_dir}" ]]; then
            local backup_name
            backup_name="$(basename "${config_dir}")"

            tar -czf "${BACKUP_DIR}/config/${backup_name}.tar.gz" "${config_dir}"
        fi
    done
}

# バックアップの暗号化
encrypt_backup() {
    local backup_archive="${BACKUP_ROOT}/backup_${TIMESTAMP}.tar.gz"
    local encrypted_file="${backup_archive}.gpg"

    echo "Creating encrypted backup archive"

    # バックアップディレクトリのアーカイブ
    tar -czf "${backup_archive}" -C "${BACKUP_ROOT}" "${TIMESTAMP}"

    # GPGで暗号化
    if [[ -n "${GPG_RECIPIENT:-}" ]]; then
        gpg --encrypt --recipient "${GPG_RECIPIENT}" \
            --output "${encrypted_file}" \
            "${backup_archive}"

        # 元のアーカイブを削除
        rm "${backup_archive}"

        echo "Encrypted backup: ${encrypted_file}"
    fi
}

# リモートストレージへのアップロード
upload_to_remote() {
    local backup_file="${1}"

    echo "Uploading backup to remote storage"

    # S3へのアップロード
    if command -v aws &>/dev/null && [[ -n "${S3_BUCKET:-}" ]]; then
        aws s3 cp "${backup_file}" "s3://${S3_BUCKET}/backups/$(basename "${backup_file}")"
    fi

    # Google Cloud Storageへのアップロード
    if command -v gsutil &>/dev/null && [[ -n "${GCS_BUCKET:-}" ]]; then
        gsutil cp "${backup_file}" "gs://${GCS_BUCKET}/backups/$(basename "${backup_file}")"
    fi

    # rsyncでのバックアップ
    if [[ -n "${REMOTE_HOST:-}" ]]; then
        rsync -avz "${backup_file}" "${REMOTE_HOST}:/backups/"
    fi
}

# 古いバックアップの削除
cleanup_old_backups() {
    echo "Cleaning up backups older than ${RETENTION_DAYS} days"

    find "${BACKUP_ROOT}" \
        -maxdepth 1 \
        -type d \
        -name "[0-9]*" \
        -mtime "+${RETENTION_DAYS}" \
        -exec rm -rf {} \;

    find "${BACKUP_ROOT}" \
        -maxdepth 1 \
        -type f \
        -name "backup_*.tar.gz*" \
        -mtime "+${RETENTION_DAYS}" \
        -delete
}

# バックアップの検証
verify_backup() {
    local backup_file="${1}"

    echo "Verifying backup integrity"

    # アーカイブの整合性チェック
    if [[ "${backup_file}" == *.tar.gz ]]; then
        tar -tzf "${backup_file}" > /dev/null || {
            echo "Backup verification failed: ${backup_file}" >&2
            return 1
        }
    elif [[ "${backup_file}" == *.gpg ]]; then
        gpg --list-packets "${backup_file}" > /dev/null || {
            echo "Encrypted backup verification failed: ${backup_file}" >&2
            return 1
        }
    fi

    echo "Backup verified successfully"
}

# メインバックアッププロセス
main_backup() {
    echo "=== Backup Started ==="

    setup_backup_dir
    backup_databases
    backup_files
    backup_configs

    encrypt_backup

    local backup_file="${BACKUP_ROOT}/backup_${TIMESTAMP}.tar.gz.gpg"

    if verify_backup "${backup_file}"; then
        upload_to_remote "${backup_file}"
        cleanup_old_backups
        echo "=== Backup Completed Successfully ==="
    else
        echo "=== Backup Failed ===" >&2
        return 1
    fi
}

main_backup
```

### リストアスクリプト

```bash
#!/usr/bin/env bash
# restore.sh - バックアップからのリストア

set -euo pipefail

# バックアップのリスト表示
list_backups() {
    local backup_source="${1:-local}"

    case "${backup_source}" in
        local)
            find "${BACKUP_ROOT}" -name "backup_*.tar.gz*" -type f | sort -r
            ;;
        s3)
            aws s3 ls "s3://${S3_BUCKET}/backups/" | awk '{print $4}' | sort -r
            ;;
    esac
}

# バックアップのダウンロード
download_backup() {
    local backup_name="${1}"
    local source="${2:-local}"

    case "${source}" in
        s3)
            aws s3 cp "s3://${S3_BUCKET}/backups/${backup_name}" "/tmp/${backup_name}"
            echo "/tmp/${backup_name}"
            ;;
        local)
            echo "${BACKUP_ROOT}/${backup_name}"
            ;;
    esac
}

# バックアップの復号化
decrypt_backup() {
    local encrypted_file="${1}"
    local output_file="${2}"

    gpg --decrypt --output "${output_file}" "${encrypted_file}"
}

# データベースのリストア
restore_database() {
    local backup_file="${1}"
    local db_type="${2}"

    case "${db_type}" in
        postgres)
            gunzip -c "${backup_file}" | psql -U postgres
            ;;
        mysql)
            gunzip -c "${backup_file}" | mysql -u root
            ;;
        mongodb)
            tar -xzf "${backup_file}" -C /tmp
            mongorestore /tmp/mongodb
            ;;
    esac
}

# ファイルのリストア
restore_files() {
    local backup_file="${1}"
    local target_dir="${2:-/}"

    tar -xzf "${backup_file}" -C "${target_dir}"
}

# 対話的リストア
interactive_restore() {
    echo "Available backups:"
    local -a backups
    mapfile -t backups < <(list_backups "local")

    local i=1
    for backup in "${backups[@]}"; do
        echo "${i}. ${backup}"
        ((i++))
    done

    read -rp "Select backup to restore (1-${#backups[@]}): " selection

    if ((selection < 1 || selection > ${#backups[@]})); then
        echo "Invalid selection" >&2
        return 1
    fi

    local selected_backup="${backups[$((selection - 1))]}"

    echo "Restoring from: ${selected_backup}"
    read -rp "Are you sure? This will overwrite existing data (yes/no): " confirmation

    if [[ "${confirmation}" != "yes" ]]; then
        echo "Restore cancelled"
        return 0
    fi

    # リストアの実行
    restore "${selected_backup}"
}
```

---

## ログ管理とローテーション

### カスタムログローテーション

```bash
#!/usr/bin/env bash
# log-rotate.sh - カスタムログローテーション

set -euo pipefail

readonly LOG_DIR="${LOG_DIR:-/var/log/app}"
readonly MAX_SIZE="${MAX_SIZE:-100M}"
readonly MAX_AGE="${MAX_AGE:-30}"
readonly MAX_BACKUPS="${MAX_BACKUPS:-10}"

# ログファイルのローテーション
rotate_log() {
    local log_file="${1}"
    local max_size_bytes

    # サイズの変換（MB to bytes）
    max_size_bytes=$((${MAX_SIZE%M} * 1024 * 1024))

    # ファイルサイズのチェック
    if [[ ! -f "${log_file}" ]]; then
        return 0
    fi

    local file_size
    file_size="$(stat -f%z "${log_file}" 2>/dev/null || stat -c%s "${log_file}" 2>/dev/null)"

    if ((file_size < max_size_bytes)); then
        return 0
    fi

    echo "Rotating ${log_file} (size: $((file_size / 1024 / 1024))MB)"

    # 既存のバックアップをシフト
    for ((i = MAX_BACKUPS; i >= 1; i--)); do
        if [[ -f "${log_file}.${i}" ]]; then
            if ((i == MAX_BACKUPS)); then
                rm "${log_file}.${i}"
            else
                mv "${log_file}.${i}" "${log_file}.$((i + 1))"
            fi
        fi
    done

    # 現在のログをローテーション
    cp "${log_file}" "${log_file}.1"
    : > "${log_file}"

    # 圧縮
    gzip "${log_file}.1"

    # アプリケーションへの通知（例: SIGHUPでログ再オープン）
    if [[ -f /var/run/app.pid ]]; then
        kill -HUP "$(cat /var/run/app.pid)" 2>/dev/null || true
    fi
}

# 古いログの削除
cleanup_old_logs() {
    echo "Cleaning up logs older than ${MAX_AGE} days"

    find "${LOG_DIR}" \
        -name "*.log.*" \
        -type f \
        -mtime "+${MAX_AGE}" \
        -delete \
        -print
}

# ログの統計情報
log_statistics() {
    echo "=== Log Statistics ==="

    local total_size
    total_size="$(du -sh "${LOG_DIR}" | cut -f1)"

    local file_count
    file_count="$(find "${LOG_DIR}" -type f | wc -l)"

    echo "Total size: ${total_size}"
    echo "File count: ${file_count}"

    echo ""
    echo "Largest log files:"
    find "${LOG_DIR}" -type f -exec du -h {} \; | sort -rh | head -10
}

# メインローテーション処理
main_rotation() {
    echo "=== Log Rotation Started ==="

    # すべてのログファイルをローテーション
    find "${LOG_DIR}" -name "*.log" -type f | while read -r log_file; do
        rotate_log "${log_file}"
    done

    cleanup_old_logs
    log_statistics

    echo "=== Log Rotation Completed ==="
}

main_rotation
```

### 集約ログ分析

```bash
#!/usr/bin/env bash
# analyze-logs.sh - ログ分析スクリプト

analyze_error_logs() {
    local log_file="${1}"
    local output_file="${2:-error-report.txt}"

    echo "Analyzing error logs from ${log_file}"

    {
        echo "=== Error Log Analysis ==="
        echo "Generated: $(date)"
        echo ""

        echo "Error Summary:"
        grep -i "error\|exception\|fatal" "${log_file}" | \
            awk '{print $NF}' | \
            sort | \
            uniq -c | \
            sort -rn | \
            head -20

        echo ""
        echo "Error Timeline:"
        grep -i "error" "${log_file}" | \
            awk '{print $1, $2}' | \
            cut -d':' -f1 | \
            uniq -c

        echo ""
        echo "Top Error Messages:"
        grep -i "error" "${log_file}" | \
            sed 's/^.*ERROR/ERROR/' | \
            sort | \
            uniq -c | \
            sort -rn | \
            head -10

    } > "${output_file}"

    echo "Report saved to ${output_file}"
}

# アクセスログ分析
analyze_access_logs() {
    local log_file="${1}"

    echo "=== Access Log Analysis ==="

    echo "Top 10 IPs:"
    awk '{print $1}' "${log_file}" | sort | uniq -c | sort -rn | head -10

    echo ""
    echo "Top 10 URLs:"
    awk '{print $7}' "${log_file}" | sort | uniq -c | sort -rn | head -10

    echo ""
    echo "Status Code Distribution:"
    awk '{print $9}' "${log_file}" | sort | uniq -c | sort

    echo ""
    echo "Traffic by Hour:"
    awk '{print $4}' "${log_file}" | cut -d: -f2 | sort | uniq -c
}
```

---

## データベースメンテナンス

### PostgreSQLメンテナンス

```bash
#!/usr/bin/env bash
# postgres-maintenance.sh - PostgreSQL定期メンテナンス

set -euo pipefail

readonly DB_NAME="${1:?Database name required}"

# VACUUM ANALYZE
perform_vacuum() {
    echo "Performing VACUUM ANALYZE on ${DB_NAME}"

    psql "${DB_NAME}" <<EOF
-- 詳細な統計情報を含むVACUUM
VACUUM (VERBOSE, ANALYZE);

-- テーブルごとの統計
SELECT
    schemaname,
    tablename,
    pg_size_pretty(pg_total_relation_size(schemaname||'.'||tablename)) AS size,
    n_live_tup,
    n_dead_tup
FROM pg_stat_user_tables
ORDER BY pg_total_relation_size(schemaname||'.'||tablename) DESC
LIMIT 20;
EOF
}

# インデックスの再構築
rebuild_indexes() {
    echo "Rebuilding indexes"

    psql "${DB_NAME}" <<EOF
-- 肥大化したインデックスを検出
SELECT
    tablename,
    indexname,
    pg_size_pretty(pg_relation_size(indexname::regclass)) AS index_size
FROM pg_indexes
WHERE schemaname = 'public'
ORDER BY pg_relation_size(indexname::regclass) DESC;

-- インデックスの再構築
REINDEX DATABASE ${DB_NAME};
EOF
}

# データベース統計の更新
update_statistics() {
    echo "Updating database statistics"

    psql "${DB_NAME}" -c "ANALYZE VERBOSE;"
}

# 接続数の監視
monitor_connections() {
    echo "Current database connections:"

    psql "${DB_NAME}" <<EOF
SELECT
    datname,
    pid,
    usename,
    application_name,
    client_addr,
    state,
    query_start,
    state_change
FROM pg_stat_activity
WHERE datname = '${DB_NAME}'
ORDER BY query_start DESC;
EOF
}

# 長時間実行中のクエリの検出
find_long_running_queries() {
    local threshold_minutes="${1:-5}"

    echo "Queries running longer than ${threshold_minutes} minutes:"

    psql "${DB_NAME}" <<EOF
SELECT
    pid,
    now() - query_start AS duration,
    state,
    query
FROM pg_stat_activity
WHERE (now() - query_start) > interval '${threshold_minutes} minutes'
  AND state != 'idle'
ORDER BY duration DESC;
EOF
}

# データベースサイズの監視
check_database_size() {
    echo "Database size information:"

    psql "${DB_NAME}" <<EOF
-- データベース全体のサイズ
SELECT
    pg_database.datname,
    pg_size_pretty(pg_database_size(pg_database.datname)) AS size
FROM pg_database
ORDER BY pg_database_size(pg_database.datname) DESC;

-- テーブルサイズ
SELECT
    schemaname,
    tablename,
    pg_size_pretty(pg_total_relation_size(schemaname||'.'||tablename)) AS total_size,
    pg_size_pretty(pg_relation_size(schemaname||'.'||tablename)) AS table_size,
    pg_size_pretty(pg_total_relation_size(schemaname||'.'||tablename) -
                   pg_relation_size(schemaname||'.'||tablename)) AS index_size
FROM pg_tables
WHERE schemaname = 'public'
ORDER BY pg_total_relation_size(schemaname||'.'||tablename) DESC
LIMIT 20;
EOF
}

# メインメンテナンス
postgres_maintenance() {
    echo "=== PostgreSQL Maintenance Started ==="

    perform_vacuum
    rebuild_indexes
    update_statistics
    monitor_connections
    find_long_running_queries 5
    check_database_size

    echo "=== PostgreSQL Maintenance Completed ==="
}

postgres_maintenance
```

---

## モニタリングとアラート

### リソースモニタリング

```bash
#!/usr/bin/env bash
# resource-monitor.sh - リソース監視スクリプト

set -euo pipefail

readonly METRICS_FILE="/var/log/metrics.log"
readonly ALERT_THRESHOLD_CPU=80
readonly ALERT_THRESHOLD_MEMORY=85
readonly ALERT_THRESHOLD_DISK=90

# メトリクスの収集
collect_metrics() {
    local timestamp
    timestamp="$(date +%s)"

    # CPU使用率
    local cpu_usage
    cpu_usage="$(top -bn1 | grep "Cpu(s)" | awk '{print int($2)}')"

    # メモリ使用率
    local memory_usage
    memory_usage="$(free | grep Mem | awk '{print int($3/$2 * 100)}')"

    # ディスク使用率
    local disk_usage
    disk_usage="$(df -h / | tail -1 | awk '{print int($5)}')"

    # ネットワーク統計
    local network_rx network_tx
    if [[ -f /sys/class/net/eth0/statistics/rx_bytes ]]; then
        network_rx="$(cat /sys/class/net/eth0/statistics/rx_bytes)"
        network_tx="$(cat /sys/class/net/eth0/statistics/tx_bytes)"
    else
        network_rx=0
        network_tx=0
    fi

    # メトリクスの記録
    echo "${timestamp},${cpu_usage},${memory_usage},${disk_usage},${network_rx},${network_tx}" >> "${METRICS_FILE}"

    # しきい値チェック
    check_thresholds "${cpu_usage}" "${memory_usage}" "${disk_usage}"
}

# しきい値のチェックとアラート
check_thresholds() {
    local cpu="${1}"
    local memory="${2}"
    local disk="${3}"

    if ((cpu >= ALERT_THRESHOLD_CPU)); then
        send_alert "HIGH_CPU" "CPU usage is ${cpu}% (threshold: ${ALERT_THRESHOLD_CPU}%)"
    fi

    if ((memory >= ALERT_THRESHOLD_MEMORY)); then
        send_alert "HIGH_MEMORY" "Memory usage is ${memory}% (threshold: ${ALERT_THRESHOLD_MEMORY}%)"
    fi

    if ((disk >= ALERT_THRESHOLD_DISK)); then
        send_alert "HIGH_DISK" "Disk usage is ${disk}% (threshold: ${ALERT_THRESHOLD_DISK}%)"
    fi
}

# アラート送信
send_alert() {
    local alert_type="${1}"
    local message="${2}"

    echo "[ALERT] ${alert_type}: ${message}"

    # Slack通知
    if [[ -n "${SLACK_WEBHOOK:-}" ]]; then
        curl -X POST "${SLACK_WEBHOOK}" \
            -H 'Content-Type: application/json' \
            -d "{
                \"text\": \":warning: ${alert_type}\",
                \"attachments\": [{
                    \"color\": \"danger\",
                    \"text\": \"${message}\",
                    \"footer\": \"$(hostname)\"
                }]
            }"
    fi

    # PagerDuty通知
    if [[ -n "${PAGERDUTY_KEY:-}" && "${alert_type}" == "HIGH_DISK" ]]; then
        curl -X POST "https://events.pagerduty.com/v2/enqueue" \
            -H 'Content-Type: application/json' \
            -d "{
                \"routing_key\": \"${PAGERDUTY_KEY}\",
                \"event_action\": \"trigger\",
                \"payload\": {
                    \"summary\": \"${message}\",
                    \"severity\": \"error\",
                    \"source\": \"$(hostname)\"
                }
            }"
    fi
}

# メトリクスのレポート生成
generate_metrics_report() {
    local period="${1:-24h}"

    echo "=== Metrics Report (last ${period}) ==="

    # 平均CPU使用率
    local avg_cpu
    avg_cpu="$(tail -1000 "${METRICS_FILE}" | awk -F',' '{sum+=$2} END {print int(sum/NR)}')"

    # 平均メモリ使用率
    local avg_memory
    avg_memory="$(tail -1000 "${METRICS_FILE}" | awk -F',' '{sum+=$3} END {print int(sum/NR)}')"

    # 最大ディスク使用率
    local max_disk
    max_disk="$(tail -1000 "${METRICS_FILE}" | awk -F',' '{max=0} {if($4>max) max=$4} END {print max}')"

    echo "Average CPU: ${avg_cpu}%"
    echo "Average Memory: ${avg_memory}%"
    echo "Max Disk: ${max_disk}%"
}

# 監視ループ
monitor_loop() {
    while true; do
        collect_metrics
        sleep 60
    done
}

monitor_loop
```

このガイドでは、メンテナンススクリプトの設計から実装までを網羅し、クリーンアップ、バックアップ、ログ管理、データベースメンテナンス、モニタリングの実践的な手法を提供しました。これらのスクリプトを活用することで、システムの安定性と可用性を維持できます。
