# Automation Scripts 完全ガイド

## 目次
1. [自動化スクリプトの基礎](#自動化スクリプトの基礎)
2. [デプロイメント自動化](#デプロイメント自動化)
3. [CI/CD統合](#cicd統合)
4. [環境管理とプロビジョニング](#環境管理とプロビジョニング)
5. [通知とレポート](#通知とレポート)
6. [並列処理と最適化](#並列処理と最適化)
7. [セキュリティとアクセス制御](#セキュリティとアクセス制御)
8. [実践的な自動化例](#実践的な自動化例)

---

## 自動化スクリプトの基礎

### 自動化の設計原則

```bash
#!/usr/bin/env bash
# automation-framework.sh - 自動化フレームワークの基本構造

set -euo pipefail

# 設定管理
readonly CONFIG_DIR="${CONFIG_DIR:-${HOME}/.automation}"
readonly LOG_DIR="${LOG_DIR:-${CONFIG_DIR}/logs}"
readonly STATE_DIR="${STATE_DIR:-${CONFIG_DIR}/state}"

# ログ設定
setup_logging() {
    local script_name="${1}"
    local log_file="${LOG_DIR}/${script_name}_$(date +%Y%m%d).log"

    mkdir -p "${LOG_DIR}"

    # ログ関数の定義
    log() {
        local level="${1}"
        shift
        local timestamp
        timestamp="$(date +'%Y-%m-%d %H:%M:%S')"
        echo "[${timestamp}] [${level}] $*" | tee -a "${log_file}"
    }

    # エクスポートして子プロセスでも使用可能に
    export -f log
    export LOG_FILE="${log_file}"
}

# 状態管理
save_state() {
    local key="${1}"
    local value="${2}"
    local state_file="${STATE_DIR}/state.json"

    mkdir -p "${STATE_DIR}"

    # jqを使用したJSON操作
    if command -v jq &>/dev/null; then
        local temp_file
        temp_file="$(mktemp)"

        if [[ -f "${state_file}" ]]; then
            jq --arg key "${key}" --arg value "${value}" \
               '.[$key] = $value' "${state_file}" > "${temp_file}"
        else
            echo "{}" | jq --arg key "${key}" --arg value "${value}" \
                          '.[$key] = $value' > "${temp_file}"
        fi

        mv "${temp_file}" "${state_file}"
    fi
}

load_state() {
    local key="${1}"
    local state_file="${STATE_DIR}/state.json"

    if [[ -f "${state_file}" ]] && command -v jq &>/dev/null; then
        jq -r --arg key "${key}" '.[$key] // empty' "${state_file}"
    fi
}

# べき等性の実装
ensure_idempotent() {
    local operation_id="${1}"
    shift
    local operation=("$@")

    local state_key="operation_${operation_id}"
    local last_run
    last_run="$(load_state "${state_key}")"

    # 既に実行済みかチェック
    if [[ -n "${last_run}" ]]; then
        log "INFO" "Operation ${operation_id} already completed at ${last_run}"
        return 0
    fi

    # 操作の実行
    log "INFO" "Executing operation: ${operation_id}"
    "${operation[@]}" || return 1

    # 状態の保存
    save_state "${state_key}" "$(date -u +%Y-%m-%dT%H:%M:%SZ)"
    log "INFO" "Operation ${operation_id} completed"
}
```

### タスクスケジューリング

```bash
#!/usr/bin/env bash
# task-scheduler.sh - タスクスケジューラー

# Cron形式のスケジュール解析
parse_cron_schedule() {
    local schedule="${1}"
    local now
    now="$(date +%s)"

    # 簡易的なcron解析（実際にはより複雑）
    # 例: "0 2 * * *" = 毎日2:00

    local minute hour day month weekday
    read -r minute hour day month weekday <<< "${schedule}"

    # 次回実行時刻の計算（簡略版）
    local next_run
    next_run="$(date -d "today ${hour}:${minute}" +%s)"

    if ((next_run < now)); then
        next_run="$(date -d "tomorrow ${hour}:${minute}" +%s)"
    fi

    echo "${next_run}"
}

# スケジュールされたタスクの実行
run_scheduled_tasks() {
    local tasks_config="${1:-tasks.json}"

    if [[ ! -f "${tasks_config}" ]]; then
        echo "Tasks configuration not found: ${tasks_config}" >&2
        return 1
    fi

    local now
    now="$(date +%s)"

    # 各タスクをチェック
    while IFS= read -r task; do
        local task_id
        task_id="$(echo "${task}" | jq -r '.id')"

        local schedule
        schedule="$(echo "${task}" | jq -r '.schedule')"

        local command
        command="$(echo "${task}" | jq -r '.command')"

        local last_run
        last_run="$(load_state "task_${task_id}_last_run")"
        last_run="${last_run:-0}"

        local next_run
        next_run="$(parse_cron_schedule "${schedule}")"

        # 実行タイミングのチェック
        if ((now >= next_run)) && ((now - last_run > 60)); then
            echo "Running scheduled task: ${task_id}"
            eval "${command}" || echo "Task ${task_id} failed" >&2
            save_state "task_${task_id}_last_run" "${now}"
        fi
    done < <(jq -c '.tasks[]' "${tasks_config}")
}

# タスク定義の例（tasks.json）
cat > tasks.json << 'EOF'
{
  "tasks": [
    {
      "id": "daily-backup",
      "schedule": "0 2 * * *",
      "command": "/usr/local/bin/backup.sh"
    },
    {
      "id": "log-cleanup",
      "schedule": "0 0 * * 0",
      "command": "/usr/local/bin/cleanup-logs.sh"
    }
  ]
}
EOF
```

---

## デプロイメント自動化

### Zero-Downtime デプロイメント

```bash
#!/usr/bin/env bash
# deploy.sh - Zero-Downtimeデプロイメントスクリプト

set -euo pipefail

readonly APP_NAME="${APP_NAME:-myapp}"
readonly DEPLOY_USER="${DEPLOY_USER:-deploy}"
readonly DEPLOY_PATH="${DEPLOY_PATH:-/var/www/${APP_NAME}}"
readonly RELEASES_PATH="${DEPLOY_PATH}/releases"
readonly SHARED_PATH="${DEPLOY_PATH}/shared"
readonly CURRENT_LINK="${DEPLOY_PATH}/current"

# デプロイの前処理
pre_deploy() {
    log "INFO" "Starting pre-deployment checks"

    # ディレクトリ構造の作成
    ssh "${DEPLOY_USER}@${TARGET_HOST}" "
        mkdir -p ${RELEASES_PATH} ${SHARED_PATH}/{logs,tmp,config}
    "

    # ヘルスチェック
    if ! curl -sf "https://${TARGET_HOST}/health" > /dev/null; then
        log "WARNING" "Pre-deployment health check failed"
    fi
}

# 新しいリリースの準備
prepare_release() {
    local release_name="${1}"
    local release_path="${RELEASES_PATH}/${release_name}"

    log "INFO" "Preparing release: ${release_name}"

    # コードのデプロイ
    rsync -avz --delete \
        --exclude='.git' \
        --exclude='node_modules' \
        --exclude='.env' \
        ./ "${DEPLOY_USER}@${TARGET_HOST}:${release_path}/"

    # 共有ファイルのシンボリックリンク作成
    ssh "${DEPLOY_USER}@${TARGET_HOST}" "
        ln -nfs ${SHARED_PATH}/logs ${release_path}/logs
        ln -nfs ${SHARED_PATH}/tmp ${release_path}/tmp
        ln -nfs ${SHARED_PATH}/config/.env ${release_path}/.env
    "

    # 依存関係のインストール
    ssh "${DEPLOY_USER}@${TARGET_HOST}" "
        cd ${release_path}
        npm ci --production
    "
}

# ビルドの実行
build_release() {
    local release_path="${1}"

    log "INFO" "Building release"

    ssh "${DEPLOY_USER}@${TARGET_HOST}" "
        cd ${release_path}
        npm run build
    "
}

# データベースマイグレーション
run_migrations() {
    local release_path="${1}"

    log "INFO" "Running database migrations"

    ssh "${DEPLOY_USER}@${TARGET_HOST}" "
        cd ${release_path}
        npm run migrate
    " || {
        log "ERROR" "Migration failed"
        return 1
    }
}

# アプリケーションの切り替え
switch_release() {
    local release_path="${1}"

    log "INFO" "Switching to new release"

    # アトミックなシンボリックリンクの更新
    ssh "${DEPLOY_USER}@${TARGET_HOST}" "
        ln -nfs ${release_path} ${CURRENT_LINK}.tmp
        mv -Tf ${CURRENT_LINK}.tmp ${CURRENT_LINK}
    "
}

# アプリケーションのリロード
reload_application() {
    log "INFO" "Reloading application"

    # PM2を使用した例
    ssh "${DEPLOY_USER}@${TARGET_HOST}" "
        cd ${CURRENT_LINK}
        pm2 reload ecosystem.config.js --update-env
    " || {
        log "ERROR" "Application reload failed"
        return 1
    }
}

# デプロイ後のヘルスチェック
post_deploy_check() {
    local max_retries=30
    local retry_delay=2

    log "INFO" "Running post-deployment health checks"

    for ((i=1; i<=max_retries; i++)); do
        if curl -sf "https://${TARGET_HOST}/health" > /dev/null; then
            log "INFO" "Health check passed"
            return 0
        fi

        log "WARNING" "Health check failed (attempt ${i}/${max_retries})"
        sleep "${retry_delay}"
    done

    log "ERROR" "Post-deployment health check failed"
    return 1
}

# ロールバック
rollback() {
    log "WARNING" "Rolling back deployment"

    local previous_release
    previous_release="$(ssh "${DEPLOY_USER}@${TARGET_HOST}" "
        ls -t ${RELEASES_PATH} | sed -n 2p
    ")"

    if [[ -z "${previous_release}" ]]; then
        log "ERROR" "No previous release found for rollback"
        return 1
    fi

    switch_release "${RELEASES_PATH}/${previous_release}"
    reload_application
}

# 古いリリースのクリーンアップ
cleanup_old_releases() {
    local keep_releases="${1:-5}"

    log "INFO" "Cleaning up old releases (keeping ${keep_releases})"

    ssh "${DEPLOY_USER}@${TARGET_HOST}" "
        cd ${RELEASES_PATH}
        ls -t | tail -n +$((keep_releases + 1)) | xargs -r rm -rf
    "
}

# メインデプロイプロセス
deploy() {
    local release_name
    release_name="$(date +%Y%m%d%H%M%S)"
    local release_path="${RELEASES_PATH}/${release_name}"

    log "INFO" "=== Starting Deployment ==="
    log "INFO" "Release: ${release_name}"
    log "INFO" "Target: ${TARGET_HOST}"

    pre_deploy || return 1
    prepare_release "${release_name}" || return 1
    build_release "${release_path}" || return 1
    run_migrations "${release_path}" || {
        log "ERROR" "Deployment failed during migrations"
        return 1
    }
    switch_release "${release_path}" || return 1
    reload_application || {
        log "ERROR" "Application reload failed, rolling back"
        rollback
        return 1
    }

    if post_deploy_check; then
        log "INFO" "Deployment successful"
        cleanup_old_releases 5
        log "INFO" "=== Deployment Complete ==="
    else
        log "ERROR" "Post-deployment checks failed, rolling back"
        rollback
        return 1
    fi
}

# 使用例
TARGET_HOST="${1:?Target host required}"
deploy
```

### Blue-Green デプロイメント

```bash
#!/usr/bin/env bash
# blue-green-deploy.sh - Blue-Greenデプロイメント

set -euo pipefail

readonly BLUE_ENV="blue"
readonly GREEN_ENV="green"
readonly LB_CONFIG="/etc/nginx/sites-enabled/app.conf"

# 現在のアクティブ環境を取得
get_active_environment() {
    if grep -q "upstream_${BLUE_ENV}" "${LB_CONFIG}"; then
        echo "${BLUE_ENV}"
    else
        echo "${GREEN_ENV}"
    fi
}

# 非アクティブ環境を取得
get_inactive_environment() {
    local active
    active="$(get_active_environment)"

    if [[ "${active}" == "${BLUE_ENV}" ]]; then
        echo "${GREEN_ENV}"
    else
        echo "${BLUE_ENV}"
    fi
}

# 環境へのデプロイ
deploy_to_environment() {
    local env="${1}"
    local version="${2}"

    log "INFO" "Deploying version ${version} to ${env} environment"

    # Dockerを使用した例
    docker-compose -f "docker-compose.${env}.yml" pull
    docker-compose -f "docker-compose.${env}.yml" up -d --force-recreate

    # ヘルスチェック
    wait_for_healthy "${env}"
}

# ヘルスチェック待機
wait_for_healthy() {
    local env="${1}"
    local url="http://localhost:$( [[ "${env}" == "${BLUE_ENV}" ]] && echo 8001 || echo 8002 )/health"
    local max_retries=30

    log "INFO" "Waiting for ${env} environment to be healthy"

    for ((i=1; i<=max_retries; i++)); do
        if curl -sf "${url}" > /dev/null; then
            log "INFO" "${env} environment is healthy"
            return 0
        fi

        sleep 2
    done

    log "ERROR" "${env} environment failed to become healthy"
    return 1
}

# トラフィックの切り替え
switch_traffic() {
    local target_env="${1}"

    log "INFO" "Switching traffic to ${target_env} environment"

    # Nginxの設定を更新
    sed -i.bak "s/upstream_[a-z]*/upstream_${target_env}/" "${LB_CONFIG}"

    # Nginxをリロード
    nginx -t && nginx -s reload || {
        log "ERROR" "Failed to reload Nginx, reverting"
        mv "${LB_CONFIG}.bak" "${LB_CONFIG}"
        nginx -s reload
        return 1
    }

    log "INFO" "Traffic switched to ${target_env}"
}

# スモークテスト
smoke_test() {
    local env="${1}"

    log "INFO" "Running smoke tests on ${env}"

    # 基本的なエンドポイントテスト
    local endpoints=("/health" "/api/version" "/api/status")

    for endpoint in "${endpoints[@]}"; do
        if ! curl -sf "https://myapp.com${endpoint}" > /dev/null; then
            log "ERROR" "Smoke test failed: ${endpoint}"
            return 1
        fi
    done

    log "INFO" "Smoke tests passed"
}

# Blue-Greenデプロイメントの実行
blue_green_deploy() {
    local version="${1:?Version required}"

    local active_env inactive_env
    active_env="$(get_active_environment)"
    inactive_env="$(get_inactive_environment)"

    log "INFO" "=== Blue-Green Deployment ==="
    log "INFO" "Active: ${active_env}, Target: ${inactive_env}"
    log "INFO" "Version: ${version}"

    # 非アクティブ環境にデプロイ
    deploy_to_environment "${inactive_env}" "${version}" || return 1

    # スモークテスト
    smoke_test "${inactive_env}" || {
        log "ERROR" "Smoke tests failed, aborting deployment"
        return 1
    }

    # トラフィックを切り替え
    switch_traffic "${inactive_env}" || {
        log "ERROR" "Traffic switch failed"
        return 1
    }

    # 本番トラフィックでの検証
    sleep 10
    if ! smoke_test "${inactive_env}"; then
        log "ERROR" "Production validation failed, rolling back"
        switch_traffic "${active_env}"
        return 1
    fi

    log "INFO" "=== Deployment Complete ==="
    log "INFO" "Old environment (${active_env}) is still running for rollback"
}

blue_green_deploy "$@"
```

---

## CI/CD統合

### GitHub Actions統合

```bash
#!/usr/bin/env bash
# ci-deploy.sh - CI/CDパイプライン用デプロイスクリプト

set -euo pipefail

# CI環境の検出
is_ci() {
    [[ "${CI:-false}" == "true" ]]
}

# CI環境変数の検証
validate_ci_environment() {
    local required_vars=(
        "GITHUB_REPOSITORY"
        "GITHUB_SHA"
        "GITHUB_REF"
    )

    for var in "${required_vars[@]}"; do
        if [[ -z "${!var:-}" ]]; then
            echo "Error: Required environment variable ${var} is not set" >&2
            return 1
        fi
    done
}

# デプロイメントステータスの作成
create_deployment_status() {
    local state="${1}"  # pending, success, failure
    local description="${2}"

    if ! is_ci; then
        return 0
    fi

    # GitHub APIを使用
    curl -X POST \
        -H "Authorization: token ${GITHUB_TOKEN}" \
        -H "Accept: application/vnd.github.v3+json" \
        "https://api.github.com/repos/${GITHUB_REPOSITORY}/deployments" \
        -d "{
            \"ref\": \"${GITHUB_SHA}\",
            \"environment\": \"${DEPLOY_ENVIRONMENT}\",
            \"description\": \"${description}\",
            \"auto_merge\": false,
            \"required_contexts\": []
        }"
}

# ビルドアーティファクトのアップロード
upload_artifacts() {
    local artifact_path="${1}"

    if ! is_ci; then
        return 0
    fi

    # GitHub Actionsのartifactsにアップロード
    echo "Uploading artifacts..."

    # actions/upload-artifact@v3 の代わりにスクリプトで実装
    tar -czf "artifacts.tar.gz" -C "$(dirname "${artifact_path}")" "$(basename "${artifact_path}")"

    # S3などへのアップロード例
    if command -v aws &>/dev/null; then
        aws s3 cp "artifacts.tar.gz" \
            "s3://my-artifacts-bucket/${GITHUB_REPOSITORY}/${GITHUB_SHA}/"
    fi
}

# デプロイメントメトリクスの記録
record_deployment_metrics() {
    local status="${1}"
    local duration="${2}"

    # Datadogへメトリクスを送信
    if [[ -n "${DATADOG_API_KEY:-}" ]]; then
        curl -X POST \
            -H "Content-Type: application/json" \
            -H "DD-API-KEY: ${DATADOG_API_KEY}" \
            -d "{
                \"series\": [{
                    \"metric\": \"deployment.duration\",
                    \"points\": [[$(date +%s), ${duration}]],
                    \"type\": \"gauge\",
                    \"tags\": [
                        \"repository:${GITHUB_REPOSITORY}\",
                        \"status:${status}\",
                        \"environment:${DEPLOY_ENVIRONMENT}\"
                    ]
                }]
            }" \
            "https://api.datadoghq.com/api/v1/series"
    fi
}

# Slackへの通知
notify_slack() {
    local status="${1}"
    local message="${2}"

    if [[ -z "${SLACK_WEBHOOK:-}" ]]; then
        return 0
    fi

    local color="good"
    [[ "${status}" == "failure" ]] && color="danger"
    [[ "${status}" == "warning" ]] && color="warning"

    local payload
    payload=$(cat <<EOF
{
    "text": "Deployment ${status}",
    "attachments": [{
        "color": "${color}",
        "fields": [
            {
                "title": "Repository",
                "value": "${GITHUB_REPOSITORY}",
                "short": true
            },
            {
                "title": "Branch",
                "value": "${GITHUB_REF#refs/heads/}",
                "short": true
            },
            {
                "title": "Commit",
                "value": "${GITHUB_SHA:0:7}",
                "short": true
            },
            {
                "title": "Environment",
                "value": "${DEPLOY_ENVIRONMENT}",
                "short": true
            },
            {
                "title": "Message",
                "value": "${message}",
                "short": false
            }
        ],
        "footer": "GitHub Actions",
        "ts": $(date +%s)
    }]
}
EOF
)

    curl -X POST "${SLACK_WEBHOOK}" \
        -H 'Content-Type: application/json' \
        -d "${payload}"
}

# CI/CDデプロイメントの実行
ci_deploy() {
    local start_time
    start_time="$(date +%s)"

    if is_ci; then
        validate_ci_environment || return 1
        create_deployment_status "pending" "Deployment in progress"
    fi

    notify_slack "started" "Deployment started"

    # デプロイメント実行
    if deploy; then
        local end_time
        end_time="$(date +%s)"
        local duration=$((end_time - start_time))

        if is_ci; then
            create_deployment_status "success" "Deployment completed"
            record_deployment_metrics "success" "${duration}"
        fi

        notify_slack "success" "Deployment completed in ${duration}s"
        return 0
    else
        local end_time
        end_time="$(date +%s)"
        local duration=$((end_time - start_time))

        if is_ci; then
            create_deployment_status "failure" "Deployment failed"
            record_deployment_metrics "failure" "${duration}"
        fi

        notify_slack "failure" "Deployment failed after ${duration}s"
        return 1
    fi
}

ci_deploy
```

### GitLab CI統合

```bash
#!/usr/bin/env bash
# gitlab-deploy.sh - GitLab CI用デプロイスクリプト

set -euo pipefail

# GitLab環境変数の活用
validate_gitlab_environment() {
    local required_vars=(
        "CI_PROJECT_NAME"
        "CI_COMMIT_SHA"
        "CI_COMMIT_REF_NAME"
        "CI_ENVIRONMENT_NAME"
    )

    for var in "${required_vars[@]}"; do
        if [[ -z "${!var:-}" ]]; then
            echo "Error: Required GitLab variable ${var} is not set" >&2
            return 1
        fi
    done
}

# GitLab環境URLの設定
set_environment_url() {
    local url="${1}"

    # environment_url.txt に書き込むとGitLabが読み取る
    echo "${url}" > environment_url.txt
}

# GitLabへのメトリクス送信
send_gitlab_metrics() {
    local metric_name="${1}"
    local value="${2}"

    # metrics.txt に書き込むとGitLabが読み取る
    echo "${metric_name} ${value}" >> metrics.txt
}

# .gitlab-ci.yml の例
cat > .gitlab-ci.yml << 'EOF'
stages:
  - build
  - test
  - deploy

variables:
  DEPLOY_ENVIRONMENT: production

build:
  stage: build
  script:
    - npm ci
    - npm run build
  artifacts:
    paths:
      - dist/
    expire_in: 1 hour

test:
  stage: test
  script:
    - npm run test
    - npm run lint

deploy:production:
  stage: deploy
  script:
    - ./gitlab-deploy.sh
  environment:
    name: production
    url: https://myapp.com
    on_stop: stop_production
  only:
    - main

stop_production:
  stage: deploy
  script:
    - ./stop-environment.sh
  when: manual
  environment:
    name: production
    action: stop
EOF
```

---

## 環境管理とプロビジョニング

### インフラストラクチャーのセットアップ

```bash
#!/usr/bin/env bash
# provision.sh - サーバープロビジョニングスクリプト

set -euo pipefail

# システムパッケージのインストール
install_system_packages() {
    log "INFO" "Installing system packages"

    if command -v apt-get &>/dev/null; then
        # Debian/Ubuntu
        apt-get update
        apt-get install -y \
            curl \
            git \
            build-essential \
            nginx \
            postgresql \
            redis-server
    elif command -v yum &>/dev/null; then
        # RHEL/CentOS
        yum update -y
        yum install -y \
            curl \
            git \
            gcc \
            nginx \
            postgresql \
            redis
    fi
}

# Node.jsのインストール
install_nodejs() {
    local node_version="${1:-18}"

    log "INFO" "Installing Node.js ${node_version}"

    # nvmを使用
    curl -o- https://raw.githubusercontent.com/nvm-sh/nvm/v0.39.0/install.sh | bash

    export NVM_DIR="${HOME}/.nvm"
    # shellcheck source=/dev/null
    source "${NVM_DIR}/nvm.sh"

    nvm install "${node_version}"
    nvm use "${node_version}"
    nvm alias default "${node_version}"
}

# アプリケーションユーザーの作成
setup_app_user() {
    local username="${1:-deploy}"

    log "INFO" "Setting up application user: ${username}"

    if ! id "${username}" &>/dev/null; then
        useradd -m -s /bin/bash "${username}"
    fi

    # SSHキーの設定
    local ssh_dir="/home/${username}/.ssh"
    mkdir -p "${ssh_dir}"
    chmod 700 "${ssh_dir}"

    if [[ -n "${SSH_PUBLIC_KEY:-}" ]]; then
        echo "${SSH_PUBLIC_KEY}" >> "${ssh_dir}/authorized_keys"
        chmod 600 "${ssh_dir}/authorized_keys}"
        chown -R "${username}:${username}" "${ssh_dir}"
    fi
}

# Nginxの設定
configure_nginx() {
    local domain="${1}"
    local app_port="${2:-3000}"

    log "INFO" "Configuring Nginx for ${domain}"

    cat > "/etc/nginx/sites-available/${domain}" <<EOF
upstream app_server {
    server 127.0.0.1:${app_port};
}

server {
    listen 80;
    server_name ${domain};

    location / {
        proxy_pass http://app_server;
        proxy_set_header Host \$host;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
    }
}
EOF

    ln -sf "/etc/nginx/sites-available/${domain}" "/etc/nginx/sites-enabled/"
    nginx -t && systemctl reload nginx
}

# SSL証明書の設定（Let's Encrypt）
setup_ssl() {
    local domain="${1}"

    log "INFO" "Setting up SSL for ${domain}"

    # Certbotのインストール
    if ! command -v certbot &>/dev/null; then
        apt-get install -y certbot python3-certbot-nginx
    fi

    # 証明書の取得
    certbot --nginx -d "${domain}" --non-interactive --agree-tos -m "admin@${domain}"

    # 自動更新の設定
    cat > /etc/cron.d/certbot-renew <<EOF
0 3 * * * root certbot renew --quiet --post-hook "systemctl reload nginx"
EOF
}

# ファイアウォールの設定
configure_firewall() {
    log "INFO" "Configuring firewall"

    if command -v ufw &>/dev/null; then
        ufw allow ssh
        ufw allow http
        ufw allow https
        ufw --force enable
    elif command -v firewall-cmd &>/dev/null; then
        firewall-cmd --permanent --add-service=ssh
        firewall-cmd --permanent --add-service=http
        firewall-cmd --permanent --add-service=https
        firewall-cmd --reload
    fi
}

# プロビジョニングの実行
provision() {
    local domain="${1:?Domain required}"

    log "INFO" "=== Starting Server Provisioning ==="

    install_system_packages
    install_nodejs 18
    setup_app_user "deploy"
    configure_nginx "${domain}" 3000
    setup_ssl "${domain}"
    configure_firewall

    log "INFO" "=== Provisioning Complete ==="
}

provision "$@"
```

---

## 通知とレポート

### マルチチャネル通知システム

```bash
#!/usr/bin/env bash
# notify.sh - 統合通知システム

set -euo pipefail

# Slack通知
notify_slack() {
    local message="${1}"
    local level="${2:-info}"

    if [[ -z "${SLACK_WEBHOOK:-}" ]]; then
        return 0
    fi

    local color="good"
    case "${level}" in
        error) color="danger" ;;
        warning) color="warning" ;;
        *) color="good" ;;
    esac

    curl -X POST "${SLACK_WEBHOOK}" \
        -H 'Content-Type: application/json' \
        -d "{
            \"text\": \"${message}\",
            \"attachments\": [{
                \"color\": \"${color}\"
            }]
        }"
}

# Discord通知
notify_discord() {
    local message="${1}"

    if [[ -z "${DISCORD_WEBHOOK:-}" ]]; then
        return 0
    fi

    curl -X POST "${DISCORD_WEBHOOK}" \
        -H 'Content-Type: application/json' \
        -d "{
            \"content\": \"${message}\"
        }"
}

# Microsoft Teams通知
notify_teams() {
    local message="${1}"
    local level="${2:-info}"

    if [[ -z "${TEAMS_WEBHOOK:-}" ]]; then
        return 0
    fi

    local theme_color="0078D4"
    case "${level}" in
        error) theme_color="D13438" ;;
        warning) theme_color="FFB900" ;;
        success) theme_color="107C10" ;;
    esac

    curl -X POST "${TEAMS_WEBHOOK}" \
        -H 'Content-Type: application/json' \
        -d "{
            \"@type\": \"MessageCard\",
            \"@context\": \"https://schema.org/extensions\",
            \"themeColor\": \"${theme_color}\",
            \"text\": \"${message}\"
        }"
}

# メール通知
notify_email() {
    local subject="${1}"
    local body="${2}"
    local recipients="${3}"

    if ! command -v mail &>/dev/null; then
        return 0
    fi

    echo "${body}" | mail -s "${subject}" "${recipients}"
}

# PagerDuty通知（インシデント作成）
notify_pagerduty() {
    local summary="${1}"
    local severity="${2:-error}"

    if [[ -z "${PAGERDUTY_ROUTING_KEY:-}" ]]; then
        return 0
    fi

    curl -X POST "https://events.pagerduty.com/v2/enqueue" \
        -H 'Content-Type: application/json' \
        -d "{
            \"routing_key\": \"${PAGERDUTY_ROUTING_KEY}\",
            \"event_action\": \"trigger\",
            \"payload\": {
                \"summary\": \"${summary}\",
                \"severity\": \"${severity}\",
                \"source\": \"$(hostname)\"
            }
        }"
}

# 統合通知関数
send_notification() {
    local message="${1}"
    local level="${2:-info}"
    local channels="${3:-slack}"

    IFS=',' read -ra channel_array <<< "${channels}"

    for channel in "${channel_array[@]}"; do
        case "${channel}" in
            slack) notify_slack "${message}" "${level}" ;;
            discord) notify_discord "${message}" ;;
            teams) notify_teams "${message}" "${level}" ;;
            pagerduty) notify_pagerduty "${message}" "${level}" ;;
            *) echo "Unknown notification channel: ${channel}" >&2 ;;
        esac
    done
}

# 使用例
send_notification "Deployment completed successfully" "success" "slack,teams"
```

### レポート生成

```bash
#!/usr/bin/env bash
# generate-report.sh - デプロイメントレポート生成

generate_deployment_report() {
    local start_time="${1}"
    local end_time="${2}"
    local status="${3}"
    local report_file="${4:-deployment-report.html}"

    local duration=$((end_time - start_time))
    local duration_formatted
    duration_formatted="$(date -u -d @${duration} +%H:%M:%S)"

    cat > "${report_file}" <<EOF
<!DOCTYPE html>
<html>
<head>
    <title>Deployment Report</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .header { background: #2196F3; color: white; padding: 20px; }
        .success { background: #4CAF50; }
        .failure { background: #f44336; }
        .metric { padding: 10px; margin: 10px 0; border-left: 4px solid #2196F3; }
        table { border-collapse: collapse; width: 100%; margin: 20px 0; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background-color: #f2f2f2; }
    </style>
</head>
<body>
    <div class="header ${status}">
        <h1>Deployment Report</h1>
        <p>Status: ${status^^}</p>
        <p>Duration: ${duration_formatted}</p>
    </div>

    <div class="metric">
        <h3>Deployment Details</h3>
        <table>
            <tr><th>Property</th><th>Value</th></tr>
            <tr><td>Repository</td><td>${GITHUB_REPOSITORY:-N/A}</td></tr>
            <tr><td>Commit</td><td>${GITHUB_SHA:-N/A}</td></tr>
            <tr><td>Branch</td><td>${GITHUB_REF:-N/A}</td></tr>
            <tr><td>Environment</td><td>${DEPLOY_ENVIRONMENT:-N/A}</td></tr>
            <tr><td>Started</td><td>$(date -d @${start_time} +'%Y-%m-%d %H:%M:%S')</td></tr>
            <tr><td>Completed</td><td>$(date -d @${end_time} +'%Y-%m-%d %H:%M:%S')</td></tr>
        </table>
    </div>
</body>
</html>
EOF

    echo "Report generated: ${report_file}"
}
```

---

## 並列処理と最適化

### GNU Parallelを使用した並列処理

```bash
#!/usr/bin/env bash
# parallel-deployment.sh - 並列デプロイメント

deploy_to_server() {
    local server="${1}"
    local version="${2}"

    echo "Deploying to ${server}..."

    ssh "${server}" "
        cd /var/www/app
        git fetch
        git checkout ${version}
        npm install
        pm2 reload all
    " && echo "✓ ${server} deployed" || echo "✗ ${server} failed"
}

export -f deploy_to_server

# サーバーリスト
servers=(
    "app1.example.com"
    "app2.example.com"
    "app3.example.com"
    "app4.example.com"
)

# 並列デプロイメント（最大2台同時）
parallel -j 2 deploy_to_server ::: "${servers[@]}" ::: "v1.2.3"
```

### デプロイメントキャッシュの活用

```bash
# キャッシュを使用した高速デプロイ
cache_dependencies() {
    local cache_key="${1}"
    local cache_dir="${HOME}/.deployment-cache"

    mkdir -p "${cache_dir}"

    # 依存関係のハッシュ
    local deps_hash
    deps_hash="$(cat package.json package-lock.json | sha256sum | cut -d' ' -f1)"
    local cache_file="${cache_dir}/${cache_key}_${deps_hash}.tar.gz"

    if [[ -f "${cache_file}" ]]; then
        echo "Restoring dependencies from cache"
        tar -xzf "${cache_file}"
        return 0
    fi

    echo "Installing dependencies"
    npm ci

    echo "Caching dependencies"
    tar -czf "${cache_file}" node_modules/
}
```

このガイドでは、自動化スクリプトの設計から実装までを網羅し、実践的なデプロイメント自動化、CI/CD統合、環境管理の手法を提供しました。これらのパターンを活用することで、信頼性が高く効率的な自動化システムを構築できます。
