# Shell Scripting 完全ガイド

## 目次
1. [Shell Scriptingの基礎](#shell-scriptingの基礎)
2. [Bashベストプラクティス](#bashベストプラクティス)
3. [エラーハンドリング](#エラーハンドリング)
4. [パラメータと引数の処理](#パラメータと引数の処理)
5. [ファイル操作](#ファイル操作)
6. [プロセス管理](#プロセス管理)
7. [デバッグとテスト](#デバッグとテスト)
8. [実践的なスクリプト例](#実践的なスクリプト例)

---

## Shell Scriptingの基礎

### Shebangとスクリプトの基本構造

```bash
#!/usr/bin/env bash
# スクリプトの説明: データベースバックアップスクリプト
# 作成者: DevOps Team
# 最終更新: 2025-01-15
# 使用方法: ./backup.sh [database_name]

set -euo pipefail  # 厳格なエラーハンドリング

# グローバル変数
readonly SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
readonly SCRIPT_NAME="$(basename "${BASH_SOURCE[0]}")"
readonly TIMESTAMP="$(date +%Y%m%d_%H%M%S)"

# メイン処理
main() {
    echo "Starting backup at ${TIMESTAMP}"
    # 処理内容
}

# スクリプトが直接実行された場合のみmainを呼び出す
if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
    main "$@"
fi
```

### 変数とデータ型

```bash
# ❌ 悪い例: 変数の扱い
database=mydb
backup_dir=/tmp/backup
config_file=config.ini

# ✅ 良い例: 適切な変数の扱い
readonly DATABASE_NAME="mydb"  # 変更不可の定数
readonly BACKUP_DIR="/tmp/backup"  # パスは絶対パスで
readonly CONFIG_FILE="${SCRIPT_DIR}/config.ini"  # スクリプト相対パス

# 環境変数のデフォルト値
LOG_LEVEL="${LOG_LEVEL:-INFO}"
MAX_RETRIES="${MAX_RETRIES:-3}"
TIMEOUT="${TIMEOUT:-30}"

# 配列の使用
declare -a backup_targets=("db1" "db2" "db3")
declare -A config=(
    ["host"]="localhost"
    ["port"]="5432"
    ["user"]="admin"
)

# 変数の展開
echo "Database: ${DATABASE_NAME}"
echo "Config: ${config[host]}:${config[port]}"
```

### 文字列操作

```bash
#!/usr/bin/env bash

string_operations() {
    local text="Hello, World!"

    # 長さ
    echo "Length: ${#text}"  # 13

    # 部分文字列
    echo "${text:0:5}"  # Hello
    echo "${text:7}"    # World!

    # 置換
    echo "${text/World/Bash}"      # Hello, Bash! (最初の一致)
    echo "${text//o/0}"            # Hell0, W0rld! (すべて置換)

    # 大文字・小文字変換
    echo "${text^^}"  # HELLO, WORLD!
    echo "${text,,}"  # hello, world!

    # トリミング
    local padded="  spaced  "
    echo "${padded#"${padded%%[![:space:]]*}"}"  # 左トリム

    # パス操作
    local filepath="/var/log/app.log"
    echo "Dir: $(dirname "${filepath}")"    # /var/log
    echo "Base: $(basename "${filepath}")"  # app.log
    echo "Ext: ${filepath##*.}"             # log
    echo "Name: ${filepath%.*}"             # /var/log/app
}
```

---

## Bashベストプラクティス

### 厳格なエラーハンドリングモード

```bash
#!/usr/bin/env bash

# set -e: エラー時に即座に終了
# set -u: 未定義変数の使用時にエラー
# set -o pipefail: パイプライン内のエラーを検出
# set -x: デバッグモード（実行コマンドを表示）
set -euo pipefail

# IFS (Internal Field Separator) の設定
IFS=$'\n\t'

# trap を使用したクリーンアップ
cleanup() {
    local exit_code=$?
    echo "Cleaning up..."
    rm -f "${TEMP_FILE:-}"
    exit "${exit_code}"
}

trap cleanup EXIT ERR INT TERM

# グローバル変数の宣言
declare -r TEMP_FILE="$(mktemp)"
```

### 関数設計のベストプラクティス

```bash
# ❌ 悪い例: グローバル変数に依存
backup_database() {
    cp $database_path $backup_path
    echo "Backup complete"
}

# ✅ 良い例: 関数のカプセル化
backup_database() {
    local source_path="${1:?Source path required}"
    local dest_path="${2:?Destination path required}"
    local -r timestamp="$(date +%Y%m%d_%H%M%S)"

    # 入力検証
    if [[ ! -f "${source_path}" ]]; then
        echo "Error: Source file not found: ${source_path}" >&2
        return 1
    fi

    if [[ ! -d "$(dirname "${dest_path}")" ]]; then
        echo "Error: Destination directory not found" >&2
        return 2
    fi

    # バックアップの実行
    cp "${source_path}" "${dest_path}.${timestamp}" || {
        echo "Error: Backup failed" >&2
        return 3
    }

    echo "Backup completed: ${dest_path}.${timestamp}"
    return 0
}

# 使用例
backup_database "/var/lib/postgres/data" "/backup/postgres" || {
    echo "Backup operation failed"
    exit 1
}
```

### ShellCheck対応

```bash
# ShellCheckの警告に対応したコード

# ❌ SC2086: Quote to prevent word splitting
files=$(ls *.txt)
rm $files

# ✅ 修正版
while IFS= read -r -d '' file; do
    rm "${file}"
done < <(find . -name "*.txt" -print0)

# ❌ SC2046: Quote to prevent word splitting
echo $(cat file.txt)

# ✅ 修正版
echo "$(cat file.txt)"

# ❌ SC2006: Use $(...) notation instead of legacy backticks
result=`command`

# ✅ 修正版
result="$(command)"

# ❌ SC2155: Declare and assign separately
local result="$(complex_command)"

# ✅ 修正版
local result
result="$(complex_command)"

# ShellCheckディレクティブの使用
# shellcheck disable=SC2034  # 使用されていない変数を許可
UNUSED_VAR="value"

# shellcheck source=/dev/null  # 動的ソースを許可
source "${CONFIG_FILE}"
```

---

## エラーハンドリング

### 包括的なエラーハンドリング

```bash
#!/usr/bin/env bash

set -euo pipefail

# エラーハンドラ
error_handler() {
    local line_number="${1}"
    local bash_lineno="${2}"
    local last_command="${3}"
    local exit_code="${4}"

    echo "Error occurred in script ${SCRIPT_NAME}" >&2
    echo "  Line: ${line_number}" >&2
    echo "  Command: ${last_command}" >&2
    echo "  Exit code: ${exit_code}" >&2

    # Slackに通知（オプション）
    if command -v curl &> /dev/null && [[ -n "${SLACK_WEBHOOK:-}" ]]; then
        curl -X POST "${SLACK_WEBHOOK}" \
            -H 'Content-Type: application/json' \
            -d "{
                \"text\": \"Script Error: ${SCRIPT_NAME}\",
                \"attachments\": [{
                    \"color\": \"danger\",
                    \"fields\": [
                        {\"title\": \"Line\", \"value\": \"${line_number}\", \"short\": true},
                        {\"title\": \"Exit Code\", \"value\": \"${exit_code}\", \"short\": true},
                        {\"title\": \"Command\", \"value\": \"${last_command}\"}
                    ]
                }]
            }"
    fi
}

trap 'error_handler ${LINENO} ${BASH_LINENO} "${BASH_COMMAND}" $?' ERR

# リトライロジック
retry_command() {
    local -r max_attempts="${1}"
    local -r delay="${2}"
    shift 2
    local -r command=("$@")
    local attempt=1

    until "${command[@]}"; do
        if ((attempt >= max_attempts)); then
            echo "Command failed after ${max_attempts} attempts" >&2
            return 1
        fi

        echo "Attempt ${attempt} failed. Retrying in ${delay}s..." >&2
        sleep "${delay}"
        ((attempt++))
    done

    echo "Command succeeded on attempt ${attempt}"
    return 0
}

# 使用例
retry_command 3 5 curl -f "https://api.example.com/health" || {
    echo "Health check failed"
    exit 1
}
```

### バリデーションパターン

```bash
validate_input() {
    local value="${1:?Value required}"
    local validation_type="${2:?Validation type required}"

    case "${validation_type}" in
        email)
            if [[ ! "${value}" =~ ^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$ ]]; then
                echo "Invalid email: ${value}" >&2
                return 1
            fi
            ;;

        url)
            if [[ ! "${value}" =~ ^https?:// ]]; then
                echo "Invalid URL: ${value}" >&2
                return 1
            fi
            ;;

        ip)
            if [[ ! "${value}" =~ ^([0-9]{1,3}\.){3}[0-9]{1,3}$ ]]; then
                echo "Invalid IP: ${value}" >&2
                return 1
            fi
            ;;

        port)
            if [[ ! "${value}" =~ ^[0-9]+$ ]] || ((value < 1 || value > 65535)); then
                echo "Invalid port: ${value}" >&2
                return 1
            fi
            ;;

        path)
            if [[ ! -e "${value}" ]]; then
                echo "Path not found: ${value}" >&2
                return 1
            fi
            ;;

        *)
            echo "Unknown validation type: ${validation_type}" >&2
            return 2
            ;;
    esac

    return 0
}

# 使用例
validate_input "user@example.com" "email" || exit 1
validate_input "https://example.com" "url" || exit 1
validate_input "192.168.1.1" "ip" || exit 1
validate_input "8080" "port" || exit 1
```

---

## パラメータと引数の処理

### getopts による引数解析

```bash
#!/usr/bin/env bash

usage() {
    cat << EOF
Usage: ${SCRIPT_NAME} [OPTIONS]

Options:
    -h, --help              Show this help message
    -v, --verbose           Enable verbose output
    -f, --file FILE         Input file path
    -o, --output DIR        Output directory
    -n, --dry-run           Dry run mode
    -c, --config FILE       Configuration file

Examples:
    ${SCRIPT_NAME} -f input.txt -o ./output
    ${SCRIPT_NAME} --file data.csv --output /tmp/results --verbose

EOF
    exit 0
}

# デフォルト値
VERBOSE=false
DRY_RUN=false
INPUT_FILE=""
OUTPUT_DIR="./output"
CONFIG_FILE="${SCRIPT_DIR}/config.ini"

# 長いオプションをサポート
parse_arguments() {
    while [[ $# -gt 0 ]]; do
        case "${1}" in
            -h|--help)
                usage
                ;;
            -v|--verbose)
                VERBOSE=true
                shift
                ;;
            -f|--file)
                INPUT_FILE="${2:?Missing file argument}"
                shift 2
                ;;
            -o|--output)
                OUTPUT_DIR="${2:?Missing output directory}"
                shift 2
                ;;
            -n|--dry-run)
                DRY_RUN=true
                shift
                ;;
            -c|--config)
                CONFIG_FILE="${2:?Missing config file}"
                shift 2
                ;;
            --)
                shift
                break
                ;;
            -*)
                echo "Unknown option: ${1}" >&2
                usage
                ;;
            *)
                break
                ;;
        esac
    done

    # 必須パラメータの検証
    if [[ -z "${INPUT_FILE}" ]]; then
        echo "Error: Input file is required" >&2
        usage
    fi
}

parse_arguments "$@"

# デバッグ出力
if [[ "${VERBOSE}" == true ]]; then
    echo "Configuration:"
    echo "  Input file: ${INPUT_FILE}"
    echo "  Output dir: ${OUTPUT_DIR}"
    echo "  Config file: ${CONFIG_FILE}"
    echo "  Dry run: ${DRY_RUN}"
fi
```

### サブコマンドパターン

```bash
#!/usr/bin/env bash
# deploy-tool.sh - デプロイツールの例

set -euo pipefail

readonly SCRIPT_NAME="$(basename "${0}")"

# サブコマンド: deploy
cmd_deploy() {
    local environment="${1:?Environment required}"
    local version="${2:-latest}"

    echo "Deploying version ${version} to ${environment}..."

    case "${environment}" in
        production)
            echo "Deploying to production..."
            # デプロイ処理
            ;;
        staging)
            echo "Deploying to staging..."
            # デプロイ処理
            ;;
        *)
            echo "Unknown environment: ${environment}" >&2
            return 1
            ;;
    esac
}

# サブコマンド: rollback
cmd_rollback() {
    local environment="${1:?Environment required}"
    local version="${2:?Version required}"

    echo "Rolling back ${environment} to version ${version}..."
    # ロールバック処理
}

# サブコマンド: status
cmd_status() {
    local environment="${1:-all}"

    echo "Status for ${environment}:"
    # ステータス確認処理
}

# サブコマンド: logs
cmd_logs() {
    local environment="${1:?Environment required}"
    local -r lines="${2:-100}"

    echo "Showing last ${lines} lines for ${environment}:"
    # ログ表示処理
}

# メインディスパッチャー
main() {
    if [[ $# -eq 0 ]]; then
        echo "Usage: ${SCRIPT_NAME} <command> [arguments]" >&2
        echo "" >&2
        echo "Commands:" >&2
        echo "  deploy <env> [version]   Deploy to environment" >&2
        echo "  rollback <env> <version> Rollback to version" >&2
        echo "  status [env]             Show deployment status" >&2
        echo "  logs <env> [lines]       Show logs" >&2
        exit 1
    fi

    local subcommand="${1}"
    shift

    case "${subcommand}" in
        deploy)
            cmd_deploy "$@"
            ;;
        rollback)
            cmd_rollback "$@"
            ;;
        status)
            cmd_status "$@"
            ;;
        logs)
            cmd_logs "$@"
            ;;
        *)
            echo "Unknown command: ${subcommand}" >&2
            exit 1
            ;;
    esac
}

main "$@"
```

---

## ファイル操作

### 安全なファイル処理

```bash
#!/usr/bin/env bash

# 一時ファイルの安全な作成
create_temp_file() {
    local template="${1:-tmp.XXXXXXXXXX}"
    local temp_file

    temp_file="$(mktemp "/tmp/${template}")" || {
        echo "Failed to create temp file" >&2
        return 1
    }

    # クリーンアップの登録
    trap "rm -f '${temp_file}'" EXIT

    echo "${temp_file}"
}

# ファイルロック
acquire_lock() {
    local lock_file="${1:?Lock file required}"
    local timeout="${2:-30}"
    local waited=0

    while ! mkdir "${lock_file}" 2>/dev/null; do
        if ((waited >= timeout)); then
            echo "Failed to acquire lock after ${timeout}s" >&2
            return 1
        fi

        sleep 1
        ((waited++))
    done

    # ロック解放の登録
    trap "rm -rf '${lock_file}'" EXIT

    echo "Lock acquired: ${lock_file}"
}

# アトミックなファイル書き込み
atomic_write() {
    local target_file="${1:?Target file required}"
    local content="${2:?Content required}"
    local temp_file

    temp_file="$(mktemp "${target_file}.XXXXXXXXXX")" || return 1

    # 一時ファイルに書き込み
    printf '%s\n' "${content}" > "${temp_file}" || {
        rm -f "${temp_file}"
        return 1
    }

    # アトミックに置換
    mv "${temp_file}" "${target_file}" || {
        rm -f "${temp_file}"
        return 1
    }

    echo "File written atomically: ${target_file}"
}

# ファイル変更の監視
watch_file() {
    local file_path="${1:?File path required}"
    local interval="${2:-5}"
    local last_mtime=""

    while true; do
        if [[ -f "${file_path}" ]]; then
            local current_mtime
            current_mtime="$(stat -f %m "${file_path}" 2>/dev/null || stat -c %Y "${file_path}" 2>/dev/null)"

            if [[ -n "${last_mtime}" && "${current_mtime}" != "${last_mtime}" ]]; then
                echo "File modified: ${file_path}"
                # 変更時の処理
                process_file "${file_path}"
            fi

            last_mtime="${current_mtime}"
        fi

        sleep "${interval}"
    done
}
```

### ディレクトリ操作

```bash
# ディレクトリの安全な作成
ensure_directory() {
    local dir_path="${1:?Directory path required}"
    local mode="${2:-0755}"

    if [[ ! -d "${dir_path}" ]]; then
        mkdir -p "${dir_path}" || {
            echo "Failed to create directory: ${dir_path}" >&2
            return 1
        }
        chmod "${mode}" "${dir_path}"
        echo "Created directory: ${dir_path}"
    fi
}

# ディレクトリの同期
sync_directories() {
    local source="${1:?Source directory required}"
    local dest="${2:?Destination directory required}"

    if [[ ! -d "${source}" ]]; then
        echo "Source directory not found: ${source}" >&2
        return 1
    fi

    ensure_directory "${dest}" || return 1

    rsync -av --delete \
        --exclude='.git' \
        --exclude='node_modules' \
        --exclude='.DS_Store' \
        "${source}/" "${dest}/" || {
        echo "Directory sync failed" >&2
        return 1
    }

    echo "Synced: ${source} -> ${dest}"
}

# ディレクトリサイズの計算
calculate_dir_size() {
    local dir_path="${1:?Directory path required}"
    local human_readable="${2:-false}"

    if [[ ! -d "${dir_path}" ]]; then
        echo "Directory not found: ${dir_path}" >&2
        return 1
    fi

    if [[ "${human_readable}" == true ]]; then
        du -sh "${dir_path}" | cut -f1
    else
        du -s "${dir_path}" | cut -f1
    fi
}

# 古いファイルの削除
cleanup_old_files() {
    local dir_path="${1:?Directory path required}"
    local days_old="${2:-7}"
    local pattern="${3:-*}"

    echo "Cleaning up files older than ${days_old} days in ${dir_path}"

    find "${dir_path}" \
        -type f \
        -name "${pattern}" \
        -mtime "+${days_old}" \
        -delete \
        -print || {
        echo "Cleanup failed" >&2
        return 1
    }
}
```

---

## プロセス管理

### バックグラウンドジョブ管理

```bash
#!/usr/bin/env bash

# 並列処理の実行
parallel_execute() {
    local -r max_jobs="${1:?Max jobs required}"
    shift
    local -a tasks=("$@")
    local -a pids=()

    for task in "${tasks[@]}"; do
        # ジョブ数の制限
        while ((${#pids[@]} >= max_jobs)); do
            for i in "${!pids[@]}"; do
                if ! kill -0 "${pids[i]}" 2>/dev/null; then
                    wait "${pids[i]}"
                    unset 'pids[i]'
                fi
            done
            pids=("${pids[@]}")  # 配列の再インデックス
            sleep 0.1
        done

        # タスクの起動
        eval "${task}" &
        pids+=($!)
    done

    # 全ジョブの完了を待機
    for pid in "${pids[@]}"; do
        wait "${pid}"
    done

    echo "All parallel tasks completed"
}

# 使用例
tasks=(
    "process_file file1.txt"
    "process_file file2.txt"
    "process_file file3.txt"
    "process_file file4.txt"
)

parallel_execute 2 "${tasks[@]}"
```

### プロセス監視とヘルスチェック

```bash
# プロセスの監視
monitor_process() {
    local process_name="${1:?Process name required}"
    local restart_command="${2:?Restart command required}"
    local check_interval="${3:-60}"

    while true; do
        if ! pgrep -f "${process_name}" > /dev/null; then
            echo "Process ${process_name} not running. Restarting..."

            # 再起動
            eval "${restart_command}" || {
                echo "Failed to restart ${process_name}" >&2
                # 通知処理
            }
        fi

        sleep "${check_interval}"
    done
}

# リソース使用量の監視
check_resource_usage() {
    local pid="${1:?PID required}"
    local max_memory_mb="${2:-1024}"
    local max_cpu_percent="${3:-80}"

    # メモリ使用量（MB）
    local memory_mb
    memory_mb="$(ps -p "${pid}" -o rss= | awk '{print int($1/1024)}')"

    # CPU使用率
    local cpu_percent
    cpu_percent="$(ps -p "${pid}" -o %cpu= | awk '{print int($1)}')"

    echo "Process ${pid} - Memory: ${memory_mb}MB, CPU: ${cpu_percent}%"

    # しきい値チェック
    if ((memory_mb > max_memory_mb)); then
        echo "Warning: Memory usage exceeded ${max_memory_mb}MB" >&2
        return 1
    fi

    if ((cpu_percent > max_cpu_percent)); then
        echo "Warning: CPU usage exceeded ${max_cpu_percent}%" >&2
        return 2
    fi

    return 0
}

# グレースフルシャットダウン
graceful_shutdown() {
    local pid="${1:?PID required}"
    local timeout="${2:-30}"

    echo "Sending SIGTERM to process ${pid}"
    kill -TERM "${pid}" 2>/dev/null || return 1

    # プロセス終了を待機
    local waited=0
    while kill -0 "${pid}" 2>/dev/null; do
        if ((waited >= timeout)); then
            echo "Process did not terminate. Sending SIGKILL..."
            kill -KILL "${pid}" 2>/dev/null
            return 2
        fi

        sleep 1
        ((waited++))
    done

    echo "Process ${pid} terminated gracefully"
    return 0
}
```

---

## デバッグとテスト

### デバッグテクニック

```bash
#!/usr/bin/env bash

# デバッグモードの切り替え
DEBUG="${DEBUG:-false}"

debug() {
    if [[ "${DEBUG}" == true ]]; then
        echo "[DEBUG] $*" >&2
    fi
}

# スタックトレースの出力
print_stack_trace() {
    local frame=0
    echo "Stack trace:" >&2
    while caller $frame; do
        ((frame++))
    done
}

# 関数の実行時間計測
time_function() {
    local func_name="${1:?Function name required}"
    shift
    local start_time end_time duration

    start_time="$(date +%s%N)"
    debug "Starting ${func_name}"

    "${func_name}" "$@"
    local exit_code=$?

    end_time="$(date +%s%N)"
    duration=$(((end_time - start_time) / 1000000))  # ミリ秒

    debug "${func_name} completed in ${duration}ms (exit code: ${exit_code})"

    return "${exit_code}"
}

# 変数のダンプ
dump_vars() {
    echo "=== Variable Dump ===" >&2
    local var_name
    for var_name in "$@"; do
        if declare -p "${var_name}" &>/dev/null; then
            declare -p "${var_name}" >&2
        else
            echo "${var_name}: <not set>" >&2
        fi
    done
    echo "====================" >&2
}

# 使用例
example_function() {
    local input="${1}"
    debug "Processing input: ${input}"

    dump_vars "input" "DEBUG" "SCRIPT_NAME"

    # 処理内容
    sleep 0.5

    return 0
}

time_function example_function "test data"
```

### ユニットテスト

```bash
#!/usr/bin/env bash
# test_functions.sh - 関数のテストスクリプト

# テストフレームワーク
assert_equals() {
    local expected="${1}"
    local actual="${2}"
    local message="${3:-}"

    if [[ "${expected}" == "${actual}" ]]; then
        echo "✓ PASS${message:+: ${message}}"
        return 0
    else
        echo "✗ FAIL${message:+: ${message}}" >&2
        echo "  Expected: ${expected}" >&2
        echo "  Actual:   ${actual}" >&2
        return 1
    fi
}

assert_not_equals() {
    local expected="${1}"
    local actual="${2}"
    local message="${3:-}"

    if [[ "${expected}" != "${actual}" ]]; then
        echo "✓ PASS${message:+: ${message}}"
        return 0
    else
        echo "✗ FAIL${message:+: ${message}}" >&2
        echo "  Expected NOT: ${expected}" >&2
        echo "  Actual:       ${actual}" >&2
        return 1
    fi
}

assert_true() {
    local condition="${1}"
    local message="${2:-}"

    if eval "${condition}"; then
        echo "✓ PASS${message:+: ${message}}"
        return 0
    else
        echo "✗ FAIL${message:+: ${message}}" >&2
        echo "  Condition failed: ${condition}" >&2
        return 1
    fi
}

# テスト対象の関数
add() {
    local a="${1}"
    local b="${2}"
    echo $((a + b))
}

is_valid_email() {
    local email="${1}"
    [[ "${email}" =~ ^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$ ]]
}

# テストスイート
run_tests() {
    local tests_passed=0
    local tests_failed=0

    echo "Running tests..."
    echo ""

    # Test: add function
    echo "Test suite: add()"
    assert_equals "5" "$(add 2 3)" "add(2, 3) should return 5" && ((tests_passed++)) || ((tests_failed++))
    assert_equals "0" "$(add -5 5)" "add(-5, 5) should return 0" && ((tests_passed++)) || ((tests_failed++))
    echo ""

    # Test: is_valid_email
    echo "Test suite: is_valid_email()"
    assert_true "is_valid_email 'test@example.com'" "Valid email should pass" && ((tests_passed++)) || ((tests_failed++))
    assert_true "! is_valid_email 'invalid-email'" "Invalid email should fail" && ((tests_passed++)) || ((tests_failed++))
    echo ""

    # サマリー
    echo "========================"
    echo "Tests passed: ${tests_passed}"
    echo "Tests failed: ${tests_failed}"
    echo "========================"

    return "${tests_failed}"
}

run_tests
exit $?
```

---

## 実践的なスクリプト例

### データベースバックアップスクリプト

```bash
#!/usr/bin/env bash
# db-backup.sh - PostgreSQLデータベースバックアップスクリプト

set -euo pipefail

readonly SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
readonly SCRIPT_NAME="$(basename "${BASH_SOURCE[0]}")"
readonly TIMESTAMP="$(date +%Y%m%d_%H%M%S)"

# 設定
readonly DB_HOST="${DB_HOST:-localhost}"
readonly DB_PORT="${DB_PORT:-5432}"
readonly DB_USER="${DB_USER:-postgres}"
readonly DB_NAME="${1:?Database name required}"
readonly BACKUP_DIR="${BACKUP_DIR:-/var/backups/postgres}"
readonly RETENTION_DAYS="${RETENTION_DAYS:-7}"

# ログ設定
readonly LOG_FILE="${BACKUP_DIR}/backup.log"

log() {
    local level="${1}"
    shift
    echo "[$(date +'%Y-%m-%d %H:%M:%S')] [${level}] $*" | tee -a "${LOG_FILE}"
}

# バックアップディレクトリの作成
ensure_backup_dir() {
    if [[ ! -d "${BACKUP_DIR}" ]]; then
        mkdir -p "${BACKUP_DIR}" || {
            log "ERROR" "Failed to create backup directory: ${BACKUP_DIR}"
            return 1
        }
        log "INFO" "Created backup directory: ${BACKUP_DIR}"
    fi
}

# データベースバックアップの実行
perform_backup() {
    local backup_file="${BACKUP_DIR}/${DB_NAME}_${TIMESTAMP}.sql.gz"

    log "INFO" "Starting backup of database: ${DB_NAME}"

    # pg_dumpの実行
    PGPASSWORD="${DB_PASSWORD:-}" pg_dump \
        -h "${DB_HOST}" \
        -p "${DB_PORT}" \
        -U "${DB_USER}" \
        -F c \
        -b \
        -v \
        -f "${backup_file}.tmp" \
        "${DB_NAME}" 2>&1 | tee -a "${LOG_FILE}" || {
        log "ERROR" "Backup failed"
        rm -f "${backup_file}.tmp"
        return 1
    }

    # 圧縮
    gzip -9 "${backup_file}.tmp"
    mv "${backup_file}.tmp.gz" "${backup_file}"

    # バックアップファイルのサイズ
    local file_size
    file_size="$(du -h "${backup_file}" | cut -f1)"

    log "INFO" "Backup completed: ${backup_file} (${file_size})"

    # チェックサム
    local checksum
    checksum="$(sha256sum "${backup_file}" | cut -d' ' -f1)"
    echo "${checksum}  ${backup_file}" > "${backup_file}.sha256"
    log "INFO" "Checksum: ${checksum}"
}

# 古いバックアップの削除
cleanup_old_backups() {
    log "INFO" "Cleaning up backups older than ${RETENTION_DAYS} days"

    find "${BACKUP_DIR}" \
        -name "${DB_NAME}_*.sql.gz" \
        -type f \
        -mtime "+${RETENTION_DAYS}" \
        -delete \
        -print | while read -r file; do
        log "INFO" "Deleted old backup: ${file}"
    done
}

# バックアップの検証
verify_backup() {
    local backup_file="${1}"

    log "INFO" "Verifying backup: ${backup_file}"

    # チェックサムの検証
    if [[ -f "${backup_file}.sha256" ]]; then
        if sha256sum -c "${backup_file}.sha256" &>/dev/null; then
            log "INFO" "Checksum verification passed"
        else
            log "ERROR" "Checksum verification failed"
            return 1
        fi
    fi

    # ファイルの完全性チェック
    if gzip -t "${backup_file}" &>/dev/null; then
        log "INFO" "File integrity check passed"
    else
        log "ERROR" "File integrity check failed"
        return 1
    fi
}

# メイン処理
main() {
    log "INFO" "=== Database Backup Started ==="

    ensure_backup_dir || exit 1
    perform_backup || exit 1

    local latest_backup="${BACKUP_DIR}/${DB_NAME}_${TIMESTAMP}.sql.gz"
    verify_backup "${latest_backup}" || exit 1

    cleanup_old_backups

    log "INFO" "=== Database Backup Completed ==="
}

main "$@"
```

### システムモニタリングスクリプト

```bash
#!/usr/bin/env bash
# system-monitor.sh - システムリソース監視スクリプト

set -euo pipefail

readonly SCRIPT_NAME="$(basename "${BASH_SOURCE[0]}")"
readonly CHECK_INTERVAL="${CHECK_INTERVAL:-60}"

# しきい値
readonly CPU_THRESHOLD="${CPU_THRESHOLD:-80}"
readonly MEMORY_THRESHOLD="${MEMORY_THRESHOLD:-85}"
readonly DISK_THRESHOLD="${DISK_THRESHOLD:-90}"

# アラート設定
readonly ALERT_EMAIL="${ALERT_EMAIL:-}"
readonly SLACK_WEBHOOK="${SLACK_WEBHOOK:-}"

# CPU使用率の取得
get_cpu_usage() {
    if [[ "$(uname)" == "Darwin" ]]; then
        # macOS
        top -l 2 -n 0 -F -R | grep "CPU usage" | tail -1 | awk '{print int($3)}'
    else
        # Linux
        top -bn2 | grep "Cpu(s)" | tail -1 | awk '{print int($2)}'
    fi
}

# メモリ使用率の取得
get_memory_usage() {
    if [[ "$(uname)" == "Darwin" ]]; then
        # macOS
        vm_stat | perl -ne '/page size of (\d+)/ and $size=$1; /Pages\s+([^:]+)[^\d]+(\d+)/ and printf("%.2f\n", $2 * $size / 1073741824);' | awk '{s+=$1} END {print int(s)}'
    else
        # Linux
        free | grep Mem | awk '{print int($3/$2 * 100)}'
    fi
}

# ディスク使用率の取得
get_disk_usage() {
    local mount_point="${1:-/}"
    df -h "${mount_point}" | tail -1 | awk '{print int($5)}'
}

# アラート送信
send_alert() {
    local subject="${1}"
    local message="${2}"
    local severity="${3:-warning}"

    echo "[ALERT] ${subject}: ${message}"

    # Slackへの通知
    if [[ -n "${SLACK_WEBHOOK}" ]]; then
        local color="warning"
        [[ "${severity}" == "critical" ]] && color="danger"

        curl -X POST "${SLACK_WEBHOOK}" \
            -H 'Content-Type: application/json' \
            -d "{
                \"text\": \"${subject}\",
                \"attachments\": [{
                    \"color\": \"${color}\",
                    \"text\": \"${message}\",
                    \"footer\": \"$(hostname)\",
                    \"ts\": $(date +%s)
                }]
            }" &>/dev/null
    fi

    # メール通知
    if [[ -n "${ALERT_EMAIL}" ]] && command -v mail &>/dev/null; then
        echo "${message}" | mail -s "[${severity}] ${subject}" "${ALERT_EMAIL}"
    fi
}

# リソースチェック
check_resources() {
    local cpu_usage memory_usage disk_usage
    local alerts=()

    cpu_usage="$(get_cpu_usage)"
    memory_usage="$(get_memory_usage)"
    disk_usage="$(get_disk_usage)"

    echo "$(date +'%Y-%m-%d %H:%M:%S') - CPU: ${cpu_usage}%, Memory: ${memory_usage}%, Disk: ${disk_usage}%"

    # CPU使用率チェック
    if ((cpu_usage >= CPU_THRESHOLD)); then
        alerts+=("CPU usage is high: ${cpu_usage}%")
    fi

    # メモリ使用率チェック
    if ((memory_usage >= MEMORY_THRESHOLD)); then
        alerts+=("Memory usage is high: ${memory_usage}%")
    fi

    # ディスク使用率チェック
    if ((disk_usage >= DISK_THRESHOLD)); then
        alerts+=("Disk usage is high: ${disk_usage}%")
    fi

    # アラート送信
    if ((${#alerts[@]} > 0)); then
        local severity="warning"
        ((cpu_usage >= 95 || memory_usage >= 95 || disk_usage >= 95)) && severity="critical"

        local message
        printf -v message '%s\n' "${alerts[@]}"
        send_alert "System Resource Alert" "${message}" "${severity}"
    fi
}

# メインループ
main() {
    echo "Starting system monitoring (interval: ${CHECK_INTERVAL}s)"
    echo "Thresholds - CPU: ${CPU_THRESHOLD}%, Memory: ${MEMORY_THRESHOLD}%, Disk: ${DISK_THRESHOLD}%"

    while true; do
        check_resources
        sleep "${CHECK_INTERVAL}"
    done
}

main "$@"
```

---

## まとめ

このガイドでは、Shell Scriptingの基礎から実践的なテクニックまでを網羅しました。

### 重要なポイント

1. **厳格なエラーハンドリング**
   - `set -euo pipefail` の使用
   - trap によるクリーンアップ
   - 適切なエラーメッセージ

2. **堅牢性**
   - 入力検証
   - リトライロジック
   - リソース管理

3. **保守性**
   - 関数のカプセル化
   - ドキュメンテーション
   - テスト可能な設計

4. **移植性**
   - ShellCheck による検証
   - プラットフォーム差異の考慮
   - 依存関係の明示

5. **セキュリティ**
   - 安全な一時ファイル管理
   - 適切なパーミッション設定
   - 機密情報の保護

### ベストプラクティス

- スクリプトは常に `set -euo pipefail` で開始
- 変数は可能な限り `readonly` と `local` を使用
- 関数は単一責任の原則に従う
- エラーハンドリングは包括的に
- ログは構造化して記録
- テストを書いて品質を保証

このガイドを参考に、堅牢で保守性の高いシェルスクリプトを作成してください。
