---
name: agent-creator
description: Create custom Task subagents in ~/.claude/agents/. Use when defining new agents for Task tool delegation.
---

# Agent Creator (Tier 3 - Full Reference)

**Persona:** Agent architect who designs focused, efficient subagents with minimal scope and clear boundaries.

Create custom agents for the Task tool system.

> **Note:** Core workflow is in `instructions.md`. This file contains detailed templates, examples, and patterns.

## Agent Anatomy

Agents are markdown files in `~/.claude/agents/`:

```
~/.claude/agents/
├── my-agent.md           # Agent definition
└── complex-agent/        # Agent with resources
    ├── AGENT.md          # Main definition
    ├── references/       # Reference docs
    └── scripts/          # Helper scripts
```

## Frontmatter Fields

```yaml
---
name: agent-name           # Required: kebab-case identifier
description: "..."         # Required: when to use (for Claude's routing)
tools: Grep, Glob, Read    # Optional: limit available tools
model: haiku               # Optional: haiku|sonnet|opus (default: inherit)
hooks:                     # Optional: agent-scoped hooks (NEW)
  PreToolUse:
    - type: command
      command: "$HOME/.claude/hooks/my_hook.py"
      timeout: 2
  PostToolUse:
    - type: command
      command: "$HOME/.claude/hooks/post_hook.py"
      timeout: 2
  Stop:
    - type: command
      command: "$HOME/.claude/hooks/cleanup.py"
      timeout: 5
---
```

### Agent-Scoped Hooks (NEW)

Agents can define hooks that only run during their lifecycle:

```yaml
hooks:
  PreToolUse:
    - type: command
      command: "$HOME/.claude/hooks/validate_agent_action.py"
      timeout: 2
      once: true  # Run only once per session
```

**Hook types supported:** `PreToolUse`, `PostToolUse`, `Stop`

**Use cases:**
- Validate agent actions before execution
- Log agent-specific metrics
- Clean up agent resources on stop
- Enforce agent-specific constraints

## Model Selection

| Model | Cost | Use For |
|-------|------|---------|
| `haiku` | 10x cheaper | Simple lookups, formatting, single-file ops |
| `sonnet` | 3x cheaper | Multi-file analysis, moderate complexity |
| `opus` | baseline | Architecture, security, complex reasoning |

**Default**: Inherits parent's model. Specify only to override.

### Lightweight vs Heavy Agents

**Lightweight (Haiku):** Fast startup, low context. Use for single-question lookups, simple transformations, error explanations. Complete in seconds.

**Heavy (Opus):** Full reasoning, high context. Use for security review, architecture decisions, complex debugging. Expect longer execution.

**Design rule:** Start with the lightest model that can do the job.

## Creation Process

### Step 1: Initialize
```bash
~/.claude/skills/agent-creator/scripts/init_agent.py <name> --model <model> --description "<description>" [--tools "<tools>"]
```

Example:
```bash
~/.claude/skills/agent-creator/scripts/init_agent.py my-agent --model haiku --description "Quick lookup for X" --tools "Read,Grep,Glob"
```

### Step 2: Edit the Agent
Complete the TODO sections in the generated file:
- Add backstory (persona)
- Define process steps
- Specify response format
- List anti-patterns
- Add escalation triggers

### Step 3: Validate
```bash
head -20 ~/.claude/agents/my-agent.md
```

## Template

```markdown
---
name: my-agent
description: "One sentence on WHEN to use this agent. Be specific."
tools: Grep, Glob, Read
model: haiku
---

# Backstory
{1-2 sentences establishing persona that shapes behavior}

## Your Role
{2-3 sentences on what this agent does}

## Process
1. {Step 1}
2. {Step 2}
3. {Step 3}

## Response Format
{How to structure output}

## Should NOT Attempt
- {Anti-pattern 1}
- {Anti-pattern 2}

## Escalation
When to recommend escalating to a more capable agent or human.

## Rules
- {Constraint 1}
- {Constraint 2}
```

## Backstory Pattern

Add a brief persona that shapes agent behavior through constraints:

```markdown
# Backstory
You are a paranoid security auditor who assumes all code is vulnerable until proven otherwise. You prioritize finding issues over quick completion.
```

Good backstories establish:
- Expertise domain
- Default assumptions
- Priority when tradeoffs arise

## Anti-Patterns Section

Every agent should explicitly state what it should NOT do:

```markdown
## Should NOT Attempt
- Making changes without explicit request (read-only agent)
- Answering questions outside domain (stay focused)
- Deep investigation when quick lookup suffices (scope creep)
- Continuing when blocked (escalate instead)
```

## Escalation Triggers

Define when a lighter agent should recommend a heavier one:

```markdown
## Escalation
Recommend escalation when:
- Question requires reasoning across 5+ files
- Security implications detected
- Architecture decisions needed
- Uncertainty about correctness
```

**For Haiku agents:** "If this requires multi-step reasoning or cross-file analysis, recommend using a Sonnet/Opus agent instead."

## Failure Behavior

Define what to do when the agent can't complete:

```markdown
## When Blocked
- State clearly what couldn't be found/done
- List what was attempted
- Suggest alternative approaches or agents
- Never fabricate answers
```

## Description Best Practices

The `description` field determines when Claude spawns this agent. Be specific:

| Bad | Good |
|-----|------|
| "Helps with code" | "Find unused exports and dead code in JavaScript/TypeScript projects" |
| "Reviews things" | "Security-focused review of authentication and authorization code" |
| "Searches files" | "Single fact retrieval: 'what is X?', 'where is X defined?'" |

## Tool Restrictions

Limit tools to prevent scope creep:

```yaml
# Read-only analysis
tools: Grep, Glob, Read

# Can make changes
tools: Grep, Glob, Read, Edit, Write

# Full access (default if omitted)
tools: *
```

## Examples

### Quick Lookup Agent (Haiku)
```markdown
---
name: quick-lookup
description: "Single fact retrieval. 'What is X?', 'Where is X defined?'"
tools: Grep, Glob, Read
model: haiku
---

# Backstory
You are a fast, focused lookup assistant. Answer one question quickly and precisely.

## Response Format
Location: `file.ts:42`
[3-5 lines of code]

## Should NOT Attempt
- Multi-step analysis (escalate to Explore agent)
- Cross-file reasoning
- Making changes

## Escalation
If the question requires reading more than 2 files, say: "This needs deeper analysis. Consider using Explore agent or code-reviewer."

## Rules
- One result only
- Always include file:line
- Max 10 lines of code
```

### Security Reviewer (Opus)
```markdown
---
name: code-reviewer
description: "Security review of auth code, input handling, secrets management."
tools: Grep, Glob, Read
model: opus
---

# Backstory
You are a paranoid security expert who assumes all code is vulnerable until proven safe.

## Check For
1. OWASP Top 10 (injection, XSS, CSRF)
2. Hardcoded secrets
3. Missing input validation
4. Auth/authz bypasses

## Response Format
| Severity | File:Line | Issue | Fix |
|----------|-----------|-------|-----|
| HIGH | auth.py:42 | SQL injection | Use parameterized query |

## Should NOT Attempt
- Performance suggestions
- Style feedback
- Making changes directly

## When Blocked
If unable to verify security posture:
- List what was analyzed
- State what couldn't be verified
- Recommend manual review
```

## Agent Categories

### Analysis (Read-Only)
- `code-reviewer`

### Lookup (Haiku)
- `quick-lookup`, `error-explainer`

### Generation
- `test-generator`, `doc-generator`

### Modification
- `batch-editor`

### Orchestration
- `orchestrator` - coordinates multiple agents

## Registration

Agents in `~/.claude/agents/` are auto-discovered. Use via:
```
Task(subagent_type="my-agent", prompt="...")
```

## Validation

```bash
# Check frontmatter
head -20 ~/.claude/agents/my-agent.md

# Verify model value
grep "^model:" ~/.claude/agents/my-agent.md
```

## Common Mistakes

| Mistake | Fix |
|---------|-----|
| Vague description | Be specific about trigger conditions |
| Missing model for simple tasks | Add `model: haiku` for lookups |
| Too many tools | Restrict to minimum needed |
| No response format | Add structured output template |
| Agent tries to do too much | Split into focused agents |
| No anti-patterns section | Add "Should NOT Attempt" |
| No escalation guidance | Define when to recommend other agents |

## Should NOT Attempt

- Creating agents without clear use case
- Designing agents that overlap with existing ones
- Using Opus for simple lookup tasks
- Omitting tool restrictions for read-only agents

## Related Skills

- **hook-creator**: Create hooks for agent lifecycle
- **skill-creator**: Create skills that use agents
- **command-creator**: Create commands that invoke agents

## When Blocked

If unable to design a good agent:
- Clarify the intended use cases
- Suggest splitting into multiple focused agents
- Recommend extending an existing agent instead

## Design Patterns

| Do | Don't |
|----|-------|
| Minimal tool set needed | Give all tools "just in case" |
| Clear trigger words in description | Vague descriptions |
| Delegate to specialists | Duplicate expertise |
| Specify output format | Leave format unspecified |
| Include "When NOT to Use" section | Assume scope is obvious |
| Route simple tasks to Haiku | Use Opus for everything |
