#!/usr/bin/env python3
"""Skill discovery script for the skill-creator skill.

This script scans skill directories to find and index existing skills,
enabling the triage phase (Phase 0) of skill creation.

Usage:
    python discover_skills.py [options]

Options:
    --path PATH     Skills directory to scan (default: current project)
    --format FORMAT Output format: text, json, or markdown (default: text)
    --search QUERY  Search for skills matching query in name or description

Example:
    python discover_skills.py --search "research"
    python discover_skills.py --format json
    python discover_skills.py --path ~/.deepagents/skills/

Exit Codes:
    0: Success
    1: No skills found or error
"""

import argparse
import json
import sys
from pathlib import Path


def parse_frontmatter(content: str) -> dict[str, str] | None:
    """Parse YAML frontmatter from SKILL.md content.

    Args:
        content: Full file content.

    Returns:
        Dictionary of frontmatter fields, or None if no valid frontmatter.
    """
    if not content.startswith("---"):
        return None

    lines = content.split("\n")
    end_idx = -1
    for i, line in enumerate(lines[1:], start=1):
        if line.strip() == "---":
            end_idx = i
            break

    if end_idx == -1:
        return None

    frontmatter: dict[str, str] = {}
    for line in lines[1:end_idx]:
        line = line.strip()
        if not line or line.startswith("#"):
            continue
        if ":" in line:
            key, _, value = line.partition(":")
            frontmatter[key.strip()] = value.strip()

    return frontmatter


def discover_skills(skills_dir: Path) -> list[dict]:
    """Discover all skills in a directory.

    Args:
        skills_dir: Path to skills directory.

    Returns:
        List of skill metadata dictionaries.
    """
    skills = []

    if not skills_dir.exists():
        return skills

    for skill_path in skills_dir.iterdir():
        if not skill_path.is_dir():
            continue

        skill_md = skill_path / "SKILL.md"
        if not skill_md.exists():
            continue

        try:
            content = skill_md.read_text(encoding="utf-8")
        except Exception:
            continue

        frontmatter = parse_frontmatter(content)
        if frontmatter is None:
            continue

        # Extract metadata
        skill_info = {
            "name": frontmatter.get("name", skill_path.name),
            "description": frontmatter.get("description", "(no description)"),
            "path": str(skill_path),
            "has_scripts": (skill_path / "scripts").exists(),
            "has_templates": (skill_path / "templates").exists(),
        }

        # Check for optional fields
        if "license" in frontmatter:
            skill_info["license"] = frontmatter["license"]
        if "allowed-tools" in frontmatter:
            skill_info["allowed_tools"] = frontmatter["allowed-tools"]

        skills.append(skill_info)

    return skills


def search_skills(skills: list[dict], query: str) -> list[dict]:
    """Search skills by name or description.

    Args:
        skills: List of skill metadata dictionaries.
        query: Search query string.

    Returns:
        Filtered list of matching skills.
    """
    query_lower = query.lower()
    matches = []

    for skill in skills:
        name = skill.get("name", "").lower()
        description = skill.get("description", "").lower()

        if query_lower in name or query_lower in description:
            matches.append(skill)

    return matches


def format_text(skills: list[dict]) -> str:
    """Format skills as plain text."""
    if not skills:
        return "No skills found."

    lines = []
    lines.append(f"Found {len(skills)} skill(s):\n")
    lines.append("=" * 60)

    for skill in skills:
        lines.append(f"\n📦 {skill['name']}")
        lines.append(f"   Description: {skill['description'][:80]}...")
        lines.append(f"   Path: {skill['path']}")

        features = []
        if skill.get("has_scripts"):
            features.append("scripts")
        if skill.get("has_templates"):
            features.append("templates")
        if features:
            lines.append(f"   Features: {', '.join(features)}")

    lines.append("\n" + "=" * 60)
    return "\n".join(lines)


def format_markdown(skills: list[dict]) -> str:
    """Format skills as markdown table."""
    if not skills:
        return "No skills found."

    lines = []
    lines.append(f"# Discovered Skills ({len(skills)})\n")
    lines.append("| Name | Description | Features |")
    lines.append("|------|-------------|----------|")

    for skill in skills:
        features = []
        if skill.get("has_scripts"):
            features.append("📜 scripts")
        if skill.get("has_templates"):
            features.append("📄 templates")

        desc = skill["description"][:50] + "..." if len(skill["description"]) > 50 else skill["description"]
        lines.append(f"| **{skill['name']}** | {desc} | {', '.join(features) or '-'} |")

    return "\n".join(lines)


def format_json(skills: list[dict]) -> str:
    """Format skills as JSON."""
    return json.dumps({"skills": skills, "count": len(skills)}, indent=2)


def get_default_skills_paths() -> list[Path]:
    """Get default skills directories to scan."""
    paths = []

    # Project-level skills (PROJECT_ROOT/skills/ - preferred location)
    cwd = Path.cwd()
    root_skills = cwd / "skills"
    if root_skills.exists():
        paths.append(root_skills)

    # Legacy .deepagents/skills/ pattern (for backward compatibility)
    deepagents_skills = cwd / ".deepagents" / "skills"
    if deepagents_skills.exists():
        paths.append(deepagents_skills)

    # User-level skills
    user_skills = Path.home() / ".deepagents" / "skills"
    if user_skills.exists():
        paths.append(user_skills)

    return paths


def main() -> int:
    """Run the main entry point."""
    parser = argparse.ArgumentParser(
        description="Discover and index existing skills for triage."
    )
    parser.add_argument(
        "--path",
        type=str,
        help="Skills directory to scan (default: auto-detect)",
    )
    parser.add_argument(
        "--format",
        choices=["text", "json", "markdown"],
        default="text",
        help="Output format (default: text)",
    )
    parser.add_argument(
        "--search",
        type=str,
        help="Search for skills matching query",
    )

    args = parser.parse_args()

    # Determine paths to scan
    if args.path:
        paths = [Path(args.path).expanduser()]
    else:
        paths = get_default_skills_paths()

    if not paths:
        print("No skills directories found. Specify --path or create skills in .deepagents/skills/")
        return 1

    # Discover all skills
    all_skills = []
    for path in paths:
        skills = discover_skills(path)
        all_skills.extend(skills)

    # Apply search filter if specified
    if args.search:
        all_skills = search_skills(all_skills, args.search)
        if not all_skills:
            print(f"No skills matching '{args.search}' found.")
            return 1

    # Format output
    if args.format == "json":
        output = format_json(all_skills)
    elif args.format == "markdown":
        output = format_markdown(all_skills)
    else:
        output = format_text(all_skills)

    print(output)
    return 0


if __name__ == "__main__":
    sys.exit(main())
