#!/usr/bin/env python3
"""Skill validation script for the skill-creator skill.

This script validates a skill's structure and frontmatter according to
the Agent Skills specification (agentskills.io).

Usage:
    python validate_skill.py <skill_path>
    python validate_skill.py ~/.deepagents/skills/my-skill/

Validation Rules:
    - SKILL.md must exist
    - YAML frontmatter must be valid
    - name: lowercase, hyphen-case, ≤64 chars
    - description: ≤1024 chars, no angle brackets
    - Required sections: "When to Use", "How to Use"

Exit Codes:
    0: Validation passed
    1: Validation failed
    10: File not found or read error
"""

import re
import sys
from pathlib import Path


class ValidationResult:
    """Container for validation results."""

    def __init__(self) -> None:
        self.passed: list[str] = []
        self.failed: list[str] = []
        self.warnings: list[str] = []

    @property
    def is_valid(self) -> bool:
        """Return True if no failures."""
        return len(self.failed) == 0

    def add_pass(self, message: str) -> None:
        """Add a passing check."""
        self.passed.append(message)

    def add_fail(self, message: str) -> None:
        """Add a failing check."""
        self.failed.append(message)

    def add_warning(self, message: str) -> None:
        """Add a warning (non-blocking)."""
        self.warnings.append(message)

    def print_report(self) -> None:
        """Print formatted validation report."""
        print("\n" + "=" * 60)
        print("SKILL VALIDATION REPORT")
        print("=" * 60)

        if self.passed:
            print("\n✅ PASSED:")
            for msg in self.passed:
                print(f"   • {msg}")

        if self.warnings:
            print("\n⚠️  WARNINGS:")
            for msg in self.warnings:
                print(f"   • {msg}")

        if self.failed:
            print("\n❌ FAILED:")
            for msg in self.failed:
                print(f"   • {msg}")

        print("\n" + "-" * 60)
        if self.is_valid:
            print("✅ VALIDATION PASSED")
        else:
            print(f"❌ VALIDATION FAILED ({len(self.failed)} error(s))")
        print("-" * 60 + "\n")


def parse_frontmatter(content: str) -> tuple[dict[str, str] | None, str]:
    """Parse YAML frontmatter from SKILL.md content.

    Args:
        content: Full file content.

    Returns:
        Tuple of (frontmatter_dict, body_content).
        frontmatter_dict is None if no valid frontmatter found.
    """
    if not content.startswith("---"):
        return None, content

    # Find the closing ---
    lines = content.split("\n")
    end_idx = -1
    for i, line in enumerate(lines[1:], start=1):
        if line.strip() == "---":
            end_idx = i
            break

    if end_idx == -1:
        return None, content

    # Parse YAML manually (simple key: value pairs)
    frontmatter: dict[str, str] = {}
    for line in lines[1:end_idx]:
        line = line.strip()
        if not line or line.startswith("#"):
            continue
        if ":" in line:
            key, _, value = line.partition(":")
            frontmatter[key.strip()] = value.strip()

    body = "\n".join(lines[end_idx + 1 :])
    return frontmatter, body


def validate_name(name: str | None, result: ValidationResult) -> None:
    """Validate skill name according to spec."""
    if not name:
        result.add_fail("Missing required field: 'name'")
        return

    # Check length
    if len(name) > 64:
        result.add_fail(f"name '{name}' exceeds 64 character limit ({len(name)} chars)")
    else:
        result.add_pass(f"name length OK ({len(name)}/64 chars)")

    # Check format: lowercase alphanumeric + hyphens
    if not re.match(r"^[a-z0-9][a-z0-9-]*[a-z0-9]$|^[a-z0-9]$", name):
        result.add_fail(
            f"name '{name}' must be lowercase, hyphen-case "
            "(start/end with alphanumeric, only a-z, 0-9, and - allowed)"
        )
    else:
        result.add_pass("name format OK (hyphen-case)")


def validate_description(description: str | None, result: ValidationResult) -> None:
    """Validate skill description according to spec."""
    if not description:
        result.add_fail("Missing required field: 'description'")
        return

    # Check length
    if len(description) > 1024:
        result.add_fail(
            f"description exceeds 1024 character limit ({len(description)} chars)"
        )
    else:
        result.add_pass(f"description length OK ({len(description)}/1024 chars)")

    # Check for angle brackets (not allowed)
    if "<" in description or ">" in description:
        result.add_fail("description contains angle brackets (< or >) which are not allowed")
    else:
        result.add_pass("description format OK (no angle brackets)")


def validate_required_sections(body: str, result: ValidationResult) -> None:
    """Validate that required sections exist."""
    required_sections = [
        ("When to Use", r"#+\s*When\s+to\s+Use"),
        ("How to Use", r"#+\s*How\s+to\s+Use"),
    ]

    for section_name, pattern in required_sections:
        if re.search(pattern, body, re.IGNORECASE):
            result.add_pass(f"Required section '{section_name}' found")
        else:
            result.add_warning(
                f"Recommended section '{section_name}' not found "
                "(consider adding for completeness)"
            )


def validate_optional_fields(frontmatter: dict[str, str], result: ValidationResult) -> None:
    """Check optional frontmatter fields."""
    valid_fields = {"name", "description", "license", "allowed-tools", "metadata"}
    extra_fields = set(frontmatter.keys()) - valid_fields

    if extra_fields:
        result.add_warning(
            f"Non-standard frontmatter fields: {', '.join(extra_fields)} "
            "(these may be ignored by some systems)"
        )


def validate_skill(skill_path: Path) -> ValidationResult:
    """Validate a skill at the given path.

    Args:
        skill_path: Path to the skill directory.

    Returns:
        ValidationResult with all checks performed.
    """
    result = ValidationResult()

    # Check SKILL.md exists
    skill_md_path = skill_path / "SKILL.md"
    if not skill_md_path.exists():
        result.add_fail(f"SKILL.md not found at {skill_md_path}")
        return result
    result.add_pass("SKILL.md exists")

    # Read content
    try:
        content = skill_md_path.read_text(encoding="utf-8")
    except Exception as e:
        result.add_fail(f"Failed to read SKILL.md: {e}")
        return result
    result.add_pass("SKILL.md is readable")

    # Parse frontmatter
    frontmatter, body = parse_frontmatter(content)
    if frontmatter is None:
        result.add_fail("No valid YAML frontmatter found (must start with ---)")
        return result
    result.add_pass("YAML frontmatter found")

    # Validate individual fields
    validate_name(frontmatter.get("name"), result)
    validate_description(frontmatter.get("description"), result)
    validate_optional_fields(frontmatter, result)
    validate_required_sections(body, result)

    return result


def main() -> int:
    """Run the main entry point."""
    if len(sys.argv) < 2:
        print("Usage: python validate_skill.py <skill_path>")
        print("Example: python validate_skill.py ~/.deepagents/skills/my-skill/")
        return 1

    skill_path = Path(sys.argv[1]).expanduser()

    if not skill_path.exists():
        print(f"Error: Path does not exist: {skill_path}")
        return 10

    if not skill_path.is_dir():
        # Maybe they passed the SKILL.md directly
        if skill_path.name == "SKILL.md":
            skill_path = skill_path.parent
        else:
            print(f"Error: Path is not a directory: {skill_path}")
            return 10

    result = validate_skill(skill_path)
    result.print_report()

    return 0 if result.is_valid else 1


if __name__ == "__main__":
    sys.exit(main())
