# Agentforce Agent Deployment Guide

> Complete DevOps guide for deploying Agentforce agents using SF CLI

## Overview

This guide covers the complete deployment lifecycle for Agentforce agents, including:
- Agent metadata types and pseudo metadata
- Sync operations (retrieve/deploy)
- Lifecycle management (activate/deactivate)
- Full deployment workflows

**Related Skills:**
- `sf-ai-agentforce` - Agent Script authoring and publishing
- `sf-deploy` - This skill - deployment orchestration

---

## Agent Metadata Types

Agentforce agents consist of multiple metadata components:

| Metadata Type | Description | Example API Name |
|---------------|-------------|------------------|
| `Bot` | Top-level chatbot definition | `Customer_Support_Agent` |
| `BotVersion` | Version configuration | `Customer_Support_Agent.v1` |
| `GenAiPlannerBundle` | Reasoning engine (LLM config) | `Customer_Support_Agent_Planner` |
| `GenAiPlugin` | Topic definition | `Order_Management_Plugin` |
| `GenAiFunction` | Action definition | `Get_Order_Status_Function` |

### Metadata Hierarchy

```
Bot (Agent Definition)
└── BotVersion (Version Config)
    └── GenAiPlannerBundle (Reasoning Engine)
        ├── GenAiPlugin (Topic 1)
        │   ├── GenAiFunction (Action 1)
        │   └── GenAiFunction (Action 2)
        └── GenAiPlugin (Topic 2)
            └── GenAiFunction (Action 3)
```

---

## Agent Pseudo Metadata Type

The `Agent` pseudo metadata type is a convenience wrapper that retrieves or deploys all agent-related components at once.

### Using the Agent Pseudo Type

```bash
# Retrieve agent + all dependencies from org
sf project retrieve start --metadata Agent:[AgentName] --target-org [alias]

# Deploy agent metadata to org
sf project deploy start --metadata Agent:[AgentName] --target-org [alias]
```

### What Gets Synced

When using `--metadata Agent:[Name]`:

**Retrieved/Deployed:**
- `Bot` - Top-level chatbot
- `BotVersion` - Version configuration
- `GenAiPlannerBundle` - Reasoning engine
- `GenAiPlugin` - All topics
- `GenAiFunction` - All actions

**NOT Included:**
- Apex classes (deploy separately)
- Flows (deploy separately)
- Named Credentials (deploy separately)

---

## Sync Operations

### Retrieving Agents from Org

```bash
# Retrieve agent using pseudo metadata type
sf project retrieve start --metadata Agent:Customer_Support_Agent --target-org myorg

# Retrieve to specific output directory
sf project retrieve start --metadata Agent:Customer_Support_Agent --output-dir ./retrieved --target-org myorg

# Retrieve multiple agents
sf project retrieve start --metadata Agent:Support_Agent,Agent:Sales_Agent --target-org myorg
```

### Retrieving Specific Components

```bash
# Retrieve just the bot definition
sf project retrieve start --metadata Bot:Customer_Support_Agent --target-org myorg

# Retrieve just the planner bundle
sf project retrieve start --metadata GenAiPlannerBundle:Customer_Support_Agent_Planner --target-org myorg

# Retrieve all plugins (topics)
sf project retrieve start --metadata GenAiPlugin --target-org myorg

# Retrieve all functions (actions)
sf project retrieve start --metadata GenAiFunction --target-org myorg
```

### Deploying Agents to Org

```bash
# Deploy agent using pseudo metadata type
sf project deploy start --metadata Agent:Customer_Support_Agent --target-org myorg

# Deploy with validation only (dry run)
sf project deploy start --metadata Agent:Customer_Support_Agent --dry-run --target-org myorg

# Deploy multiple agents
sf project deploy start --metadata Agent:Support_Agent,Agent:Sales_Agent --target-org myorg
```

---

## Agent Lifecycle Management

### Activate Agent

Makes an agent available to users.

```bash
sf agent activate --api-name [AgentName] --target-org [alias]
```

**Requirements:**
- Agent must be published first (`sf agent publish`)
- All Apex classes and Flows must be deployed
- `default_agent_user` must be a valid org user with Agentforce permissions

**Post-Activation:**
- Agent is immediately available to users
- Preview command can be used for testing
- Changes require deactivation first

### Deactivate Agent

Deactivates an agent for modifications. **Required before making changes.**

```bash
sf agent deactivate --api-name [AgentName] --target-org [alias]
```

**When Deactivation is Required:**
- Adding or removing topics
- Modifying action configurations
- Changing system instructions
- Updating variable definitions

### Modification Workflow

```bash
# 1. Deactivate agent
sf agent deactivate --api-name Customer_Support_Agent --target-org myorg

# 2. Make changes to Agent Script

# 3. Re-publish
sf agent publish --api-name Customer_Support_Agent --target-org myorg

# 4. Re-activate
sf agent activate --api-name Customer_Support_Agent --target-org myorg
```

---

## Agent Preview

Preview allows testing agent behavior before production deployment.

### Preview Modes

| Mode | Command | Use When |
|------|---------|----------|
| **Simulated** | `sf agent preview --api-name X` | Testing logic, Apex/Flows not ready |
| **Live** | `sf agent preview --api-name X --use-live-actions --client-app Y` | Integration testing with real data |

### Simulated Mode (Default)

```bash
sf agent preview --api-name Customer_Support_Agent --target-org myorg
```

- LLM simulates action responses
- No actual Apex/Flow execution
- Safe for testing - no data changes

### Live Mode

```bash
sf agent preview --api-name Customer_Support_Agent --use-live-actions --client-app MyAgentApp --target-org myorg
```

**Requirements:**
- Connected app configured
- Apex classes and Flows deployed
- Agent must be active

### Preview with Debug Output

```bash
sf agent preview --api-name Customer_Support_Agent --output-dir ./preview-logs --apex-debug --target-org myorg
```

**See `sf-ai-agentforce/docs/agent-preview-guide.md`** for connected app setup.

---

## Full Deployment Workflows

### New Agent Deployment

Complete workflow for deploying a new agent:

```bash
# 1. Deploy Apex classes (if any)
sf project deploy start --metadata ApexClass --target-org myorg

# 2. Deploy Flows
sf project deploy start --metadata Flow --target-org myorg

# 3. Validate Agent Script
sf afdx agent validate --api-name Customer_Support_Agent --target-org myorg

# 4. Publish agent
sf agent publish --api-name Customer_Support_Agent --target-org myorg

# 5. Preview (simulated mode)
sf agent preview --api-name Customer_Support_Agent --target-org myorg

# 6. Activate
sf agent activate --api-name Customer_Support_Agent --target-org myorg

# 7. Preview (live mode - optional)
sf agent preview --api-name Customer_Support_Agent --use-live-actions --client-app MyApp --target-org myorg
```

### Update Existing Agent

```bash
# 1. Deactivate
sf agent deactivate --api-name Customer_Support_Agent --target-org myorg

# 2. Deploy updated dependencies (if any)
sf project deploy start --metadata ApexClass,Flow --target-org myorg

# 3. Validate
sf afdx agent validate --api-name Customer_Support_Agent --target-org myorg

# 4. Re-publish
sf agent publish --api-name Customer_Support_Agent --target-org myorg

# 5. Preview
sf agent preview --api-name Customer_Support_Agent --target-org myorg

# 6. Re-activate
sf agent activate --api-name Customer_Support_Agent --target-org myorg
```

### Sync Between Orgs

```bash
# 1. Retrieve from source org
sf project retrieve start --metadata Agent:Customer_Support_Agent --target-org source-org

# 2. Deploy dependencies to target org first
sf project deploy start --source-dir force-app/main/default/classes --target-org target-org
sf project deploy start --source-dir force-app/main/default/flows --target-org target-org

# 3. Deploy agent metadata
sf project deploy start --metadata Agent:Customer_Support_Agent --target-org target-org

# 4. Publish agent in target org
sf agent publish --api-name Customer_Support_Agent --target-org target-org

# 5. Activate in target org
sf agent activate --api-name Customer_Support_Agent --target-org target-org
```

### CI/CD Pipeline Integration

Example deployment script for CI/CD:

```bash
#!/bin/bash
# deploy-agent.sh

set -e  # Exit on error

ORG_ALIAS=$1
AGENT_NAME=$2

echo "🚀 Deploying agent: $AGENT_NAME to $ORG_ALIAS"

# Step 1: Deploy dependencies
echo "📦 Deploying Apex classes..."
sf project deploy start --metadata ApexClass --target-org $ORG_ALIAS --wait 10

echo "📦 Deploying Flows..."
sf project deploy start --metadata Flow --target-org $ORG_ALIAS --wait 10

# Step 2: Validate agent script
echo "✅ Validating Agent Script..."
sf afdx agent validate --api-name $AGENT_NAME --target-org $ORG_ALIAS

# Step 3: Check if agent exists (deactivate if needed)
echo "🔍 Checking agent status..."
if sf agent deactivate --api-name $AGENT_NAME --target-org $ORG_ALIAS 2>/dev/null; then
    echo "⏸️ Agent deactivated for update"
fi

# Step 4: Publish agent
echo "📤 Publishing agent..."
sf agent publish --api-name $AGENT_NAME --target-org $ORG_ALIAS --wait 10

# Step 5: Activate agent
echo "▶️ Activating agent..."
sf agent activate --api-name $AGENT_NAME --target-org $ORG_ALIAS

echo "✅ Agent deployment complete: $AGENT_NAME"
```

Usage:
```bash
./deploy-agent.sh myorg Customer_Support_Agent
```

---

## Dependency Deployment Order

**Critical:** Dependencies must be deployed BEFORE the agent.

```
1. Custom Objects/Fields (sf-metadata)
   ↓
2. Apex Classes (sf-apex)
   ↓
3. Flows (sf-flow)
   ↓
4. Named Credentials (sf-integration, if external APIs)
   ↓
5. Agent Metadata (sf-ai-agentforce publish)
   ↓
6. Agent Activation
```

### Deployment Commands by Order

```bash
# 1. Objects/Fields
sf project deploy start --metadata CustomObject,CustomField --target-org myorg

# 2. Apex
sf project deploy start --metadata ApexClass --target-org myorg

# 3. Flows
sf project deploy start --metadata Flow --target-org myorg

# 4. Named Credentials (if needed)
sf project deploy start --metadata NamedCredential --target-org myorg

# 5. Publish agent
sf agent publish --api-name My_Agent --target-org myorg

# 6. Activate
sf agent activate --api-name My_Agent --target-org myorg
```

---

## Troubleshooting

### "Internal Error, try again later"

**Causes:**
- Invalid `default_agent_user`
- Dependencies not deployed
- Flow/action variable name mismatch

**Solutions:**
```bash
# Verify user exists
sf data query --query "SELECT Id, Username FROM User WHERE Username = 'agent@example.com'" --target-org myorg

# Deploy dependencies first
sf project deploy start --metadata ApexClass,Flow --target-org myorg
```

### "No active agents found"

**Cause:** Agent not activated

**Solution:**
```bash
sf agent activate --api-name My_Agent --target-org myorg
```

### "Agent must be deactivated before changes"

**Cause:** Trying to modify active agent

**Solution:**
```bash
sf agent deactivate --api-name My_Agent --target-org myorg
# Make changes
sf agent publish --api-name My_Agent --target-org myorg
sf agent activate --api-name My_Agent --target-org myorg
```

### Deployment Fails with Missing Dependencies

**Cause:** Apex/Flows not deployed before agent

**Solution:** Follow the dependency deployment order above.

---

## Cross-Skill Integration

| From Skill | To Skill | Purpose |
|------------|----------|---------|
| sf-ai-agentforce | sf-deploy | Publish and activate agents |
| sf-apex | sf-deploy | Deploy Apex before agent |
| sf-flow | sf-deploy | Deploy Flows before agent |
| sf-integration | sf-deploy | Deploy Named Credentials for external APIs |

### Integration Pattern

```bash
# 1. sf-apex creates InvocableMethod class
# 2. sf-flow creates wrapper Flow
# 3. sf-ai-agentforce creates agent with flow:// action
# 4. sf-deploy orchestrates deployment in correct order
```

---

## Command Reference

### Agent-Specific Commands

| Command | Description |
|---------|-------------|
| `sf agent publish --api-name X` | Publish authoring bundle |
| `sf agent activate --api-name X` | Activate published agent |
| `sf agent deactivate --api-name X` | Deactivate agent for changes |
| `sf agent preview --api-name X` | Preview agent behavior |
| `sf afdx agent validate --api-name X` | Validate Agent Script syntax |

### Metadata Commands with Agent Pseudo Type

| Command | Description |
|---------|-------------|
| `sf project retrieve start --metadata Agent:X` | Retrieve agent + components |
| `sf project deploy start --metadata Agent:X` | Deploy agent metadata |
| `sf project retrieve start --metadata Bot:X` | Retrieve bot definition only |
| `sf project retrieve start --metadata GenAiPlannerBundle:X` | Retrieve planner bundle |
| `sf project retrieve start --metadata GenAiPlugin` | Retrieve all plugins |
| `sf project retrieve start --metadata GenAiFunction` | Retrieve all functions |

### Management Commands

| Command | Description |
|---------|-------------|
| `sf org open agent --api-name X` | Open agent in Agentforce Builder |
| `sf org list metadata --metadata-type Bot` | List bots in org |
| `sf org list metadata --metadata-type GenAiPlannerBundle` | List planner bundles |

---

## Related Documentation

- [Agent CLI Reference](../../sf-ai-agentforce/docs/agent-cli-reference.md)
- [Agent Preview Guide](../../sf-ai-agentforce/docs/agent-preview-guide.md)
- [Agent Script Syntax](../../sf-ai-agentforce/docs/agent-script-syntax.md)
