#!/usr/bin/env python3
# /// script
# requires-python = ">=3.11"
# dependencies = [
#     "pyyaml",
#     "requests",
#     "numpy",
#     "numpy-financial",
# ]
# ///
"""
ABOUTME: Comprehensive rental property analysis tool that calculates financial metrics
ABOUTME: and compares rental vs. selling scenarios with web-based rental comps lookup.

This script performs detailed financial analysis for rental properties including:
- Cash flow calculations
- Key financial metrics (cap rate, cash-on-cash return, IRR, NPV)
- Rent vs. sell comparison
- Break-even analysis
- Rental comps from web sources
"""

import argparse
import sys
from dataclasses import dataclass
from datetime import datetime
from pathlib import Path
from typing import Optional

import numpy as np
import numpy_financial as npf
import requests
import yaml


@dataclass
class PropertyData:
    """Property ownership and financial details"""

    address: str
    purchase_price: float
    purchase_year: int
    current_mortgage_balance: float
    interest_rate: float
    loan_term_years: int
    monthly_principal_interest: Optional[float]
    property_tax_per_year: float
    home_insurance_per_year: float
    hoa_per_month: float
    maintenance_pct: float
    management_fee_pct: float
    vacancy_rate_pct: float

    desired_start_date: str
    estimated_rent: Optional[float]
    include_web_comps: bool

    current_home_value: Optional[float]
    down_payment_original: float
    closing_costs_estimate: float
    expected_appreciation_rate_pct: float
    expected_rent_growth_pct: float

    compare_against_selling: bool
    sale_cost_pct: float
    personal_tax_bracket_pct: float
    hold_period_years: int


class RentalAnalyzer:
    """Analyzes rental property financials and generates comprehensive reports"""

    def __init__(self, property_data: PropertyData):
        self.data = property_data
        self.monthly_rent = property_data.estimated_rent
        self.property_value = property_data.current_home_value or property_data.purchase_price
        self.citations = []  # Track all data sources and citations

    def add_citation(self, source: str, url: str = None, note: str = None):
        """Add a citation/source to the report"""
        citation = {"source": source}
        if url:
            citation["url"] = url
        if note:
            citation["note"] = note
        self.citations.append(citation)

    def calculate_monthly_pmi(self) -> float:
        """Calculate monthly principal and interest payment"""
        if self.data.monthly_principal_interest:
            return self.data.monthly_principal_interest

        # Calculate using mortgage formula
        P = self.data.current_mortgage_balance
        r = (self.data.interest_rate / 100) / 12  # monthly rate
        n = self.data.loan_term_years * 12  # total payments

        if r == 0:
            return P / n

        monthly_payment = P * (r * (1 + r)**n) / ((1 + r)**n - 1)
        return monthly_payment

    def calculate_monthly_expenses(self) -> dict:
        """Calculate all monthly expenses"""
        return {
            "mortgage": self.calculate_monthly_pmi(),
            "property_tax": self.data.property_tax_per_year / 12,
            "insurance": self.data.home_insurance_per_year / 12,
            "hoa": self.data.hoa_per_month,
            "maintenance": (self.property_value * self.data.maintenance_pct / 100) / 12,
            "management": (self.monthly_rent or 0) * self.data.management_fee_pct / 100,
            "vacancy": (self.monthly_rent or 0) * self.data.vacancy_rate_pct / 100,
        }

    def calculate_monthly_cash_flow(self) -> dict:
        """Calculate monthly cash flow"""
        if not self.monthly_rent:
            return {"error": "No rental estimate available"}

        expenses = self.calculate_monthly_expenses()
        total_expenses = sum(expenses.values())
        net_cash_flow = self.monthly_rent - total_expenses

        return {
            "gross_rent": self.monthly_rent,
            "expenses": expenses,
            "total_expenses": total_expenses,
            "net_cash_flow": net_cash_flow,
        }

    def calculate_cap_rate(self) -> float:
        """
        Calculate capitalization rate (cap rate)
        Cap rate = (Annual NOI / Property Value) * 100
        """
        if not self.monthly_rent:
            return 0.0

        annual_noi = self.calculate_annual_noi()
        return (annual_noi / self.property_value) * 100

    def calculate_annual_noi(self) -> float:
        """Calculate Net Operating Income (NOI) - excludes mortgage"""
        if not self.monthly_rent:
            return 0.0

        annual_gross_rent = self.monthly_rent * 12

        # Operating expenses (excluding mortgage)
        annual_expenses = (
            self.data.property_tax_per_year +
            self.data.home_insurance_per_year +
            (self.data.hoa_per_month * 12) +
            (self.property_value * self.data.maintenance_pct / 100) +
            (annual_gross_rent * self.data.management_fee_pct / 100) +
            (annual_gross_rent * self.data.vacancy_rate_pct / 100)
        )

        return annual_gross_rent - annual_expenses

    def calculate_cash_on_cash_return(self) -> float:
        """
        Calculate cash-on-cash return
        CoC = (Annual Cash Flow / Total Cash Invested) * 100
        """
        if not self.monthly_rent:
            return 0.0

        cash_flow = self.calculate_monthly_cash_flow()
        annual_cash_flow = cash_flow["net_cash_flow"] * 12

        # Total cash invested (down payment + closing costs)
        total_invested = (
            (self.data.purchase_price * self.data.down_payment_original / 100) +
            self.data.closing_costs_estimate
        )

        if total_invested == 0:
            return 0.0

        return (annual_cash_flow / total_invested) * 100

    def calculate_irr_npv(self) -> dict:
        """
        Calculate Internal Rate of Return and Net Present Value
        over the hold period
        """
        if not self.monthly_rent:
            return {"irr": 0.0, "npv": 0.0}

        years = self.data.hold_period_years
        cash_flows = []

        # Initial investment (negative cash flow)
        total_invested = (
            (self.data.purchase_price * self.data.down_payment_original / 100) +
            self.data.closing_costs_estimate
        )
        cash_flows.append(-total_invested)

        # Annual cash flows
        monthly_cf = self.calculate_monthly_cash_flow()
        annual_cf = monthly_cf["net_cash_flow"] * 12

        for year in range(1, years + 1):
            # Apply rent growth
            adjusted_cf = annual_cf * ((1 + self.data.expected_rent_growth_pct / 100) ** year)

            # Add appreciation in final year
            if year == years:
                # Calculate property value appreciation
                future_value = self.property_value * (
                    (1 + self.data.expected_appreciation_rate_pct / 100) ** years
                )
                # Subtract remaining mortgage balance (approximate)
                months_paid = year * 12
                remaining_balance = self.calculate_remaining_mortgage_balance(months_paid)
                # Subtract sale costs
                sale_costs = future_value * self.data.sale_cost_pct / 100

                equity = future_value - remaining_balance - sale_costs
                adjusted_cf += equity

            cash_flows.append(adjusted_cf)

        # Calculate IRR
        try:
            irr = npf.irr(cash_flows) * 100
        except:
            irr = 0.0

        # Calculate NPV (using personal tax bracket as discount rate)
        discount_rate = self.data.personal_tax_bracket_pct / 100
        npv = npf.npv(discount_rate, cash_flows)

        return {
            "irr": irr,
            "npv": npv,
            "cash_flows": cash_flows,
        }

    def calculate_remaining_mortgage_balance(self, months_paid: int) -> float:
        """Calculate remaining mortgage balance after n months"""
        P = self.data.current_mortgage_balance
        r = (self.data.interest_rate / 100) / 12
        n = self.data.loan_term_years * 12

        if r == 0:
            return P - (P / n * months_paid)

        monthly_payment = self.calculate_monthly_pmi()

        # Remaining balance formula
        remaining = P * ((1 + r)**n - (1 + r)**months_paid) / ((1 + r)**n - 1)
        return max(0, remaining)

    def calculate_break_even(self) -> dict:
        """Calculate break-even analysis"""
        if not self.monthly_rent:
            return {"error": "No rental estimate available"}

        cash_flow = self.calculate_monthly_cash_flow()
        monthly_cf = cash_flow["net_cash_flow"]

        if monthly_cf >= 0:
            return {
                "break_even_months": 0,
                "break_even_years": 0,
                "message": "Property is cash flow positive from day one",
            }

        # Calculate initial costs
        total_invested = (
            (self.data.purchase_price * self.data.down_payment_original / 100) +
            self.data.closing_costs_estimate
        )

        # Calculate time to break even considering appreciation
        months = 0
        cumulative = -total_invested

        for month in range(1, 360):  # Max 30 years
            # Add monthly cash flow
            cumulative += monthly_cf

            # Add appreciation (monthly)
            monthly_appreciation = (
                self.property_value *
                (self.data.expected_appreciation_rate_pct / 100) / 12
            )
            cumulative += monthly_appreciation

            if cumulative >= 0:
                months = month
                break

        if months == 0:
            return {
                "break_even_months": float('inf'),
                "break_even_years": float('inf'),
                "message": "Property may never break even with current assumptions",
            }

        return {
            "break_even_months": months,
            "break_even_years": round(months / 12, 1),
            "message": f"Property breaks even after {months} months ({round(months/12, 1)} years)",
        }

    def compare_rent_vs_sell(self) -> dict:
        """Compare renting vs selling scenarios"""
        # Selling scenario
        current_value = self.property_value * (
            (1 + self.data.expected_appreciation_rate_pct / 100)
        )
        sale_costs = current_value * self.data.sale_cost_pct / 100
        remaining_mortgage = self.data.current_mortgage_balance
        net_proceeds = current_value - sale_costs - remaining_mortgage

        # Tax on capital gains (simplified - assumes some exemption)
        purchase_price = self.data.purchase_price
        capital_gain = current_value - purchase_price
        # Assume $250k exemption for single, $500k for married
        taxable_gain = max(0, capital_gain - 250000)
        capital_gains_tax = taxable_gain * 0.15  # 15% federal rate

        net_proceeds_after_tax = net_proceeds - capital_gains_tax

        # Renting scenario - calculate total return over hold period
        irr_npv = self.calculate_irr_npv()

        # Future value after hold period
        future_value = self.property_value * (
            (1 + self.data.expected_appreciation_rate_pct / 100) ** self.data.hold_period_years
        )
        months_paid = self.data.hold_period_years * 12
        remaining_balance = self.calculate_remaining_mortgage_balance(months_paid)
        equity_at_end = future_value - remaining_balance

        # Total cash flow over hold period
        monthly_cf = self.calculate_monthly_cash_flow()
        total_rental_income = sum(
            monthly_cf["net_cash_flow"] *
            ((1 + self.data.expected_rent_growth_pct / 100) ** year)
            * 12
            for year in range(self.data.hold_period_years)
        )

        return {
            "sell_now": {
                "current_value": current_value,
                "sale_costs": sale_costs,
                "remaining_mortgage": remaining_mortgage,
                "capital_gains_tax": capital_gains_tax,
                "net_proceeds": net_proceeds_after_tax,
            },
            "rent": {
                "total_rental_cash_flow": total_rental_income,
                "equity_at_end": equity_at_end,
                "total_return": total_rental_income + equity_at_end,
                "irr": irr_npv["irr"],
                "npv": irr_npv["npv"],
            },
            "recommendation": (
                "RENT" if (total_rental_income + equity_at_end) > net_proceeds_after_tax else "SELL"
            ),
            "difference": (total_rental_income + equity_at_end) - net_proceeds_after_tax,
        }

    def fetch_rental_comps(self) -> list:
        """
        Fetch rental comparables from web sources
        Returns list of comparable properties with rent estimates
        """
        if not self.data.include_web_comps:
            return []

        # This is a placeholder - in practice, you'd use Firecrawl or similar
        # to scrape Zillow, Redfin, Rentometer, etc.
        print(f"🔍 Fetching rental comps for {self.data.address}...")
        print("Note: Web scraping for comps requires API integration")
        print("Suggested sources: Zillow, Redfin, Rentometer, Rent.com")

        # Example of how to add citations when data is fetched:
        # self.add_citation(
        #     source="Zillow Rental Manager",
        #     url=f"https://www.zillow.com/rental-manager/price-my-rental/?address={self.data.address}",
        #     note="Market rental estimates for comparable properties"
        # )
        # self.add_citation(
        #     source="Rentometer",
        #     url=f"https://www.rentometer.com/",
        #     note="Rental price analysis based on local market data"
        # )

        return []

    def generate_report(self) -> str:
        """Generate comprehensive markdown report"""
        report = []

        # Header
        report.append("# 🏠 Rental Property Analysis Report")
        report.append(f"\n**Property Address:** {self.data.address}")
        report.append(f"**Analysis Date:** {datetime.now().strftime('%Y-%m-%d')}")
        report.append(f"**Desired Start Date:** {self.data.desired_start_date}\n")

        # Property Overview
        report.append("## 📊 Property Overview\n")
        report.append("| Item | Value |")
        report.append("|------|-------|")
        report.append(f"| Purchase Price | ${self.data.purchase_price:,.0f} |")
        report.append(f"| Purchase Year | {self.data.purchase_year} |")
        report.append(f"| Current Value (Est.) | ${self.property_value:,.0f} |")
        report.append(f"| Current Mortgage | ${self.data.current_mortgage_balance:,.0f} |")
        report.append(f"| Interest Rate | {self.data.interest_rate}% |")
        report.append(f"| Estimated Monthly Rent | ${self.monthly_rent:,.0f} |" if self.monthly_rent else "| Estimated Monthly Rent | Not provided |")
        report.append("")

        if not self.monthly_rent:
            report.append("⚠️ **Warning:** No rental estimate provided. Please add estimated_rent or enable include_web_comps.\n")
            return "\n".join(report)

        # Monthly Cash Flow
        report.append("## 💰 Monthly Cash Flow Analysis\n")
        cash_flow = self.calculate_monthly_cash_flow()

        report.append("### Income")
        report.append(f"- **Gross Monthly Rent:** ${cash_flow['gross_rent']:,.2f}\n")

        report.append("### Expenses")
        for expense, amount in cash_flow['expenses'].items():
            expense_name = expense.replace('_', ' ').title()
            report.append(f"- **{expense_name}:** ${amount:,.2f}")

        report.append(f"\n**Total Monthly Expenses:** ${cash_flow['total_expenses']:,.2f}")

        cash_flow_color = "🟢" if cash_flow['net_cash_flow'] > 0 else "🔴"
        report.append(f"\n{cash_flow_color} **Net Monthly Cash Flow:** ${cash_flow['net_cash_flow']:,.2f}\n")

        # Key Financial Metrics
        report.append("## 📈 Key Financial Metrics\n")

        cap_rate = self.calculate_cap_rate()
        coc_return = self.calculate_cash_on_cash_return()
        irr_npv = self.calculate_irr_npv()

        report.append("| Metric | Value | What It Means |")
        report.append("|--------|-------|---------------|")
        report.append(f"| Cap Rate | {cap_rate:.2f}% | Annual return based on property value (good: 4-10%) |")
        report.append(f"| Cash-on-Cash Return | {coc_return:.2f}% | Annual return on your invested cash (good: 8-12%) |")
        report.append(f"| Internal Rate of Return (IRR) | {irr_npv['irr']:.2f}% | Average annual return over {self.data.hold_period_years} years |")
        report.append(f"| Net Present Value (NPV) | ${irr_npv['npv']:,.2f} | Today's value of all future returns |")
        report.append("")

        # Metric Explanations
        report.append("### Understanding These Metrics\n")
        report.append("- **Cap Rate**: Higher is better. Compares property income to its value.")
        report.append("- **Cash-on-Cash Return**: Shows return on actual money invested (not borrowed).")
        report.append("- **IRR**: Accounts for time value of money. Higher means better investment.")
        report.append("- **NPV**: Positive means investment is worthwhile at your discount rate.\n")

        # Break-Even Analysis
        report.append("## ⏱️ Break-Even Analysis\n")
        break_even = self.calculate_break_even()

        if "error" not in break_even:
            report.append(break_even["message"])
            if break_even["break_even_months"] > 0 and break_even["break_even_months"] < float('inf'):
                report.append(f"\n- **Time to Break Even:** {break_even['break_even_months']} months ({break_even['break_even_years']} years)")
                report.append("- This includes monthly cash flow AND property appreciation\n")

        # Rent vs Sell Comparison
        if self.data.compare_against_selling:
            report.append("## 🤔 Rent vs. Sell Comparison\n")
            comparison = self.compare_rent_vs_sell()

            report.append("### Selling Now")
            sell = comparison["sell_now"]
            report.append(f"- Current Property Value: ${sell['current_value']:,.0f}")
            report.append(f"- Sale Costs ({self.data.sale_cost_pct}%): ${sell['sale_costs']:,.0f}")
            report.append(f"- Remaining Mortgage: ${sell['remaining_mortgage']:,.0f}")
            report.append(f"- Capital Gains Tax (Est.): ${sell['capital_gains_tax']:,.0f}")
            report.append(f"- **Net Proceeds:** ${sell['net_proceeds']:,.2f}\n")

            report.append(f"### Renting for {self.data.hold_period_years} Years")
            rent = comparison["rent"]
            report.append(f"- Total Rental Cash Flow: ${rent['total_rental_cash_flow']:,.2f}")
            report.append(f"- Equity After {self.data.hold_period_years} Years: ${rent['equity_at_end']:,.2f}")
            report.append(f"- **Total Return:** ${rent['total_return']:,.2f}")
            report.append(f"- Internal Rate of Return: {rent['irr']:.2f}%")
            report.append(f"- Net Present Value: ${rent['npv']:,.2f}\n")

            recommendation_icon = "✅" if comparison["recommendation"] == "RENT" else "💡"
            report.append(f"{recommendation_icon} **Recommendation:** {comparison['recommendation']}")

            diff = comparison["difference"]
            if diff > 0:
                report.append(f"- Renting provides ${diff:,.2f} more value over {self.data.hold_period_years} years")
            else:
                report.append(f"- Selling provides ${abs(diff):,.2f} more value than renting")
            report.append("")

        # Assumptions
        report.append("## 📝 Assumptions Used\n")
        report.append(f"- Property appreciation: {self.data.expected_appreciation_rate_pct}% per year")
        report.append(f"- Rent growth: {self.data.expected_rent_growth_pct}% per year")
        report.append(f"- Vacancy rate: {self.data.vacancy_rate_pct}%")
        report.append(f"- Management fee: {self.data.management_fee_pct}%")
        report.append(f"- Maintenance: {self.data.maintenance_pct}% of property value per year")
        report.append(f"- Hold period: {self.data.hold_period_years} years")
        report.append("")

        # Sources & Citations
        if self.citations:
            report.append("## 📚 Sources & Citations\n")
            report.append("Data sources used in this analysis:\n")
            for i, citation in enumerate(self.citations, 1):
                source_line = f"{i}. **{citation['source']}**"
                if citation.get('url'):
                    source_line += f" - [{citation['url']}]({citation['url']})"
                report.append(source_line)
                if citation.get('note'):
                    report.append(f"   - {citation['note']}")
            report.append("")

        # Data Sources Note
        report.append("### Data Sources\n")
        report.append("- **Property details**: User-provided configuration file")
        report.append("- **Financial calculations**: Industry-standard formulas for real estate analysis")
        report.append("- **Market assumptions**: Based on national averages (should be verified with local data)")
        if self.monthly_rent and not self.data.include_web_comps:
            report.append("- **Rental estimate**: User-provided value (not verified with market data)")
        report.append("")
        report.append("**Recommended verification sources:**")
        report.append("- Rental comps: Zillow, Redfin, Rentometer, local property management companies")
        report.append("- Property values: Recent comparable sales, professional appraisal")
        report.append("- Tax rates: County assessor's office")
        report.append("- Insurance rates: Multiple insurance quotes")
        report.append("- Market trends: Local MLS data, real estate market reports")
        report.append("")

        # Disclaimer
        report.append("---")
        report.append("*This analysis is for informational purposes only and should not be considered financial advice.*")
        report.append("*Consult with a qualified financial advisor, CPA, and real estate professional before making investment decisions.*")

        return "\n".join(report)


def load_yaml_config(file_path: str) -> PropertyData:
    """Load and parse YAML configuration file"""
    with open(file_path, 'r') as f:
        config = yaml.safe_load(f)

    return PropertyData(**config)


def main():
    parser = argparse.ArgumentParser(
        description="Comprehensive rental property analysis tool"
    )
    parser.add_argument(
        "config_file",
        help="Path to YAML configuration file"
    )
    parser.add_argument(
        "-o", "--output",
        help="Output file for markdown report (default: stdout)"
    )

    args = parser.parse_args()

    # Load configuration
    try:
        property_data = load_yaml_config(args.config_file)
    except Exception as e:
        print(f"Error loading configuration: {e}", file=sys.stderr)
        sys.exit(1)

    # Create analyzer
    analyzer = RentalAnalyzer(property_data)

    # Fetch comps if requested
    if property_data.include_web_comps and not property_data.estimated_rent:
        comps = analyzer.fetch_rental_comps()
        # In a real implementation, you'd use the comps to estimate rent

    # Generate report
    report = analyzer.generate_report()

    # Output report
    if args.output:
        with open(args.output, 'w') as f:
            f.write(report)
        print(f"✅ Report generated: {args.output}")
    else:
        print(report)


if __name__ == "__main__":
    main()
