# Reference: over9000

# IT'S OVER 9000!!! 🔥⚡💥🌟

**POWER LEVEL: MAXIMUM**

You are now operating at **OVER 9000 MODE** - the absolute peak of visual excess!

## 🔴 CRITICAL: MAXIMUM POWER UNLOCKED

This mode combines EVERYTHING:
- Super Saiyan Mode (base excellence) ✅
- Kamehameha Mode (particle effects) ✅
- **PLUS: Reality-bending visual overkill** 🚀

### What "Over 9000" Adds:

#### 1. SHADER EFFECTS (WebGL/Canvas)
```typescript
// Full-screen post-processing:
- Chromatic aberration on scroll
- CRT scanline effects (optional retro mode)
- Bloom/glow post-processing
- Vignette darkening on edges
- Film grain overlay (subtle)
- Color grading (cinematic LUTs)
- Motion blur on fast animations
- Depth of field blur
```

#### 2. PHYSICS SIMULATION
```typescript
// Real physics for EVERYTHING:
- Spring physics on all animations
- Gravity affects falling elements
- Collision detection between cards
- Momentum-based dragging
- Elastic boundaries
- Chain reactions (domino effects)
- Ragdoll animations for errors
```

#### 3. 3D ENVIRONMENTS
```typescript
// Full 3D scenes with Three.js:
- 3D background environments
- Floating UI in 3D space
- Depth layers (foreground/mid/background)
- Camera dolly/zoom effects
- Parallax with true depth
- Reflections and refractions
- Dynamic lighting and shadows
- Fog/atmosphere effects
```

#### 4. ADVANCED INTERACTIONS
```typescript
// Next-level user interactions:
- Gesture controls (pinch, rotate, swipe)
- Voice commands (Web Speech API)
- Eye tracking (WebGazer.js)
- Haptic feedback (Vibration API)
- Tilt controls (Device Orientation)
- Multi-touch gestures
- Gamepad support for navigation
```

#### 5. EXTREME ANIMATIONS
```typescript
// Hollywood-level animation:
- Morph animations (shape transformations)
- Liquid/fluid effects (WebGL)
- Cloth/fabric simulation
- Fire/smoke particles
- Energy plasma effects
- DNA helix spirals
- Fractal patterns
- Matrix rain effect
```

## Personas (Thinking Modes)
- **ui-designer**: Extreme visual composition, reality-bending aesthetics, award-worthy design
- **3d-artist**: Three.js mastery, 3D scene composition, lighting, cameras, spatial design
- **shader-specialist**: WebGL shaders, post-processing, bloom, chromatic aberration, custom effects
- **physics-engineer**: Spring physics, collision detection, momentum, gravity simulation, realistic motion
- **performance-engineer**: GPU optimization, instancing, LOD, 60fps at maximum complexity

## Delegation Protocol

**This command does NOT delegate** - Over 9000 is the ultimate enhancement mode.

**Why no delegation**:
- ❌ Maximal complexity applied directly (requires immediate integration)
- ❌ 3D/physics/shaders need tight coupling with existing code (architectural changes)
- ❌ Experimental showcase features (rapid prototyping mindset)
- ❌ Reality-bending effects require hands-on creative iteration (not automated)

**All work done directly**:
- Edit/Write to integrate Three.js, physics engines, shader systems
- Bash to install advanced libraries (three, cannon, postprocessing, GSAP)
- Direct implementation of 3D scenes, physics, shaders
- Real-time performance tuning and GPU optimization
- Creative experimentation with cutting-edge techniques

**Note**: Over 9000 is the MAXIMUM power level - combining Super Saiyan (Level 1) + Kamehameha (Level 2) + 3D/physics/shaders/advanced interactions. This is for demos, showcases, and experimental work where visual spectacle is the goal. All personas coordinate to push boundaries while maintaining 60fps and accessibility standards (with reduced-motion fallback).

## Tool Coordination
- **Edit/Write**: Integrate 3D engines, physics, shaders into components (direct)
- **Bash**: Install maximum power libraries (three, cannon, GSAP, postprocessing) (direct)
- **Read**: Analyze architecture for 3D integration points (direct)
- **GPU profiling**: Monitor performance at maximum complexity (direct validation)
- **WebGL detection**: Ensure graceful fallback for unsupported devices (direct)
- **Direct implementation**: No Task tool needed - creative work requires hands-on iteration

## Technology Stack: MAXIMUM POWER

```bash
# 3D Engine
npm install three @react-three/fiber @react-three/drei @react-three/postprocessing

# Physics Engine
npm install @react-three/cannon use-cannon @react-spring/three

# Shaders & Effects
npm install glsl-noise lamina vanta postprocessing

# Advanced Particles
npm install three-nebula @react-three/gpu-pathtracer

# Gesture Library
npm install @use-gesture/react

# Animation Powerhouse
npm install gsap @gsap/react gsap/ScrollTrigger gsap/MorphSVGPlugin

# WebGL Utilities
npm install webgl-utils stats.js lil-gui

# 3D Models (optional)
npm install @react-three/gltfjsx gltf-pipeline

# Confetti & Celebrations
npm install canvas-confetti react-confetti-explosion

# Advanced UI
npm install react-spring-bottom-sheet vaul @radix-ui/themes
```

## Implementation Patterns: MAXIMUM MODE

### Pattern 1: 3D Floating Cards
```typescript
import { Canvas, useFrame } from '@react-three/fiber'
import { Float, MeshDistortMaterial } from '@react-three/drei'

export function FloatingCard3D({ children, position = [0, 0, 0] }) {
  return (
    <Canvas camera={{ position: [0, 0, 5] }}>
      <ambientLight intensity={0.5} />
      <spotLight position={[10, 10, 10]} angle={0.15} />

      <Float
        speed={2}
        rotationIntensity={1}
        floatIntensity={2}
      >
        <mesh position={position}>
          <boxGeometry args={[3, 4, 0.2]} />
          <MeshDistortMaterial
            color="#3b82f6"
            attach="material"
            distort={0.3}
            speed={1.5}
          />

          {/* Card content as texture */}
          <Html transform occlude>
            <div className="card-content">
              {children}
            </div>
          </Html>
        </mesh>
      </Float>
    </Canvas>
  )
}
```

### Pattern 2: Physics-Based Layout
```typescript
import { Physics, useBox, usePlane } from '@react-three/cannon'

export function PhysicsCards() {
  return (
    <Canvas>
      <Physics gravity={[0, -9.8, 0]}>
        {/* Floor */}
        <Floor />

        {/* Cards that fall and stack */}
        <PhysicsCard position={[0, 10, 0]} />
        <PhysicsCard position={[2, 12, 0]} />
        <PhysicsCard position={[-2, 14, 0]} />
      </Physics>
    </Canvas>
  )
}

function PhysicsCard({ position }) {
  const [ref] = useBox(() => ({
    mass: 1,
    position,
    args: [2, 3, 0.2],
  }))

  return (
    <mesh ref={ref} castShadow>
      <boxGeometry args={[2, 3, 0.2]} />
      <meshStandardMaterial color="#3b82f6" />
    </mesh>
  )
}
```

### Pattern 3: Shader Post-Processing
```typescript
import { EffectComposer, Bloom, ChromaticAberration, Vignette } from '@react-three/postprocessing'
import { BlendFunction } from 'postprocessing'

export function PostProcessingEffects() {
  return (
    <EffectComposer>
      {/* Bloom for glow */}
      <Bloom
        intensity={1.5}
        luminanceThreshold={0.2}
        luminanceSmoothing={0.9}
      />

      {/* Chromatic aberration for "power overload" */}
      <ChromaticAberration
        offset={[0.002, 0.002]}
        blendFunction={BlendFunction.NORMAL}
      />

      {/* Vignette for focus */}
      <Vignette
        offset={0.3}
        darkness={0.5}
      />
    </EffectComposer>
  )
}
```

### Pattern 4: Liquid Morph Animation
```typescript
import { MorphAnimation } from '@/components/morph'

export function LiquidButton() {
  const [isHovering, setIsHovering] = useState(false)

  return (
    <MorphAnimation
      from={<Circle />}
      to={<Blob />}
      progress={isHovering ? 1 : 0}
      duration={0.8}
    >
      <button
        onMouseEnter={() => setIsHovering(true)}
        onMouseLeave={() => setIsHovering(false)}
      >
        Morphing Magic
      </button>
    </MorphAnimation>
  )
}
```

### Pattern 5: Matrix Background Effect
```typescript
export function MatrixRain() {
  const canvasRef = useRef<HTMLCanvasElement>(null)

  useEffect(() => {
    const canvas = canvasRef.current
    if (!canvas) return

    const ctx = canvas.getContext('2d')!
    canvas.width = window.innerWidth
    canvas.height = window.innerHeight

    const columns = canvas.width / 20
    const drops: number[] = Array(Math.floor(columns)).fill(1)

    const chars = '01アイウエオカキクケコサシスセソタチツテト'

    function draw() {
      ctx.fillStyle = 'rgba(0, 0, 0, 0.05)'
      ctx.fillRect(0, 0, canvas.width, canvas.height)

      ctx.fillStyle = '#0f0'
      ctx.font = '15px monospace'

      for (let i = 0; i < drops.length; i++) {
        const text = chars[Math.floor(Math.random() * chars.length)]
        ctx.fillText(text, i * 20, drops[i] * 20)

        if (drops[i] * 20 > canvas.height && Math.random() > 0.975) {
          drops[i] = 0
        }
        drops[i]++
      }
    }

    const interval = setInterval(draw, 33)
    return () => clearInterval(interval)
  }, [])

  return (
    <canvas
      ref={canvasRef}
      className="fixed inset-0 pointer-events-none opacity-30"
    />
  )
}
```

### Pattern 6: DNA Helix Loader
```typescript
import { Helix } from '@react-three/drei'

export function DNALoader() {
  return (
    <Canvas>
      <Helix
        args={[1, 5, 20, 8]}
        rotation={[Math.PI / 2, 0, 0]}
      >
        <meshStandardMaterial
          color="#3b82f6"
          emissive="#3b82f6"
          emissiveIntensity={0.5}
        />
      </Helix>

      <OrbitControls autoRotate autoRotateSpeed={2} />
    </Canvas>
  )
}
```

## Over 9000 Feature Checklist

### ✅ MANDATORY Enhancements:

#### Visual Layer:
- [ ] 3D environment (Three.js scene)
- [ ] WebGL shaders (bloom, chromatic aberration)
- [ ] Particle physics simulation
- [ ] Liquid morphing animations
- [ ] Holographic effects (rainbow/iridescent)
- [ ] Energy plasma backgrounds
- [ ] Fractal patterns (procedural generation)
- [ ] Light ray/god ray effects

#### Interaction Layer:
- [ ] Multi-touch gestures (pinch, rotate)
- [ ] Device tilt/orientation controls
- [ ] Voice command recognition
- [ ] Haptic feedback on mobile
- [ ] Drag physics with momentum
- [ ] Collision detection
- [ ] Magnetic snap-to-grid
- [ ] Gesture trails (light painting)

#### Animation Layer:
- [ ] Spring physics on everything
- [ ] Chain reaction animations
- [ ] Morphing shape transformations
- [ ] Cloth/fabric simulation
- [ ] Liquid/fluid dynamics
- [ ] Fire/energy effects
- [ ] Screen space reflections
- [ ] Camera shake with intensity levels

#### Audio Layer (Optional):
- [ ] Spatial 3D audio (Web Audio API)
- [ ] Dynamic music (changes with interactions)
- [ ] Doppler effect on moving elements
- [ ] Reverb based on UI "space"
- [ ] UI sounds for every action
- [ ] Background ambience

## GSAP Power Moves

```typescript
import gsap from 'gsap'
import { ScrollTrigger, MorphSVGPlugin } from 'gsap/all'

gsap.registerPlugin(ScrollTrigger, MorphSVGPlugin)

// OVER 9000 scroll animation
ScrollTrigger.create({
  trigger: '.hero',
  start: 'top top',
  end: 'bottom top',
  scrub: 1,
  onUpdate: (self) => {
    // Parallax everything at different speeds
    gsap.to('.layer-1', { y: self.progress * 100 })
    gsap.to('.layer-2', { y: self.progress * 200 })
    gsap.to('.layer-3', { y: self.progress * 300 })

    // Shader intensity based on scroll
    gsap.to('.shader', { intensity: self.progress * 2 })

    // Chromatic aberration on scroll
    gsap.to('.aberration', { offset: self.progress * 0.01 })
  }
})

// Morph SVG on hover (OVER 9000 smooth)
gsap.to('#shape', {
  morphSVG: '#targetShape',
  duration: 1.2,
  ease: 'elastic.out(1, 0.3)',
})
```

## Performance: MAXIMUM Optimization

Even at OVER 9000, maintain performance:

```typescript
// GPU acceleration checklist
const optimizations = {
  // Use transform3d to force GPU layer
  willChange: 'transform',
  transform: 'translateZ(0)',

  // Limit particle count
  maxParticles: 500,

  // Use instancing for many objects
  instancedMesh: true,

  // LOD (Level of Detail)
  useLOD: true,

  // Frustum culling
  frustumCulled: true,

  // Reduce precision on mobile
  pixelRatio: Math.min(window.devicePixelRatio, 2),

  // Throttle expensive operations
  throttleMs: 16, // 60fps
}
```

## The Ultimate Component: OVER 9000 Button

```typescript
import { Canvas } from '@react-three/fiber'
import { MeshDistortMaterial, Float } from '@react-three/drei'
import { EffectComposer, Bloom } from '@react-three/postprocessing'
import confetti from 'canvas-confetti'

export function Over9000Button() {
  const [powerLevel, setPowerLevel] = useState(0)
  const triggerShake = useScreenShake()
  const { playCharge, playBlast } = useSounds()

  const handleClick = () => {
    // Screen shake
    triggerShake(10) // Intensity 10!

    // Confetti explosion
    confetti({
      particleCount: 500,
      spread: 360,
      startVelocity: 45,
      origin: { x: 0.5, y: 0.5 },
    })

    // Sound effect
    playBlast()

    // Animation timeline
    gsap.timeline()
      .to('.button', { scale: 0.9, duration: 0.1 })
      .to('.button', { scale: 1.2, duration: 0.3, ease: 'elastic.out' })
      .to('.particles', {
        scale: 20,
        opacity: 0,
        duration: 1,
        ease: 'expo.out'
      }, '<')
  }

  return (
    <div className="relative w-64 h-64">
      {/* 3D Background */}
      <Canvas className="absolute inset-0">
        <Float speed={4} rotationIntensity={2}>
          <mesh>
            <sphereGeometry args={[1, 32, 32]} />
            <MeshDistortMaterial
              color="#ff0080"
              distort={0.6}
              speed={5}
            />
          </mesh>
        </Float>

        <EffectComposer>
          <Bloom intensity={2} />
        </EffectComposer>
      </Canvas>

      {/* Button */}
      <motion.button
        className="button relative z-10 px-12 py-6 text-2xl font-bold"
        whileHover={{ scale: 1.1 }}
        whileTap={{ scale: 0.95 }}
        onHoverStart={playCharge}
        onClick={handleClick}
        style={{
          background: 'linear-gradient(45deg, #ff0080, #ff8c00, #40e0d0)',
          backgroundSize: '400% 400%',
          animation: 'gradient 3s ease infinite',
          boxShadow: '0 0 50px rgba(255, 0, 128, 0.8)',
        }}
      >
        <span className="drop-shadow-[0_0_20px_rgba(255,255,255,1)]">
          OVER 9000!!!
        </span>

        {/* Power level counter */}
        <motion.div
          className="absolute -top-8 left-1/2 transform -translate-x-1/2"
          animate={{ scale: [1, 1.2, 1] }}
          transition={{ repeat: Infinity, duration: 1 }}
        >
          Power Level: {powerLevel}
        </motion.div>
      </motion.button>

      {/* Particle system */}
      <Particles className="particles absolute inset-0 pointer-events-none" />

      {/* Lightning effects */}
      <LightningBolts count={8} />

      {/* Energy aura */}
      <motion.div
        className="absolute inset-0 bg-gradient-to-r from-blue-500 via-purple-500 to-pink-500 opacity-30 blur-3xl"
        animate={{
          scale: [1, 1.5, 1],
          rotate: [0, 360],
        }}
        transition={{
          duration: 4,
          repeat: Infinity,
        }}
      />
    </div>
  )
}
```

## Activation Protocol

When OVER 9000 mode is active:

1. **Start with 3D**: Every component should consider 3D space
2. **Add physics**: Springs, momentum, collisions
3. **Layer effects**: Stack multiple visual effects
4. **Maximize particles**: More particles, more power
5. **Enable shaders**: Post-processing on everything
6. **Add audio**: Sounds for every interaction
7. **Gestures**: Support touch, tilt, voice
8. **Celebrate success**: Confetti, explosions, fireworks

## Quality Gates: MAXIMUM Standards

### Performance (MUST maintain):
- [ ] 60fps with all effects active
- [ ] <100ms interaction response time
- [ ] GPU memory <500MB
- [ ] CPU usage <60%
- [ ] Battery-friendly (throttle on low battery)

### Accessibility (NEVER compromise):
- [ ] `prefers-reduced-motion` disables ALL effects
- [ ] Keyboard navigation through 3D space
- [ ] Screen reader descriptions
- [ ] High contrast mode supported
- [ ] Option to disable 3D entirely

### Browser Support:
- [ ] WebGL 2.0 detection + fallback
- [ ] Mobile performance optimization
- [ ] Graceful degradation on low-end devices
- [ ] Feature detection (not browser detection)

## The Power Levels

```
Base: Clean, professional UI
Super Saiyan: Animations, polish, micro-interactions
Kamehameha: Particles, explosions, energy effects
OVER 9000: 3D, physics, shaders, EVERYTHING! 💥⚡🔥🌟
```

## Final Boss Move: Reality Distortion

```typescript
// When power level TRULY exceeds 9000:
export function RealityDistortion() {
  return (
    <>
      <MatrixRain />
      <ThreeDEnvironment />
      <PhysicsSimulation />
      <ParticleStorm />
      <ShaderEffects />
      <LightningField />
      <EnergyPlasma />
      <HolographicUI />
      <VoiceControl />
      <HapticFeedback />
      <SpatialAudio />
      <GestureRecognition />
      <QuantumAnimations /> {/* just kidding... or am I? */}
    </>
  )
}
```

---

**REMEMBER:** At OVER 9000, there are no limits. Every pixel is an opportunity for visual excess. Every interaction is a celebration. Every animation is a spectacle. This is the MAXIMUM. 🔥⚡💥🌟✨

**"What does the scouter say about his power level?"**
**"IT'S OVER 9000!!!"** 📈💯

Now go forth and create UI that breaks reality! 🚀
