# PyRevit Extension Architecture Patterns for Enterprise Revit Workflows



## Extension Bundle Structure



A well-organized **extension bundle** is the foundation for maintainable PyRevit tools. PyRevit uses a folder-based convention for grouping scripts into tabs, panels, and buttons on the Revit Ribbon. To create an extension, you make a folder named `<YourExtensionName>.extension` and then add subfolders for UI groupings. For example, if your plugin is called **AwesomeTools**, you would create **AwesomeTools.extension** and inside it organize tools into tabs/panels and command bundles:



```

AwesomeTools.extension/  

|-- AwesomeTools.tab  

|   |-- Manage.panel  

|   |   |-- Cleanup.pushbutton  

|   |   |   |-- script.py  

|   |   |   `-- icon.png  

|   |   `-- Reports.pushbutton  

|   |       |-- script.py  

|   |       `-- icon.png  

|   `-- Data.panel  

|       |-- Export.pushbutton  

|       |   |-- script.py  

|       |   `-- icon.png  

|       `-- BulkEdit.pushbutton  

|           |-- script.py  

|           `-- icon.png  

|-- lib/  

|   `-- utils.py  

`-- extension.json

```



*(The above example shows an extension with one Ribbon Tab "AwesomeTools", containing two Panels ("Manage" and "Data"), each with a few push-button tools. A shared `lib` folder holds common code, and an `extension.json` file can store metadata like name and version.)* This structure follows PyRevit's conventions: each tool is a folder ending in `.pushbutton` (for a command button) or another bundle type, containing a `script.py` that executes when clicked, an `icon.png` for the button image, and optionally a `bundle.yaml` or `extension.json` for advanced settings. Group folders like `.panel` or `.pulldown` bundle multiple commands and can include a `_layout` file to control the order of buttons, separators, and slide-out menus in the UI. Use `_layout` to create logical groupings - for example, list important tools first, insert `"---"` to add a separator, or use `">>>"` to move less-used tools into an expandable flyout.



**Organizing large extensions:** In enterprise environments, you might end up with many tools. It's wise to break them into multiple panels or even multiple tabs for clarity. For instance, you could have separate panels for "Data Management" vs. "QA/QC" tools under one tab. If your extension serves multiple user groups (architecture, structure, MEP, etc.), consider separating each into its own tab or even separate extension bundles for each discipline. PyRevit supports multiple tabs in one extension, but note that Revit has a limit on the number of API-created tabs, so don't go overboard. Some firms split a very large toolset into discipline-specific extensions (e.g. ARCH.extension, STRU.extension) to keep things modular. The key is to ensure the hierarchy (Tab -> Panel -> Button) mirrors how users will logically use the tools.



**Resource sharing:** Avoid duplicating common code or assets across multiple scripts. PyRevit allows **library folders** for sharing resources:



* Include a **`lib/` folder** inside an extension or any bundle. Any Python modules in a `lib` folder are added to the IronPython module search path for that extension or group. For example, placing a `lib` folder under a panel bundle makes its modules available to all pushbuttons in that panel. In the structure above, `utils.py` in the extension's `lib` can be imported by all scripts (e.g., `from utils import some_helper`).

* If you have utilities that should be shared **across different extensions**, use a **Library Extension** (a folder ending in `.lib`). PyRevit will add all `.lib` extension paths to `sys.path` for every script, making those modules globally available. This is useful for enterprise-wide libraries (for example, a custom **RevitPythonWrapper** library or common API wrappers that many extensions use).

* Share non-code resources like icons or data files by keeping them in a known location within the extension and referencing them via relative paths in your scripts. For instance, you might have a subfolder for icons if you prefer, or store JSON data files that multiple scripts read. The PyRevit runtime gives your script a `__file__` variable; you can derive the extension path from that to locate resources. This approach prevents hardcoding file paths.

* **Best practice:** Keep your extension bundle independent of the core pyRevit install. Never modify pyRevit's core files; instead, use your own extension folder (configured via pyRevit settings to load your extensions). This isolation ensures you can update pyRevit or your tools separately without conflicts.



## Script Loading & Performance



Enterprise Revit use means some scripts will be heavy. Optimizing how and when code runs will improve both performance and Revit's stability during long sessions.



**IronPython engine considerations:** PyRevit runs on IronPython 2.7 within Revit's process. All your extension scripts execute in this IronPython environment, which persists as long as Revit is open. Modules imported by one script are **cached in memory**, so importing a module in one tool makes it available to others without needing to re-import (unless pyRevit reloads or you explicitly unload it). Take advantage of this:



* **Avoid redundant imports:** If you have large .NET libraries or complex Python modules (that are IronPython-compatible) used by many tools, import them in a common module (e.g. your extension's `lib`) so they load once. Subsequent scripts can just do `from lib import that_module` without the heavy import cost each time.

* **Lazy import heavy modules:** Conversely, do not import everything up-front in every script - import inside functions when needed to keep initial load fast. For example, if a script uses `xlsxwriter` or a large XML library only when exporting, import it in the export function, not at the top of the script. This defers the cost until the user actually invokes that part of the code.

* **Startup initialization:** PyRevit allows defining a **startup script** (named `startup.py` at the extension root) that runs when pyRevit is loaded (and on each pyRevit reload). You can use this to perform one-time setup, such as reading a configuration file, initializing a database connection, or caching frequently needed data in memory. Because it runs only once per Revit session, heavy setup code here won't slow down every button click. (Be cautious not to make Revit's startup too slow; try to keep startup tasks lightweight or asynchronous.)



**Heavy computations and CPython integration:** IronPython has limitations - notably it can't use CPython C-extension packages like NumPy/Pandas, and it may be slower for intensive math. A common pattern for enterprise tools is to offload heavy data processing to a separate **CPython** process:



* Use IronPython for what it's best at: interacting with Revit's .NET API and the UI forms. For computational or data-heavy tasks (large Excel operations, geometry calculations, machine learning, etc.), delegate to an external Python script/program. PyRevit's runtime can launch CPython as needed, but since the pyRevit API isn't fully available in CPython yet, the typical approach is to use the **`subprocess`** module. For example, your IronPython script can call `subprocess.check_output(['python', 'do_heavy_calc.py', args])` to run an external Python 3 program, then capture its output.

* **Ensure communication is efficient:** Since real-time communication between IronPython and CPython is limited, have the subprocess do as much work as possible on its own. Exchange data via files or standard I/O in a simple format (JSON, CSV, etc.). For instance, to handle a complex data analysis across multiple Revit models, your IronPython tool could collect minimal inputs (like element IDs or parameter values), invoke a CPython script (which maybe uses Pandas/NumPy) to process data and write results to a JSON, then IronPython reads the JSON to apply results in Revit. This adds complexity but greatly expands capability.

* **Use Poetry/virtual environments for CPython:** Since you'll manage external Python packages separately, use a tool like **Poetry** to lock dependencies in a virtual environment. This ensures that all users running the CPython part have the same library versions. You can even package the virtual environment with the extension or instruct IT to install it on all machines. Some developers automate bundling pure-python packages into the extension - e.g., creating a `.lib` with needed packages by installing them in a venv and copying the site-packages (this works for pure Python packages, but not ones with C extensions). In short, manage your Python dependencies as a proper project, even though pyRevit itself doesn't use pip.



**Memory management in long Revit sessions:** Revit may stay open all day, so any memory leaks in your scripts can accumulate. IronPython uses .NET's garbage collector, which may not immediately reclaim Python objects especially if circular references exist. Keep an eye on memory:



* When working with Revit API objects, dispose of them if possible. For example, if you open a background Document in an IronPython script (say to read data from another Revit file), make sure to call its `Dispose()` method after closing it. This frees the unmanaged memory. Similarly, large transactions or operations might hold onto memory; you can sometimes force .NET GC to run (via `gc.collect()` in IronPython) after heavy operations, though this is usually handled by Revit/pyRevit.

* Remove references to large Python data structures when done. If you have module-level lists or dictionaries caching data, and they're no longer needed, set them to None or reinitialize them. This helps the GC know it can clean up. Be wary of static or global variables that grow over time.

* **Symptom:** If you notice Revit getting slower with each run of a tool (e.g., UI lag or increasing memory usage), it could be leftover objects. One known tip is to use the Autodesk API classes directly instead of pyRevit wrappers if performance is critical. The pyRevit wrappers (like `rpw` or `pyrevit.framework` helpers) can simplify coding but might introduce overhead. In time-sensitive code, calling the Autodesk Revit API classes/methods directly (e.g., `FilteredElementCollector` from `Autodesk.Revit.DB` vs. a wrapper) can be faster.

* **Profile and test:** For enterprise-scale scripts (scanning tens of thousands of elements, etc.), do some performance testing. Use timers and the pyRevit output to identify slow sections. This will guide you on where to optimize (e.g., reducing API calls, using efficient LINQ queries via `Autodesk.Revit.DB` where possible, etc.).



**Optimizing user feedback:** A script that *appears* to freeze will frustrate users even if it's working correctly. Always provide feedback for long tasks (we'll cover specific UI feedback in a later section). From a performance standpoint, giving feedback (like a progress bar) can also allow you to strategically yield control periodically, so Revit's UI remains responsive and doesn't think the add-in has hung. PyRevit's `forms.ProgressBar` is handy - you can update it less frequently for very large tasks (e.g., update every 10% instead of every single item) to reduce overhead.



In summary, **load what you need, when you need it**. Use the persistent nature of pyRevit's IronPython to your benefit (caching modules and data), but also clean up after yourself. And when IronPython isn't enough, don't be afraid to create a hybrid approach using CPython for heavy lifting - many enterprise PyRevit deployments use this pattern to combine Revit's API (IronPython) with modern data science libraries (CPython).



## Inter-Script Communication and State Management



Within a large extension, you'll want tools to work together smoothly. For example, one button might collect data that another button analyzes, or you may need to maintain state (like user settings or cached results) across multiple script runs. PyRevit provides mechanisms to **share data and communicate** between scripts in a controlled way.



* **Shared libraries and modules:** The simplest way for scripts to share logic or data is to put that logic in a common module that each script can import. As mentioned, a `lib/` folder at the extension or panel level is ideal for this. You might create a module (e.g. `common_utils.py`) that contains functions and classes used by multiple commands. Each `script.py` can import those without duplication. If you need to share *runtime data* (not just functions), you can also use a module as a data container. For example, `common_utils.py` could have a dictionary or a singleton object that one script populates and others read. Because the IronPython engine persists, that module (once imported) stays in memory, effectively acting as an in-memory database. **Note:** This works only within one Revit session; once Revit is closed, the memory is gone. Also, be careful to manage concurrency - if two scripts run almost simultaneously (rare in Revit, since typically one command executes at a time), they could interfere with shared data. In practice, setting a value in one script and reading it in another later is fine.



* **Persistent data across sessions or for later use:** Often you need to save information so that it's available the next time the user runs the tool (or opens the model). PyRevit's **script configuration** API is useful here. You can call `pyrevit.script.get_config()` to get a config storage for your script, and `set_config()` to save values to it. Under the hood, pyRevit stores these in the user's `%APPDATA%/pyRevit/pyRevit_config.ini` file. This is great for user-specific settings like "remember last used option" or toggles. The data persists between Revit sessions for that user. Additionally, pyRevit offers `script.store_data(key, data_obj)` and `script.load_data(key)` which let you save more complex data (not just simple INI strings) by pickling it behind the scenes. For example, one pushbutton could compile a list of elements or a dictionary of results and call `script.store_data("AnalysisResults", that_dict)`. Another button (or a later run of the same button) can retrieve it with `results = script.load_data("AnalysisResults")`. PyRevit ensures this is stored in a safe location (often tied to the extension or script name) so it doesn't conflict with other scripts. This method is excellent for caching - say you have a time-consuming operation (like reading all room data), you can store it once and reuse it until the data changes or the model closes. **Caveat:** The stored data is typically local to the user (not shared between different users) and might need refresh if the model changes significantly.



* **Project-wide data (across users):** If you need to share state across multiple users in the same Revit project (for example, a list of "approved" elements or a timestamp of when a tool was last run on the model), consider using the Revit API's **Extensible Storage**. This allows you to attach custom data to the Revit document itself. PyRevit doesn't abstract this, but you can use the Revit API (e.g., `DataStorage` elements and `Schema` classes) to save information inside the model. For instance, you could save a JSON string of certain results in a hidden `DataStorage` element. All users who open the model (with your extension) can then retrieve that data via the API. This approach is more complex (and requires handling schema GUIDs, etc.), but it's robust for shared data. One pattern is to save per-user data in extensible storage by encoding the username in the key, so that each user's preferences for that project travel with the file (or simply mark data with the user name). Use this sparingly - only for data that truly needs to live in the model.



* **Event-driven workflows:** PyRevit supports **event hooks** that let you run scripts automatically on certain Revit events, which is a powerful way to coordinate scripts without user clicks. By creating a `hooks/` folder in your extension and placing scripts with special names, pyRevit will subscribe them to Revit events. Key hooks include:



  * `startup.py` in `hooks/`: runs when Revit starts (after pyRevit loads) - good for initialization if you need to conditionally set up anything.

  * `doc-opened.py`: runs when a document is opened (or created). You get an event argument with the opened `Document`. This could trigger scripts like auto-loading of shared parameters, checking if the model has certain data (and injecting if not), or displaying a welcome/message.

  * `doc-closed.py`: runs on document close, useful for cleanup (e.g., release file locks, clear caches).

  * `view-activated.py`: runs when the user switches to a new view. Could be used for dynamic UI updates or tracking.

  * Many others exist (e.g., `document-saved`, `document-printing` etc., as supported by Revit API events). You simply name the hook script `<event-name>.py` (with hyphen replacing spaces, and prefix with anything or nothing before the event name). PyRevit matches it to the Revit event. Inside a hook script, you can access `pyrevit.EXEC_PARAMS` (or import as `hook`) to get event details.



  **Using hooks for coordination:** Suppose you have a two-step process where one button marks certain elements and another button processes them. Rather than forcing the user to click two buttons in sequence, you could have the first button do its marking and then store some state (maybe a simple flag via `script.set_config` or a list of IDs via `store_data`). Then you could have a `doc-opened` or `view-activated` hook check if that state exists and automatically trigger the second process (or simply alert the user). However, be cautious: hooks run for every user who has the extension, which is great for things like enforcement (e.g., "every time anyone opens a model, check for standard compliance") but could surprise users if overused. Always provide a way to disable a hook if it's not universally needed (even if that means the hook script checks a config setting and exits immediately if the user opted out).



* **Direct scripting vs. modular functions:** In an enterprise scenario, you might have multiple buttons that need to run parts of a shared workflow. A good pattern is to refactor common functionality into functions or classes in your `lib` modules. Then each button's `script.py` becomes a short orchestrator that calls those functions. This not only avoids code duplication but also means one tool can **call** another internally. For example, your "Quality Check" button could internally call the same functions that the "Fix Issues" button uses, just in a different mode. If you find yourself wanting one button to trigger another button, it's usually better to move the core logic into a shared module and call it directly, rather than simulate a button click.



In summary, **PyRevit is flexible in sharing state**: use in-memory modules for within-session data, config storage for persisted user settings, extensible storage for model-bound data, and hooks for automating reactions to events. These patterns ensure your extension behaves more like a cohesive application rather than isolated scripts.



## Error Handling and User Experience



In a production environment, robustness and user experience are paramount. Your scripts should handle errors gracefully and give feedback to the user, rather than just throwing tracebacks. Let's break down best practices for error handling, progress indication, and graceful degradation of features:



**Exception handling and logging:** Wrap your scripts in try/except blocks where appropriate to catch exceptions and prevent crashes from propagating to Revit. PyRevit will catch unhandled exceptions and print them to its output window, but it's better to manage them yourself:



* **User-friendly messages:** If a foreseeable error occurs (e.g., no elements selected, a required file missing, a server not reachable), catch it and inform the user in a friendly way. The `pyrevit.forms.alert()` function is very handy for this - it pops up a modal dialog with a message. You can use `forms.alert(msg, sub_msg="Details...", warn_icon=True)` for warnings or errors. For example, in a script if the user hasn't configured a setting, you might do:



  ```python

  if not config.get("server_url"):

      forms.alert("Server is not configured", sub_msg="Please set the server URL in settings before running this tool.", warn_icon=True, exitscript=True)

  ```



  The `exitscript=True` parameter will stop the script execution after the user closes the alert. This is cleaner than letting the script run into a None attribute and throw an exception. Reserve detailed technical messages for logs; keep user-facing alerts concise and actionable.



* **Use logging for diagnostics:** PyRevit provides a logger object you can retrieve with `script.get_logger()`. This logger is configured to write to the PyRevit output panel and also to the PyRevit log files on disk. In enterprise usage, you should log important events and errors. For example, wrap your main logic in a try/except and in the except block do:



  ```python

  import traceback

  logger = script.get_logger()

  logger.error("Failed to process rooms: {}".format(traceback.format_exc()))

  ```



  This will record a stack trace in the log file which you can ask users to send, or aggregate later. You can also use `logger.warning`, `logger.info` for less severe messages. Having logs is crucial when troubleshooting issues on a user's machine where you can't easily step through the code. In addition, consider logging significant actions (e.g., "20 doors updated by user X") to a central file or database if your security policy allows - this can be part of a **telemetry** solution (more on that in Deployment & Monitoring).



* **Fail safe:** If an error is non-recoverable, ensure your script exits cleanly. You can call `script.exit()` (which is basically raising a special exception that PyRevit catches) or simply ensure no further Revit API calls happen after an error. Always roll back open transactions if you're in the middle of one when an error hits, otherwise Revit might leave a transaction open. A pattern is:



  ```python

  t = Transaction(doc, "MyTool")

  try:

      t.Start()

      # ... do stuff

      t.Commit()

  except Exception as err:

      logger.error("Transaction failed: {}".format(err))

      if t.HasStarted():

          t.RollBack()

      forms.alert("Something went wrong. No changes were saved.", sub_msg=str(err), warn_icon=True)

  ```



  This ensures that if anything goes wrong after the transaction started, we rollback and inform the user that nothing was changed. **Never leave a transaction open** - it can lock the model.



**Progress feedback for long operations:** Enterprise tools often need to process hundreds or thousands of elements, or perform complex calculations, which can take several seconds or even minutes. During that time, Revit's UI is basically locked unless you consciously give feedback or yield control. PyRevit provides a couple of ways to keep users informed and Revit responsive:



* **Status messages and progress bar:** Use the built-in progress bar at the top of the Revit window. The `pyrevit.forms.ProgressBar` context manager is straightforward. Example:



  ```python

  total = len(elements)

  with forms.ProgressBar(title="Processing elements", step=10) as pb:  # update every 10%

      for i, elem in enumerate(elements, start=1):

          # ... process elem ...

          pb.update_progress(i, total)

          if pb.cancelled:  # if user clicked cancel

              break

  ```



  In this snippet, `step=10` means the bar will visually update only every 10 steps (10% increments), which is more efficient for large loops. If you set `cancellable=True`, a Cancel button appears; check `pb.cancelled` to see if the user pressed it and gracefully abort if so. Even if you can't safely stop mid-way, at least you can inform the user "Cancelling... please wait" and then exit when possible. For unknown-length tasks (e.g., waiting on an external service), you can use `indeterminate=True` to show a marquee progress bar.



* **Output window messages:** The PyRevit output window (console) is a useful place to print status updates. You can do `print("Processing element {} of {}".format(i, total))` and the messages will appear in real-time. If you want richer formatting (colors, bold text), use the `pyrevit.output` module's methods. For example, `output = script.get_output(); output.print_html("<p><strong>Finished!</strong></p>")` will format HTML in the output. This is good for final summaries or reports. However, for real-time progress, the ProgressBar is more visible to the user.



* **Don't block Revit unnecessarily:** If a task is extremely long, consider splitting it into chunks and using `yield` or even a modeless approach. While PyRevit doesn't support true multi-threading for Revit API calls (Revit API must run on the main thread), you could break work into smaller transactions or use the Idling event to do work in slices, allowing Revit to process events in between. This is an advanced technique - in most cases, a ProgressBar and careful coding (avoiding super heavy operations in a single loop) will suffice. The key is the user shouldn't wonder if the tool is "hung". Even a simple running counter or message like "Processed 100 of 5000 elements..." in the output can reassure them.



**Graceful degradation and feature toggles:** In an enterprise setting, your extension might run in varied environments. Some users might not have the same external tools installed, or you might support multiple Revit versions. Plan for this:



* **Version checks:** If your extension supports Revit 2019-2023, and you use an API call introduced in 2021, protect it with a condition. You can get the Revit version from `HOST_APP.version` (pyRevit provides this) or by checking `doc.Application.VersionNumber`. Then, either disable certain functionality or use alternative code for older versions. It's good to document in your extension which Revit versions are supported and test accordingly.

* **Optional dependencies:** If a feature relies on something like an external database or a CPython library, check for its availability at runtime. For example, try importing the library in a try/except and if it fails, disable the feature (perhaps the button's `__selfinit__` returns False so it doesn't appear). You could also catch the error when the user clicks and show a message: "This tool requires XYZ. Please install it or contact BIM support." This is much better than a cryptic ImportError. In PyRevit, using a **smart button bundle** (`.smartbutton`) is an elegant way to handle this at UI load time. The `__selfinit__` function can perform checks and return False to **hide the button entirely** if prerequisites aren't met. For example, a smart button could check for a specific DLL or an environment variable and disable itself if not present - preventing user confusion.

* **Fallback workflows:** Provide alternatives if possible. If your extension normally uses a web API but the user is offline, maybe allow an offline mode that uses cached data or asks them to connect VPN. Or if a feature fails, ensure it fails without side effects. Perhaps log the error and notify the user "X feature is not available, continuing without it." This way other parts of the script can still run.

* **UI/UX niceties:** Small touches improve user experience. Use **pyRevit forms** for input prompts (instead of raw `input()` calls). The `forms.CommandSwitchWindow` or `forms.SelectFromList` provide user-friendly selection dialogs. For example, instead of asking the user to type a category name, present a checklist of categories. Also leverage `forms.WarningBar` for minor warnings (it shows a yellow banner in Revit). These make the tools feel like integrated parts of Revit. Always consider international users or standards - if your company spans regions, ensure text is clear (PyRevit supports localization, though that may be beyond current scope).



Finally, **test error scenarios** as part of development. Deliberately break connections or provide bad input to see how your script behaves. It's much better that you catch an unhandled exception during testing than a user seeing a big red traceback during a critical production deadline. By handling errors and providing feedback, you build trust with users that your tools won't disrupt their work.



## Deployment and Distribution in Enterprise



Developing a powerful extension is only half the battle - you also need to **deploy and manage it across potentially hundreds of user machines**. Enterprise deployment should ensure everyone has the latest version, the right dependencies, and a way to receive updates and send feedback (like error reports). Here are best practices for distributing PyRevit extensions in a corporate environment:



**Extension packaging and structure:** Treat your extension as a standalone software product. Keep it in source control (e.g., a Git repository) and use a consistent versioning scheme. Include an `extension.json` or manifest with the extension's name, author, and version. This metadata can be displayed in PyRevit's extension manager and is useful for troubleshooting ("Are you on version 1.2.0 or 1.3.0?"). Some teams even include a "Check for Updates" or "About" button in the extension UI that shows the current version and perhaps a changelog.



**Deployment methods:** There are a few common ways to deploy PyRevit extensions, each with pros/cons:



* **Shared Network Drive:** A straightforward approach is to place the extension folder on a network file server that all users can access (read-only is fine). Then you configure PyRevit on each machine to look at that network path (via **pyRevit Settings > Custom Extensions** or using the `pyrevit configs` CLI). Once set up, this means you maintain a **single copy** of the extension - users automatically run whatever is on the server. The benefit is instant updates: if you update a script or fix a bug on the server, everyone gets it the next time they restart Revit or hit "Reload". Nobody is stuck on an old version and you don't have to touch individual PCs beyond the initial setup. This method is great for a controlled environment with a reliable network. However, as the forum discussions note, performance depends on network speed - a slow network can become a bottleneck at Revit startup. Also, if users travel or go offline (off VPN), they might not have access to the network drive, disabling the tools. If using this method, ensure the network is highly available or advise users accordingly. It's also wise to script the initial setup (e.g., a one-time `.bat` file that adds the network path to pyRevit for the user) to save IT time.



* **Git repository (distributed clone):** A more scalable approach is to use Git for distribution. In this model, you host the extension in a Git repository (GitHub, Bitbucket, Azure DevOps, etc.). Each user will **clone** the repo to their local machine, and pyRevit will load the extension from that local clone. PyRevit's CLI has commands to assist: for example, you can run `pyrevit extend ui AwesomeTools "https://github.com/YourCompany/AwesomeTools.extension.git" --branch=main` to tell pyRevit to clone the repo and load it as an extension. This can be done by the user or automated via a script. The beauty of this approach is that each user's extension is local (so no network lag at runtime), but updating is centralized: when you push a new version to the Git repo, users can pull the updates. You can automate this pull. One common pattern is to include a **startup hook or script** that runs `pyrevit extensions update <YourExtensionName>` on Revit launch. With a Git host that doesn't require credentials (or using a read-only access token embedded in the clone URL), this can happen silently. In practice, the first user to open Revit in the morning triggers the update and gets the latest code; others who open later also get up-to-date code (since the repo was updated on disk). This addresses the network issues (since only a diff is pulled, and only at startup) and allows working offline (the extension will still load, just won't update until online). Another advantage is version control: you have a full history of changes and can use branches or tags to manage releases. For instance, you might have a stable "release" branch that everyone's pointed to, while you develop on "dev" branch; when ready, merge to release and users auto-update. If something goes wrong, you can quickly revert the commit and instruct users to update (or they'll auto-update next launch). Some organizations even integrate this with CI/CD: when you merge to main, an automated pipeline could run tests and then push the new extension version out.



* **Local installation (manual or automated):** The least fancy method is manually copying the extension to each user's machine (e.g., via an SCCM package or simply emailing a zip and instructions). This is generally not ideal for ongoing maintenance because it's hard to enforce updates. However, you might use it in addition to the above methods for certain situations (for example, consultants outside your network - you give them a static copy of the extension). If you do this, make sure to include documentation on where to place the folder and how to add it to pyRevit, and clearly version it. Without an update mechanism, you'll likely need to redistribute the package for every update, which can become a nightmare in a large firm. Therefore, consider this only as a last resort or for one-off deployments. If you find yourself going this route, it might be better to use the pyRevit CLI to create an installer or at least an easy script for users to run.



**Version management:** Use version numbers and communicate changes. It's good practice to embed a version in your extension (in code or extension.json) and possibly display it. Some teams put the version in each tool's tooltip or in an "About" dialog. Keep a CHANGELOG file or similar. This helps both users and developers know what's currently deployed. For example, if a user reports a bug, you can ask "What version are you on?" and they can check a tooltip or about window that says "v2.5.0". As part of release, update the version and list changes (even briefly). Users appreciate knowing what's new or fixed, and it sets expectation that the extension is maintained. In an enterprise, where multiple people might be contributing to the extension, version control also helps coordinate (avoid two devs both calling theirs "1.0"). Semantic versioning (MAJOR.MINOR.PATCH) works well (e.g., increment minor for new features, patch for small fixes). Document compatibility too: if version 2.0 drops support for Revit 2019, note that in the release notes or in documentation.



**Testing and CI:** Before deployment, test your extension in all target environments. This means checking it on each Revit version you support (Revit 2019, 2020, 2021, etc.) - APIs can change, so something might break on 2019 that works on 2023. Also test with different user privileges if applicable (some environments lock down file access, etc.). If possible, automate some tests. You can write unit tests for your pure-python logic (outside Revit) and run them with CI (GitHub Actions or others) whenever you push changes. While you can't easily run Revit in CI to test the actual Revit API calls, you can structure your code to have as much logic as possible in plain Python that can be tested in isolation. At minimum, always run a quick sanity test of key tools in Revit before rolling out an update - perhaps have a small pilot group or use a separate "beta" extension that you load for testing new features, then merge into main extension for everyone. Enterprises often adopt a ring deployment: BIM team tests first, then power users, then all users.



**Configuration across user environments:** Your extension might need certain environment-specific values, such as:



* Server URLs or file paths (for databases, CSV repositories, etc.)

* User-specific preferences

* Feature toggles (maybe some users opt into experimental features)



Manage these via configuration files or user settings, not by modifying code. One approach is to have a **settings JSON/INI** within the extension that an admin can edit. For example, `extension.json` can be used not only for metadata but also to store some config (since you can read it as a file in your script). Or have a separate `config.yaml` that ships with the extension - in your startup, read it and apply settings. For things each user might customize (like "don't show this warning again"), use pyRevit's `script.get_config()` as mentioned to persist that per user. You can combine these: e.g., an enterprise config defines defaults, and if a user has an override in their user config, use that instead. If you have many settings, consider building a small UI for them (a "Settings" pushbutton in a panel that opens a form where users can input their preferences, which you then save via set\_config). This avoids requiring users to hand-edit files.



For multi-location firms, you might even use environment variables or network lookups - e.g., detect office location and choose a nearest server. These kinds of configurations can greatly improve performance (pointing each user to local data sources) but require planning. The main goal is to **avoid hard-coded values** in your scripts. Keep anything that might change in a config file or at least a clearly marked section.



**Centralized logging and monitoring:** To manage an extension at enterprise scale, it helps to have insight into its usage and issues:



* PyRevit has a built-in **telemetry system** that can log usage data to a central server. If enabled, it can record every time a user runs a tool, along with timing and (optionally) results or errors. You would set up a small server (PyRevit provides a Go-based server that logs to MongoDB/PostgreSQL). This might be overkill for some, but if your management is interested in usage metrics ("which tools are actually being used?") or you want to see if a particular user is encountering errors frequently, telemetry is valuable. It's opt-in, so you'd have to configure it and possibly inform users if required by company policy.

* Even without the full telemetry, you can implement lightweight logging. For example, have your scripts write to a network share log file when certain important things happen (ensure you handle file locks or use a simple approach like each event is one line appended to a central log via a small REST API or database insert). Even a daily log per user that they can send in helps. One creative solution is to use the company's logging infrastructure if one exists (some IT setups have central event log collection - you could emit events that are caught there).

* **Error reporting:** You don't want errors to silently fester. If a user hits an exception and doesn't tell the BIM team, it might never get fixed. Encourage reporting by making it easy: for instance, in an error alert, you could include a note "An error occurred. A log file has been saved to X. Please send this to the BIM support team." Or even automate it: if policy allows, catch exceptions and send an HTTP request to a logging server with the error info. Always respect privacy and performance (don't spam or send huge data), but a one-line error summary could be invaluable. Some teams integrate with issue trackers: e.g., an error could create a ticket automatically (again, only if approved by management).

* **User feedback:** Beyond error logs, consider adding a mechanism for users to easily provide feedback - even a "Send Feedback" button that opens an email or Teams message pre-filled with extension info can encourage communication. This isn't architecture per se, but it closes the loop between development and usage.



**Maintainability:** With multiple developers, enforce a structure (which we've outlined) so that everyone puts things in the right place. Use source control branching and pull requests to review changes. Possibly write documentation (even a simple README) for your extension so future team members understand the architecture. Over time, as the extension grows, periodically refactor: if a panel is too crowded, split it; if scripts get too long, break out library functions; if performance is lagging, revisit the code with fresh eyes or new PyRevit features (for example, PyRevit 5.X might introduce new APIs or IronPython 3 upgrade, etc.).



In conclusion, **enterprise deployment of PyRevit tools** benefits from automation and foresight: automate installation (via network or Git), automate updates (so users always have the latest), and automate monitoring (so you know how it's doing). Coupled with the robust architecture patterns - structured extension bundles, optimized script performance, inter-script synergy, and solid error handling - your PyRevit extension will be a sustainable, high-impact solution for your BIM workflows for years to come. By following these practices, you ensure that the extension remains **performant, maintainable, and user-friendly** even as it scales up in complexity and is adopted across large project teams.



**Sources:**



1. E. Iran-Nejad, *pyRevit Documentation*: *Extensions and Commands* - Guidelines on extension folder structure and bundle types.

2. B. Bakerman, *ArchiLabs Blog* - "How to Share pyRevit Plugins" - Best practices for packaging extensions and distributing via network or Git.

3. PyRevit Forums - Discussions on enterprise deployment (network vs. git) and performance optimization (IronPython memory, using Autodesk API directly).

4. PyRevit Documentation - *Effective Output/Input* - usage of ProgressBar and other UI feedback tools.

5. I. Krachkovskii, *Medium* - "pyRevit and event hooks" - Example of using a `doc-opened` hook script.

6. PyRevit Forums - Guidance on `get_config()` / `set_config()` for storing user settings in pyRevit config and using extensible storage for project data.

7. PyRevit GitHub Issue - CPython integration: limitations and subprocess workaround.

8. PyRevit Documentation - *Understanding pyRevit Architecture* - Overview of pyRevit components and telemetry capabilities.



Return back: [`pyrevit_overview.md`](pyrevit_overview.md)
Return to root: [`README.md`](../../../../README.md)
