---
name: cetus-dlmm-interface
description: |
  Enables Claude to effectively work with the Cetus Dynamic Liquidity Market Maker (DLMM) protocol on Sui blockchain. This skill provides comprehensive knowledge of the DLMM architecture, Move smart contracts, Rust SDK, and development workflows. Claude can assist with building, testing, and modifying DLMM contracts, creating swap simulations, implementing new features, and debugging protocol issues. The skill covers all 15 Move modules, multi-bin architecture, reward distribution, ACL system, and flash swap functionality.

  Trigger contexts include: working with Sui Move smart contracts for decentralized exchanges, implementing dynamic liquidity management, creating multi-bin AMM protocols, building swap simulations with Rust SDK, modifying DLMM protocol parameters, debugging pool operations, implementing reward systems, and working with Cetus DLMM interface codebase.
context: |
  This skill is for the Cetus DLMM protocol - a sophisticated Dynamic Liquidity Market Maker on Sui.
  Key characteristics:
  - 15 Move modules in packages/dlmm/sources/
  - Multi-bin architecture (up to 1000 bins per position)
  - 5 reward types per position
  - Comprehensive ACL and restriction system
  - Flash swap support
  - Rust SDK for off-chain calculations
  - Move 2024 edition with modern best practices
---

# Cetus DLMM Interface Skill

## Overview

The Cetus Dynamic Liquidity Market Maker (DLMM) is a sophisticated automated market maker protocol on the Sui blockchain. This skill enables Claude to effectively work with all aspects of the DLMM protocol, including Move smart contracts, Rust SDK, development workflows, and integration patterns.

**Key Protocol Features:**
- **Multi-bin Architecture**: Supports up to 1000 bins per position for granular liquidity management
- **Dynamic Liquidity**: Automatic rebalancing across price ranges
- **Reward Distribution**: Up to 5 different reward types per position
- **Access Control**: Comprehensive ACL system with user/position restrictions
- **Flash Swaps**: Support for flash loan functionality
- **Version Management**: Upgradeable protocol with version tracking

**Project Structure:**
- `packages/dlmm/sources/` - 15 Move smart contract modules
- `sdk/swap-sdk/` - Rust SDK for off-chain swap calculations
- `CLAUDE.md` - Comprehensive Move development guidelines

## Development Environment

### Prerequisites
- **Sui CLI**: For building and testing Move packages
- **Rust**: For SDK development and testing
- **Bun**: For running prettier-move formatter
- **Git**: For dependency management

### Installation

#### Dependencies

Add the following dependencies to your `Move.toml`:

```toml
[dependencies]
CetusDlmm = { git = "https://github.com/CetusProtocol/cetus-dlmm-interface.git", subdir = "packages/dlmm", rev = "mainnet-v0.5.0", override = true }
IntegerMate = { git = "https://github.com/CetusProtocol/integer-mate.git", rev = "mainnet-v1.3.0", override = true }
MoveSTL = { git = "https://github.com/CetusProtocol/move-stl.git", rev = "mainnet-v1.3.0", override = true }
```

#### Address Configuration

Configure the package address in your `Move.toml`:

```toml
[addresses]
cetusdlmm = "0x0"  # Replace with actual deployed address
```

### Project Structure
```
cetus-dlmm-interface/
├── packages/dlmm/              # Move smart contracts
│   ├── Move.toml              # Package manifest
│   ├── Move.lock              # Dependency lock file
│   ├── README.md              # Package documentation
│   └── sources/               # 15 Move modules
└── sdk/swap-sdk/              # Rust SDK
    ├── Cargo.toml            # Rust package manifest
    ├── Cargo.lock            # Rust dependency lock
    ├── README.md             # SDK documentation
    └── src/                  # SDK source code
```

### Essential Commands
```bash
# Build Move package
cd packages/dlmm
sui move build --skip-fetch-latest-git-deps

# Run Move tests
sui move test

# Format Move code
bunx prettier-move -c *.move --write

# Build and test Rust SDK
cd sdk/swap-sdk
cargo build
cargo test
```

## Core Components

### Move Smart Contracts
The DLMM protocol consists of 15 interconnected Move modules. For detailed documentation on each module, see [Move Modules Reference](references/move-modules.md).

**Primary Modules:**
- **pool.move** (66KB): Core swap and liquidity operations
- **position.move**: LP position management with fee collection
- **bin.move**: Bin system for organizing liquidity across price ranges
- **registry.move**: Global pool creation and tracking
- **config.move**: Protocol-wide configuration and access control
- **reward.move**: Reward distribution mechanism
- **dlmm_math.move**: Mathematical utilities for DLMM calculations

### Rust SDK
The standalone Rust SDK provides accurate swap simulations and price impact calculations without blockchain dependencies. See [SDK API Reference](references/sdk-api.md) for complete documentation.

**Key Components:**
- **Swap Orchestration**: Multi-bin traversal for exact-input/exact-output swaps
- **Mathematical Utilities**: Q64x64 fixed-point arithmetic for precision
- **Configuration Structures**: Pool parameters and bin step configurations

### Protocol Constants (Implemented as Macros)
- **Maximum bins per position**: 1000 (`max_bin_per_position()`)
- **Maximum reward types per position**: 5 (`max_reward_num()`)
- **Fee precision**: 1,000,000,000 (1e9) (`fee_precision()`)
- **Maximum fee rate**: 100,000,000 (10%) (`max_fee_rate()`)
- **Maximum protocol fee rate**: 300,000,000 (30%) (`max_protocol_fee_rate()`)
- **Reward period**: 7 days (604,800 seconds) (`reward_period()`)

Constants are defined as macros in `constants.move`.

## Common Workflows

### Creating a New Pool
1. Define token pair and fee parameters
2. Configure bin steps and active bin range
3. Call `registry::create_pool_v3` with appropriate parameters
4. Verify pool creation and initial configuration

**Reference**: See [Move Modules Reference](references/move-modules.md) for module details.

### Adding and Managing Liquidity
1. Create position with specified bin ranges and amounts
2. Add liquidity to existing positions
3. Remove liquidity with fee collection
4. Monitor position performance and rewards

### Executing Swaps
1. Calculate swap amounts using SDK for simulation
2. Construct swap transaction with appropriate parameters
3. Handle exact-input vs exact-output swaps
4. Process flash swaps if enabled

**Reference**: See [SDK API Reference](references/sdk-api.md) for swap simulation details.

### Managing Rewards and Fees
1. Configure reward types and distribution schedules
2. Set partner fee structures
3. Collect accumulated fees from positions
4. Monitor reward distribution across positions

## Integration Patterns

### With Existing Sui Skills
This skill works with other Sui ecosystem skills:

1. **sui-transaction-building**: For constructing complex DLMM transactions
2. **sui-bcs**: For serializing pool parameters and swap data
3. **sui-client**: For reading on-chain pool state and positions
4. **move**: General Move language assistance

See [Integration Patterns Reference](references/integration-patterns.md) for detailed examples.

### Cross-Module Dependencies
Understanding module relationships is crucial:
- **pool** depends on: bin, position, config, reward, parameters
- **position** depends on: bin, reward
- **registry** depends on: pool, config, parameters
- **config** provides global settings used by all modules

## Best Practices

### Code Quality Checklist
Follow the comprehensive guidelines in `CLAUDE.md`. Key highlights:

**Move 2024 Edition Patterns:**
- Use module label syntax: `module my_package::my_module;`
- Group `use` statements with `Self`: `use my_package::my_module::{Self, OtherMember};`
- Error constants in `EPascalCase`, regular constants in `ALL_CAPS`
- Capabilities should be suffixed with `Cap`: `AdminCap`
- Events should be named in past tense: `UserRegistered`

**Function Design:**
- Objects go first (except for Clock), capabilities second
- Write composable functions for PTBs (Programmable Transaction Blocks)
- Use struct methods over free functions where available
- Implement proper error handling with module-specific error codes

**Development Workflow:**
1. Make changes to Move code
2. Build with `sui move build --skip-fetch-latest-git-deps`
3. Run tests with `sui move test`
4. Format with `bunx prettier-move -c *.move --write`
5. Verify no compilation errors or test failures

### Error Handling
- Each module defines its own error codes
- Use descriptive error messages for debugging
- Test error conditions thoroughly
- Consider gas implications of error checks

See [Error Codes Reference](references/error-codes.md) for complete error documentation.

## Troubleshooting

### Common Issues

**Build Failures:**
- Check dependencies in `Move.toml`
- Verify named addresses are correctly configured
- Ensure all imports are valid and accessible

**Test Failures:**
- Review test scenarios and expected outcomes
- Check for integer overflow/underflow conditions
- Verify mock data matches expected formats

**Swap Calculation Discrepancies:**
- Compare SDK simulations with on-chain results
- Verify bin configuration and active ranges
- Check fee calculations and rounding behavior

### Debugging Strategies
1. **Isolate the Issue**: Determine if problem is in Move contracts or SDK
2. **Check Logs**: Review transaction events and error messages
3. **Simulate First**: Use SDK to simulate before on-chain execution
4. **Test Incrementally**: Make small changes and test each step

### Performance Considerations
- **Gas Optimization**: Minimize storage operations and complex calculations
- **Batch Operations**: Use PTBs for multiple related operations
- **Caching**: Consider off-chain caching for frequently accessed data

## Related Skills

For comprehensive Sui development, also consider:
- **sui-transaction-building**: Building complex blockchain transactions
- **sui-bcs**: Binary Canonical Serialization for Sui objects
- **sui-client**: Interacting with Sui blockchain RPC
- **move**: General Move language syntax and patterns

## References

### Detailed Documentation
- [Move Modules Reference](references/move-modules.md) - Complete documentation of all 15 Move modules
- [SDK API Reference](references/sdk-api.md) - Rust SDK structures and functions
- [Development Workflows](references/development-workflows.md) - Build, test, and deployment patterns
- [Error Codes Reference](references/error-codes.md) - Comprehensive error documentation
- [Integration Patterns](references/integration-patterns.md) - Working with other Sui skills

### Project Resources
- **Main Repository**: `packages/dlmm/sources/` - Move smart contracts
- **SDK**: `sdk/swap-sdk/` - Rust SDK for off-chain calculations
- **CLAUDE.md**: Project-specific Move development guidelines
- **README.md**: Project overview and setup instructions

### External Resources
- [Sui Documentation](https://docs.sui.io/) - Official Sui blockchain docs
- [Move Language](https://move-language.github.io/move/) - Move programming language
- [Cetus Protocol](https://cetus.xyz/) - Cetus protocol documentation

---

*This skill is designed to work with Claude Code to provide comprehensive assistance for Cetus DLMM protocol development. Always verify changes with appropriate testing before deployment.*