# Integration Patterns

Based on the Cetus DLMM protocol architecture and actual usage patterns.

## Protocol Architecture Integration

### Module Dependencies

The DLMM protocol modules have specific dependency relationships:

1. **Registry creates Pools** - `registry.move` → `pool.move`
2. **Pools manage Positions and Bins** - `pool.move` → `position.move`, `bin.move`
3. **Configuration controls access** - `config.move` used by all modules
4. **Mathematical utilities support calculations** - `dlmm_math.move`, `price_math.move`

### Common Integration Points

1. **Pool Creation**: Use `registry::create_pool_v3` to create new pools
2. **Swap Execution**: Use `pool::swap_exact_amount_in` or `pool::swap_exact_amount_out`
3. **Liquidity Management**: Use `position::` functions for position operations
4. **Fee Collection**: Use `pool::collect_fee` and `position::collect_fee`

## SDK Integration

### Off-Chain Simulation
```rust
// Use the Rust SDK for swap simulations
use swap_sdk::{Pool, Bin, VariableParameters};

let mut pool = Pool::new(active_id, base_fee_rate, v_parameters, bins);
let result = pool.swap_exact_amount_in(amount_in, a2b, timestamp)?;
```

### Integration with On-Chain Data
1. Fetch pool data from Sui blockchain
2. Convert to SDK `Pool` structure
3. Simulate swaps off-chain
4. Execute verified swaps on-chain

## Development Integration

### With Sui CLI
- Use `sui client call` for interacting with deployed contracts
- Use `sui move build` and `sui move test` for development
- Follow patterns in official documentation

### Code Quality Tools
- Move: `bunx prettier-move` for formatting
- Rust: `cargo fmt`, `cargo clippy` for code quality
- Testing: `sui move test`, `cargo test`

## Source Code Reference

For actual integration patterns, refer to:
- Move module source code in `packages/dlmm/sources/`
- SDK source code in `sdk/swap-sdk/src/`
- Project examples and tests
- Official Cetus documentation