# Move Modules Reference

Based on the official Cetus DLMM README and actual source code in `packages/dlmm/sources/`.

## Core Modules

The Cetus DLMM protocol consists of 15 Move modules:

1. **`pool.move`** - Main pool functionality handling swaps, liquidity management, and position operations
2. **`position.move`** - Position management for liquidity providers with fee collection and reward distribution
3. **`bin.move`** - Bin management system for organizing liquidity across price ranges
4. **`registry.move`** - Global pool registry for creating and tracking all pools
5. **`config.move`** - Global configuration management with access control and restrictions
6. **`reward.move`** - Reward distribution and management system
7. **`partner.move`** - Partner fee management and tracking
8. **`restriction.move`** - User and position blocking with operation-level restrictions
9. **`acl.move`** - Access Control List implementation for permission management
10. **`admin_cap.move`** - Administrative capabilities and permissions
11. **`parameters.move`** - Variable parameter management for pools
12. **`dlmm_math.move`** - Mathematical utilities for price calculations and liquidity management
13. **`price_math.move`** - Price calculation utilities
14. **`versioned.move`** - Version management and upgrade mechanisms
15. **`constants.move`** - Protocol constants and configuration values (implemented as macros)

## Position Module Details

The `position.move` module manages liquidity positions in the DLMM protocol. Each position can span multiple price bins (up to 1000 bins) and tracks liquidity shares, fees, and rewards.

### Position Structure (`position.move:107-120`)

```move
public struct Position has key, store {
    id: UID,                    // Unique identifier
    pool_id: ID,                // Pool ID this position belongs to
    index: u64,                 // Position index within the pool
    coin_type_a: String,        // Token A type name
    coin_type_b: String,        // Token B type name
    description: String,        // Description
    name: String,               // Display name
    uri: String,                // Metadata URI
    lower_bin_id: I32,          // Lower bound bin ID
    upper_bin_id: I32,          // Upper bound bin ID
    liquidity_shares: vector<u128>, // Liquidity shares for each bin
    flash_count: u64,           // Number of flash swaps
}
```

### PositionInfo Structure (`position.move:133-139`)

```move
public struct PositionInfo has copy, store {
    id: ID,                     // Position ID
    fee_owned_a: u64,           // Accumulated token A fees
    fee_owned_b: u64,           // Accumulated token B fees
    rewards_owned: vector<u64>, // Accumulated rewards for each reward type
    stats: vector<BinStat>,     // Statistics for each bin in the position
}
```

### BinStat Structure (`position.move:152-158`)

```move
public struct BinStat has copy, store {
    bin_id: I32,                // Bin ID
    liquidity_share: u128,      // Current liquidity share in this bin
    fee_a_growth: u128,         // Fee growth for token A at last update
    fee_b_growth: u128,         // Fee growth for token B at last update
    rewards_growth: vector<u128>, // Reward growth for each reward type
}
```

### Reading Position Properties

#### Basic Property Getters (`position.move:318-440`)

```move
// Get position width (number of bins)
public fun width(position: &Position): u16

// Get pool ID
public fun pool_id(position: &Position): ID

// Get position index
public fun index(position: &Position): u64

// Get position name
public fun name(position: &Position): String

// Get metadata URI
public fun uri(position: &Position): String

// Get token A type
public fun coin_type_a(position: &Position): String

// Get token B type
public fun coin_type_b(position: &Position): String

// Get liquidity shares vector
public fun liquidity_shares(position: &Position): vector<u128>

// Get position description
public fun description(position: &Position): String

// Get lower bin ID
public fun lower_bin_id(position: &Position): I32

// Get upper bin ID
public fun upper_bin_id(position: &Position): I32

// Get bin index within position
public fun bin_idx(position: &Position, bin_id: I32): u64
```

#### Fee and Reward Getters (`position.move:245-276`)

```move
// Get accumulated fees (fee_a, fee_b)
public fun info_fees(position_info: &PositionInfo): (u64, u64)

// Get rewards vector
public fun info_rewards(position_info: &PositionInfo): &vector<u64>

// Get bin statistics
public fun info_stats(position_info: &PositionInfo): &vector<BinStat>
```

#### Position Amount and Detail Functions (`pool.move`)

```move
// Get position token amounts (token A, token B)
public fun get_position_amounts<CoinTypeA, CoinTypeB>(
    pool: &Pool<CoinTypeA, CoinTypeB>,
    position_id: ID,
): (u64, u64)

// Refresh and get complete position details
public fun refresh_position_info<CoinTypeA, CoinTypeB>(
    pool: &mut Pool<CoinTypeA, CoinTypeB>,
    position_id: ID,
    versioned: &Versioned,
    clk: &Clock,
    ctx: &TxContext,
): PositionDetail

// V2 version with expected active ID parameter
public fun refresh_position_info_v2<CoinTypeA, CoinTypeB>(
    pool: &mut Pool<CoinTypeA, CoinTypeB>,
    position_id: ID,
    mut expected_active_id: Option<I32>,
    versioned: &Versioned,
    clk: &Clock,
    ctx: &TxContext,
): PositionDetail
```

### PositionDetail Structure

```move
public struct PositionDetail has copy, drop {
    position_id: ID,            // Position ID
    amount_a: u64,              // Token A amount
    amount_b: u64,              // Token B amount
    fee_a: u64,                 // Token A fees
    fee_b: u64,                 // Token B fees
    rewards: VecMap<TypeName, u64>, // Rewards
    update_tx: vector<u8>,      // Update transaction
}
```

### Usage Patterns

1. **Read Basic Properties**: Use Position getter functions like `pool_id(position)`, `name(position)`
2. **Read Fees and Rewards**: Access via `PositionInfo` using `info_fees()`, `info_rewards()`
3. **Get Token Amounts**: Call `get_position_amounts()` function
4. **Get Complete Details**: Use `refresh_position_info()` or `refresh_position_info_v2()` for `PositionDetail`
5. **Bin-level Information**: Read from `BinStat` array for individual price range details

DLMM supports multi-bin positions (up to 1000 bins) with granular liquidity management across price ranges.

## Key Notes

- **Move 2024 Edition**: The codebase uses Move 2024 edition features
- **Constants as Macros**: Protocol constants are defined as macros (e.g., `fee_precision()`, `max_fee_rate()`)
- **Error Handling**: Error constants follow `EPascalCase` naming convention
- **Module Structure**: Uses modern module label syntax

## Source Code Location

All Move modules are located in `packages/dlmm/sources/`.

For the most accurate and up-to-date information, always refer to the actual source files.