---
name: backoffice-fullstack
description: Fullstack development skill for WL Backoffice system. Handles feature implementation across Kirby (FE), Coloris (BE API), Promodia (Promotion), and Monika (SQL Database). Use when users want to "create new feature", "implement backend API", "create frontend component", "add new stored procedure", "create database table", or any fullstack development task for the backoffice system.
metadata:
  version: 3.2.0
  last_updated: 2025-01-08
  author: Claude
  tags: [fullstack, kirby, coloris, monika, promodia, backoffice, vue, csharp, sql]
---

# Backoffice Fullstack Development Skill

## ⚠️ FIRST: Load Memory

Before starting any task, read memory files to restore context:

```
.memory/learnings.json  ← Patterns, conventions, dev settings
.memory/tables.md       ← Table structures discovered
.memory/domain/         ← Feature-specific knowledge
```

**Development Settings (from memory):**
- WebId = 1 (1company)
- Check tables in Monika project files, never SELECT *
- Use `/artemis-debug-secure` for STG queries

---

A comprehensive skill for developing features across the WL Backoffice system stack.

## Performance

| Layer | Sequential | Parallel (with sub-agents) | Improvement |
|-------|------------|---------------------------|-------------|
| Full feature | 30-45 min | 10-15 min | **3x faster** |
| FE only | 15-20 min | 5-10 min | **2x faster** |
| BE only | 10-15 min | 5-8 min | **2x faster** |

## Features

| Feature | Description |
|---------|-------------|
| Multi-Layer Stack | Monika (DB) → Coloris (API) → Kirby (FE) |
| Parallel Agents | FE and BE can run simultaneously after DB ready |
| Pattern Learning | Follows CustomerRecordReport flow (FE) |
| SP Versioning | Semantic versioning: `Feature_Action_1.0.0` |
| Domain Knowledge | 12 backoffice modules documented |
| **v3.0.0** Parallel Execution | BE + FE run simultaneously after DB |
| **v3.0.0** Pattern Detection | 8 auto-detected patterns with violation checking |
| **v3.1.0** FE Testing | Playwright tests with screenshot proof |
| **v3.2.0** Programmatic Pattern Validator | Python scripts for code pattern validation |
| **v3.2.0** Full Automation Suite | Stub generator, mock server, feature scaffolding |
| **v3.2.0** Expanded Templates | 22 templates (FE: 7, BE: 5, DB: 9, Contract: 1) |

---

## v3.0.0 Features

### Multi-Agent Parallel Execution

3x faster development by running agents in parallel:

```
Phase 1: DB (Sequential)     → 5-8 min
Phase 2: BE + FE (Parallel)  → 5-8 min (instead of 20-25 min)
Phase 3: FE Integration      → 2-4 min
Total: 12-20 min (was 30-45 min)
```

**Files:**
- `agents/parallel-execution-guide.md` - Full documentation
- `prompts/agent-dependencies.json` - Task dependency graph
- `prompts/batch-spawn-templates.md` - Ready-to-use spawn prompts

### Automatic Pattern Detection

8 core patterns automatically detected and validated:

| Layer | Patterns |
|-------|----------|
| FE | API Wrapper, ForDisplay Getter, Error Handling, Dynamic Columns |
| BE | Controller, Repository, Validation |
| DB | SP Versioning, Registration, Soft Delete, Pagination |

**Files:**
- `pattern-detection/pattern-signatures.json` - Regex patterns
- `pattern-detection/pattern-detector.md` - Detection workflow
- `pattern-detection/code-generation.md` - Stub generation

---

## v3.2.0 Features

### Programmatic Pattern Validation

Validate generated code against pattern-signatures.json:

```bash
# Validate specific layer
python scripts/pattern-validator.py --path /path/to/feature --layer fe

# Validate all layers
python scripts/pattern-validator.py --path /path/to/feature --layer all

# Generate markdown report
python scripts/pattern-report.py --input violations.json --output report.md
```

### Full Automation Suite

| Script | Purpose | Usage |
|--------|---------|-------|
| `stubgen.py` | Generate code from API contract | `python stubgen.py --contract api.json` |
| `mock-server.py` | API mock server for FE dev | `python mock-server.py --port 3000` |
| `scaffold-feature.py` | Full feature scaffolding | `python scaffold-feature.py --feature PlayerNote` |
| `sp-validator.py` | SQL SP validation | `python sp-validator.py --path /path/to/sp.sql` |
| `validate-templates.py` | Template syntax check | `python validate-templates.py` |
| `be-test-template.py` | BE API testing | `python be-test-template.py --feature playernote` |

### Expanded Template Library (22 Total)

| Layer | Templates | New in v3.2.0 |
|-------|-----------|---------------|
| FE | 7 | Dialog, Filter Form, Form Validation |
| BE | 5 | Controller Tests |
| DB | 9 | Search SP, Bulk SP |
| Contract | 1 | - |

**New Templates:**
- `fe-dialog-template.vue` - Modal dialog for create/edit
- `fe-filter-form-template.vue` - Reusable filter component
- `fe-form-validation-template.ts` - Form validation rules
- `be-test-template.cs` - xUnit tests for controllers
- `db-sp-search-template.sql` - Full-text search SP
- `db-sp-bulk-template.sql` - Bulk operations SP

---

## v3.1.0 Features

### FE Testing with Screenshot Proof

Test Kirby frontend with Playwright and capture visual proof:

```bash
python scripts/fe-test-template.py \
  --url "http://localhost:8080/v2/feature-page" \
  --output "/tmp/fe-tests"
```

**Test Patterns:**
- Page Load Test
- Table Data Test
- Filter Form Test
- Column Toggle Test
- Export Test
- Form Validation Test

**Files:**
- `playbooks/fe-testing.md` - Full testing guide
- `scripts/fe-test-template.py` - Test script template

---

## Architecture Overview

```
┌─────────────────────────────────────────────────────────────────┐
│                        FRONTEND (Kirby)                         │
│  Vue 3 + TypeScript + Element Plus                              │
│  Path: /Users/rithytep_1/projects/Nuxt/kirby                    │
├─────────────────────────────────────────────────────────────────┤
│                        BACKEND API (Coloris)                    │
│  .NET Framework + C# + Web API                                  │
│  Path: /Users/rithytep_1/projects/BackEnd/Coloris               │
├─────────────────────────────────────────────────────────────────┤
│                     PROMOTION SERVICE (Promodia)                │
│  Microservice for promotion logic                               │
├─────────────────────────────────────────────────────────────────┤
│                        DATABASE (Monika)                        │
│  SQL Server + Stored Procedures                                 │
│  Path: /Users/rithytep_1/projects/BackEnd/monika                │
└─────────────────────────────────────────────────────────────────┘
```

---

## Development Flow

### Full Stack Order

```
Database (Monika) → Backend (Coloris) → Frontend (Kirby)
```

### Frontend Flow (CustomerRecordReport Pattern)

```
Model → apiCalling.ts → apis.ts → FakeData → Composable → Component → Router
```

### Backend Flow (Coloris)

```
Model → Repository → Service → Controller → Register DI
```

### Database Flow (Monika)

```
Table → Stored Procedure → InsertData (sp_lookup registration)
```

---

## File Reference

| Need | File |
|------|------|
| Quick start | `INDEX.md` |
| FE agent guide | `agents/kirby-fe-agent.md` |
| BE agent guide | `agents/coloris-be-agent.md` |
| DB agent guide | `agents/monika-db-agent.md` |
| FE templates | `templates/fe-*.ts`, `templates/fe-*.vue` |
| BE templates | `templates/be-*.cs` |
| DB templates | `templates/db-*.sql` |
| Domain knowledge | `docs/backoffice/*.md` |
| Playbooks | `playbooks/*.md` |
| Learnings | `.memory/learnings.json` |

---

## Quick Reference

### API Layer (ONLY 2 FILES)

```
src/libraries/apiCalling.ts  → ALL API methods (callGetFeature)
src/libraries/apis.ts        → Wrapper layer (composables use this)
```

### SP Versioning (REQUIRED)

```sql
-- Format: {Service}_{Feature}_{Action}_{Major}.{Minor}.{Patch}
Coloris_Player_GetNotes_1.0.0
Promodia_GetMemberPromotion_1.2.0
```

### SP Registration (InsertData.sql)

```sql
INSERT [dbo].[SimpleSettings]
  ([Website], [IsUAT], [Type], [Id], [Value], ...)
VALUES
  (N'Coloris', 0, N'sp_lookup', N'Coloris_Player_GetNotes',
   N'[dbo].[Coloris_Player_GetNotes_1.0.0]', ...)
```

### Error Codes

| Code | Message | Usage |
|------|---------|-------|
| 0 | Success | All operations |
| 330 | Duplicate exists | Conflict on create |
| 400 | Bad request | Validation failure |
| 404 | Not found | Record not found |

### Translation (i18n)

**Tool:** [WL Translation Engine](https://marketplace.visualstudio.com/items?itemName=RithyTep.kirby-i18n)

```typescript
// In composables
import { t } from '@/libraries/vue-i18n'
const msg = t('translation_key')
```

```vue
<!-- In templates -->
{{ $t('translation_key') }}
```

---

## Usage Examples

### Example 1: Full Feature

```
User: Create a player notes feature

Claude:
1. Load domain knowledge (03-membership.md)
2. Design PlayerNote table in Monika
3. Create stored procedures (CRUD)
4. Register SP in InsertData.sql
5. Create C# models, repository, service, controller
6. Create TypeScript model with ForDisplay getters
7. Add methods to apiCalling.ts and apis.ts
8. Create composable using apis wrapper
9. Create Vue component with ColumnToggle
10. Add route with authorizeRoute
```

### Example 2: FE Only

```
User: Add a new report page for withdrawal summary

Claude:
1. Follow CustomerRecordReport pattern
2. Create model in src/models/report/
3. Add to apiCalling.ts and apis.ts
4. Create composable with IColumnsDynamic
5. Create Vue component
6. Add route entry
```

### Example 3: DB Only

```
User: Add audit log table for player actions

Claude:
1. Create table in appropriate schema
2. Create versioned SP (1.0.0)
3. Add to InsertData.sql for sp_lookup
4. Document in shared-memory.md
```

---

## Domain Knowledge

Load relevant files from `docs/backoffice/` before implementation:

| File | Content |
|------|---------|
| 01-dashboard.md | Cap limit, metrics, star icons |
| 02-sub-account.md | Permissions, encryption, IP whitelist |
| 03-membership.md | Member list, KYC (L1-3), VIP (9 levels), labels |
| 04-reports.md | Win/Lost, betting records, player summary |
| 05-financial-center.md | Deposit/withdrawal, auto-approval, banking |
| 06-promotions.md | Bonus, rebate, reward types, promo wallet |
| 07-referral.md | Referrer/referee relationships |
| 08-game-platform.md | Providers, games, bet limits |
| 09-platform-settings.md | System params, currency, themes |
| 10-cash-agent.md | Agent hierarchy, PT adjustment |
| 11-risk-control.md | IP/FP detection, alerts |
| 12-security.md | Password policy, 2FA |

---

## Sub-Agents

This skill uses specialized sub-agents for parallel development:

### Kirby FE Agent

- **Focus:** Vue components, composables, API integration
- **Pattern:** CustomerRecordReport flow
- **Key files:** apiCalling.ts, apis.ts, FakeAPI.ts
- **Guide:** `agents/kirby-fe-agent.md`

### Coloris BE Agent

- **Focus:** C# Controllers, Services, Repositories
- **Pattern:** Repository pattern with Dapper
- **Key files:** Controllers/, Services/, Repository/
- **Guide:** `agents/coloris-be-agent.md`

### Monika DB Agent

- **Focus:** SQL tables, stored procedures
- **Pattern:** Versioned SPs with InsertData registration
- **Key files:** Tables/, Stored Procedures/, InsertData.sql
- **Guide:** `agents/monika-db-agent.md`

### Promodia Agent

- **Focus:** Promotion logic, VIP features
- **Pattern:** gRPC + Strategy pattern (not REST)
- **SP Prefix:** `Promodia_`
- **Guide:** `agents/promodia-agent.md`

---

## Learning Prompts

After completing an implementation, capture knowledge for future sessions:

| Trigger | Action |
|---------|--------|
| Discovered new pattern? | Update `.memory/learnings.json` |
| Found table structure? | Update `.memory/tables.md` |
| New domain/feature? | Create `.memory/domain/{feature}.md` |
| Common issue resolved? | Add to learnings.json `common_issues` |

**Workflow:**
```
1. Complete implementation
2. Commit code changes
3. Update memory files
4. Commit memory (separate commit)
```

**Example learning entry:**
```json
{
  "pattern_name": "description",
  "learned_from": "source file",
  "usage": "when to apply"
}
```

---

## Git Contribution

### Commit Message Format

```
[BackofficeFullstack] {Type} v{Major}.{Minor}.{Patch} - {Description}
```

**Types:**
| Type | Usage |
|------|-------|
| `Init` | Initial release |
| `Feature` | New feature or major addition |
| `Fix` | Bug fix |
| `Docs` | Documentation only |
| `Refactor` | Code restructure without new features |

### Git Tag Format

```bash
# Create annotated tag after version commit
git tag -a backoffice-fullstack-v{version} {commit} -m "BackofficeFullstack v{version} - {description}"

# Example
git tag -a backoffice-fullstack-v2.2.0 2bc7ba9 -m "BackofficeFullstack v2.2.0 - Learning & Validation"

# Push tags to remote
git push origin --tags
```

### Version Bump Workflow

```
1. Implement changes
2. Update VERSION.md changelog
3. Update SKILL.md version + changelog
4. Commit: [BackofficeFullstack] Feature vX.Y.Z
5. Tag: git tag -a backoffice-fullstack-vX.Y.Z HEAD -m "..."
6. Push: git push && git push --tags
```

---

## CLI Commands

```bash
# Frontend dev server (staging)
cd /Users/rithytep_1/projects/Nuxt/kirby && npm run serve:stg

# Build Coloris
cd /Users/rithytep_1/projects/BackEnd/Coloris/coloris/Coloris && dotnet build

# Test API endpoint
curl -X POST "http://localhost/api/{feature}/v2/{endpoint}" \
  -H "Authorization: Bearer {token}" \
  -H "Content-Type: application/json" \
  -d '{"WebId": 1, "PageIndex": 1, "PageSize": 25}'
```

---

## Changelog

### v3.2.0 (2025-01-08)

- **Programmatic Pattern Validation** - Python scripts for code validation
- **Full Automation Suite** - 6 new automation scripts
- **Expanded Template Library** - 22 templates (6 new)
- Added `scripts/pattern-validator.py` - Pattern detection with line numbers
- Added `scripts/pattern-report.py` - Markdown report generation
- Added `scripts/be-test-template.py` - BE API testing with requests
- Added `scripts/validate-templates.py` - Template syntax validation
- Added `scripts/sp-validator.py` - SQL SP naming convention validator
- Added `scripts/stubgen.py` - Code stub generator from API contract
- Added `scripts/mock-server.py` - Flask-based API mock server
- Added `scripts/scaffold-feature.py` - Full feature scaffolding
- Added `templates/fe-dialog-template.vue` - Modal dialog component
- Added `templates/fe-filter-form-template.vue` - Reusable filter form
- Added `templates/fe-form-validation-template.ts` - Form validation rules
- Added `templates/be-test-template.cs` - xUnit controller tests
- Added `templates/db-sp-search-template.sql` - Full-text search SP
- Added `templates/db-sp-bulk-template.sql` - Bulk operations SP

### v3.1.0 (2025-01-08)

- **FE Testing** - Playwright tests with screenshot proof
- Added `playbooks/fe-testing.md` - Testing patterns and examples
- Added `scripts/fe-test-template.py` - Reusable test script
- URL format: `http://localhost:8080/v2/{page}`

### v3.0.0 (2025-01-08)

- **Multi-Agent Parallel Execution** - 3x faster feature development
- **Automatic Pattern Detection** - 8 patterns with violation checking
- Added `agents/parallel-execution-guide.md`
- Added `prompts/agent-dependencies.json`
- Added `prompts/batch-spawn-templates.md`
- Added `pattern-detection/pattern-signatures.json`
- Added `pattern-detection/pattern-detector.md`
- Added `pattern-detection/code-generation.md`
- Updated orchestrator with v3.0.0 protocol

### v2.2.1 (2025-01-08)

- Added Git Contribution section (commit format, tag format, workflow)
- Created git tags for all versions (v1.0.0 → v2.2.0)

### v2.2.0 (2025-01-08)

- Added FE composable template with export and column toggle
- Added Promodia agent guide (gRPC architecture)
- Added validation playbook (FE/BE/DB patterns)
- Added learning prompts section for knowledge capture
- Updated sub-agents section with Promodia

### v2.1.0 (2025-01-08)

- Added auto-load memory on session start
- Added BE templates (controller, service, repository, model)
- Added DB templates (table, SP CRUD, insertdata)
- Added memory strategy for session persistence
- Renamed memory/ to .memory/ (hidden folder)

### v2.0.0 (2025-01-08)

- Added CustomerRecordReport flow pattern for FE
- Clarified API layer: only apiCalling.ts + apis.ts
- Added apis.ts wrapper layer documentation
- Added SP versioning and InsertData patterns
- Added IColumnsDynamic for dynamic columns
- Reorganized folder structure (docs, templates, playbooks)
- Added INDEX.md quick reference
- Added VERSION.md for tracking

### v1.0.0 (2025-01-08)

- Initial skill creation
- Basic FE/BE/DB workflows
- Domain knowledge documentation
