# Parallel Execution Guide v3.0.0

## Overview

This guide enables 3x faster feature development by running agents in parallel.

| Metric | Sequential | Parallel |
|--------|------------|----------|
| Full feature | 30-45 min | 10-15 min |
| FE + BE | 25-35 min | 10-12 min |

---

## 3-Phase Execution Model

```
Phase 1: Database (SEQUENTIAL - Blocker)
└── Monika Agent
    ├── Create tables
    ├── Create stored procedures
    └── Register in InsertData.sql

Phase 2: Backend + Frontend (PARALLEL)
├── Coloris Agent
│   ├── Create models
│   ├── Create repository
│   ├── Create service
│   └── Create controller
└── Kirby Agent
    ├── Create TypeScript models
    ├── Create apiCalling.ts methods
    ├── Create apis.ts wrappers
    └── Create FakeData

Phase 3: Frontend Integration (SEQUENTIAL)
└── Kirby Agent
    ├── Create composable
    ├── Create component
    └── Add router entry
```

---

## Shared Memory Coordination

### Directory Structure

```
.memory/features/{feature-name}/
├── specification.md    # Requirements (Orchestrator writes first)
├── api-contract.json   # Shared API schema (all agents read)
├── db-schema.sql       # Database design (Monika writes)
└── status.json         # Real-time status tracking
```

### Status Tracking

Each agent updates `status.json` as they progress:

```json
{
  "feature": "player-notes",
  "created": "2025-01-08T12:00:00Z",
  "updated": "2025-01-08T12:15:00Z",
  "db": {
    "tables": "completed",
    "procedures": "completed",
    "registration": "completed"
  },
  "be": {
    "models": "in_progress",
    "repository": "pending",
    "service": "pending",
    "controller": "pending"
  },
  "fe": {
    "models": "completed",
    "apiCalling": "completed",
    "apis": "completed",
    "fakeData": "completed",
    "composable": "blocked",
    "component": "blocked",
    "router": "blocked"
  }
}
```

### Status Values

| Status | Meaning |
|--------|---------|
| `pending` | Not started |
| `in_progress` | Currently working |
| `completed` | Finished successfully |
| `blocked` | Waiting for dependency |
| `failed` | Error occurred |

---

## API Contract (Communication Hub)

Before spawning any agent, write the API contract:

```json
{
  "feature": "player-notes",
  "endpoints": [
    {
      "name": "GetList",
      "method": "POST",
      "path": "/api/player-note/v2/get-list",
      "request": {
        "WebId": "int",
        "PlayerId": "int",
        "PageIndex": "int",
        "PageSize": "int"
      },
      "response": {
        "ErrorCode": "int",
        "ErrorMessage": "string",
        "Data": {
          "List": "PlayerNoteItem[]",
          "TotalCount": "int"
        }
      }
    }
  ]
}
```

**Why:** Both BE and FE agents read this independently. No direct coordination needed.

---

## Dependency Matrix

| Task | BLOCKS | BLOCKED_BY | CAN_PARALLEL |
|------|--------|------------|--------------|
| DB Tables | DB SPs | None | None |
| DB SPs | BE Repository | DB Tables | None |
| DB InsertData | None | DB SPs | None |
| BE Models | None | API Contract | FE Models |
| BE Repository | BE Service | DB SPs, BE Models | FE API |
| BE Service | BE Controller | BE Repository | FE API |
| BE Controller | FE Composable | BE Service | FE FakeData |
| FE Models | FE API | API Contract | BE Models |
| FE apiCalling | FE apis | FE Models | BE Models |
| FE apis | FE FakeData | FE apiCalling | BE Repository |
| FE FakeData | None | FE apis | BE Controller |
| FE Composable | FE Component | FE apis, BE Controller | None |
| FE Component | FE Router | FE Composable | None |
| FE Router | None | FE Component | None |

---

## Orchestrator Protocol

### Step 1: Initialize Feature

```markdown
1. Create feature directory: .memory/features/{feature-name}/
2. Write specification.md with requirements
3. Define and write api-contract.json
4. Initialize status.json with all tasks "pending"
```

### Step 2: Spawn Phase 1 (Database)

```markdown
SPAWN: Monika DB Agent

Task: Implement database layer for {feature}

Read:
- .memory/features/{feature}/specification.md
- .memory/features/{feature}/api-contract.json

Create:
- Tables with proper schema
- Stored procedures (versioned: {Service}_{Feature}_{Action}_1.0.0)
- InsertData.sql registration

Update:
- .memory/features/{feature}/status.json → db.* = "completed"
- .memory/features/{feature}/db-schema.sql
```

### Step 3: Wait for Phase 1 Completion

```markdown
POLL: status.json
UNTIL: db.tables = "completed" AND db.procedures = "completed"
```

### Step 4: Spawn Phase 2 (Parallel)

```markdown
SPAWN GROUP (parallel):

1. Coloris BE Agent
   Task: Implement backend API layer
   Read: api-contract.json, db-schema.sql
   Create: Models, Repository, Service, Controller
   Update: status.json → be.* = "completed"

2. Kirby FE Agent
   Task: Implement frontend API layer (models + API only)
   Read: api-contract.json
   Create: Models, apiCalling.ts, apis.ts, FakeData
   Update: status.json → fe.models/apiCalling/apis/fakeData = "completed"
```

### Step 5: Wait for BE Controller

```markdown
POLL: status.json
UNTIL: be.controller = "completed"
```

### Step 6: Spawn Phase 3 (FE Integration)

```markdown
SPAWN: Kirby FE Agent (continuation)

Task: Complete frontend integration
Read: api-contract.json, specification.md
Create: Composable, Component, Router entry
Update: status.json → fe.composable/component/router = "completed"
```

### Step 7: Validation

```markdown
VERIFY all status.json entries = "completed"
RUN integration tests if available
REPORT completion to user
```

---

## Checkpoint Validations

### After Phase 1 (DB)

```markdown
DB_CHECKPOINT:
  [ ] All tables created with correct schema
  [ ] All SPs follow versioning: {Service}_{Feature}_{Action}_X.Y.Z
  [ ] All SPs registered in InsertData.sql
  [ ] Schema matches api-contract.json
```

### After Phase 2 (BE)

```markdown
BE_CHECKPOINT:
  [ ] Models match api-contract.json
  [ ] Repository calls correct SPs
  [ ] Service has validation logic
  [ ] Controller routes match API paths
  [ ] DI registration complete
```

### After Phase 2 (FE)

```markdown
FE_API_CHECKPOINT:
  [ ] Models have ForDisplay getters
  [ ] apiCalling.ts methods added
  [ ] apis.ts wrapper methods added
  [ ] FakeData returns valid test data
```

### After Phase 3 (FE Integration)

```markdown
FE_INTEGRATION_CHECKPOINT:
  [ ] Composable uses apis (not apiCalling)
  [ ] Composable has IColumnsDynamic
  [ ] Component renders with Element Plus
  [ ] Router entry with authorization
```

---

## Error Handling

### If Agent Fails

```markdown
1. Check error type:
   - Validation error → Agent fixes and retries
   - Missing dependency → Set status = "blocked"
   - Critical failure → Set status = "failed", report to user

2. Prevent cascade:
   - Set dependent tasks to "blocked"
   - Don't spawn dependent agents
   - Wait for user intervention

3. Recovery:
   - Fix the issue
   - Reset status to "pending"
   - Re-spawn the agent
```

### If Dependency Missing

```markdown
1. Agent detects missing dependency
2. Sets own status to "blocked"
3. Reports: "Blocked waiting for {dependency}"
4. Orchestrator checks predecessor status
5. When predecessor completes, unblocks and resumes
```

---

## Example: Player Notes Feature

### Orchestration Flow

```
[Orchestrator]
  │
  ├─► Create .memory/features/player-notes/
  │   ├── specification.md
  │   ├── api-contract.json
  │   └── status.json (all pending)
  │
  ├─► SPAWN: Monika Agent
  │   └── Creates tables, SPs, InsertData
  │   └── Updates: db.* = "completed"
  │
  ├─► WAIT: db.procedures = "completed"
  │
  ├─► SPAWN PARALLEL:
  │   │
  │   ├─► Coloris Agent
  │   │   └── Creates BE layer
  │   │   └── Updates: be.* = "completed"
  │   │
  │   └─► Kirby Agent (API only)
  │       └── Creates models, apiCalling, apis, FakeData
  │       └── Updates: fe.models/apiCalling/apis/fakeData = "completed"
  │
  ├─► WAIT: be.controller = "completed"
  │
  └─► SPAWN: Kirby Agent (integration)
      └── Creates composable, component, router
      └── Updates: fe.composable/component/router = "completed"

[DONE: ~12 minutes total]
```

---

## Quick Reference

### Spawn Commands

```markdown
# Phase 1: Database
"Implement database layer for {feature}. Read specification and api-contract from shared memory. Create tables, versioned SPs, and InsertData registration. Update status.json when complete."

# Phase 2a: Backend (parallel)
"Implement backend API for {feature}. Read api-contract and db-schema. Create Models, Repository, Service, Controller. Update status.json for each component."

# Phase 2b: Frontend API (parallel)
"Implement frontend API layer for {feature}. Read api-contract. Create TypeScript models with ForDisplay getters, apiCalling.ts, apis.ts, FakeData. Update status.json."

# Phase 3: Frontend Integration
"Complete frontend for {feature}. Create composable using apis wrapper, Vue component with IColumnsDynamic, router entry. Update status.json."
```

### Time Estimates

| Phase | Sequential | With Parallel |
|-------|------------|---------------|
| Phase 1 (DB) | 5-10 min | 5-8 min |
| Phase 2 (BE+FE) | 20-25 min | 5-8 min |
| Phase 3 (FE int) | 5-10 min | 2-4 min |
| **Total** | **30-45 min** | **12-20 min** |
