# Promodia Agent Guide

Promodia is a **gRPC-based microservice** for promotion and VIP logic. It has a fundamentally different architecture from Coloris (REST).

## Architecture Overview

```
┌─────────────────────────────────────────────────────────────────┐
│                        PROMODIA                                  │
│  .NET 7+ | gRPC | Protocol Buffers | Strategy Pattern           │
│  Path: /Users/rithytep_1/projects/BackEnd/promodia              │
├─────────────────────────────────────────────────────────────────┤
│  gRPC Services                                                   │
│  ├── PromotionService (2,365 lines)                             │
│  └── VipService (12,701 lines)                                  │
├─────────────────────────────────────────────────────────────────┤
│  Strategy Layer (5,279 lines)                                   │
│  ├── PromotionType/ (10+ strategies)                            │
│  └── PromotionSetting/ (91 setting strategies)                  │
├─────────────────────────────────────────────────────────────────┤
│  Repositories                                                    │
│  ├── PromotionRepository                                        │
│  ├── VipRepository                                              │
│  └── GameProviderRepository                                     │
└─────────────────────────────────────────────────────────────────┘
```

## Promodia vs Coloris Comparison

| Aspect | Promodia | Coloris |
|--------|----------|---------|
| Protocol | gRPC (Protocol Buffers) | REST (HTTP/JSON) |
| Framework | .NET 7+ | .NET Framework |
| Service Layer | gRPC Services | MVC Controllers |
| DI | builder.Services | Autofac |
| Async | Full async/await | Mixed sync/async |
| Data Models | Proto messages + DTOs | Plain C# classes |

## Project Structure

```
promodia/Promodia/
├── Model/                     # 97 subdirectories - DTOs
│   ├── Account/
│   ├── Affiliate/
│   ├── DepositPromotionOption/
│   ├── Entities/
│   ├── GameProvider/
│   ├── Payment/
│   ├── Promotion/
│   ├── Referral/
│   └── Vip/
├── Repositories/              # Database access layer
│   ├── IPromotionRepository.cs
│   ├── PromotionRepository.cs
│   ├── VipRepository.cs
│   └── MainRepository.cs
├── Services/                  # gRPC services
│   ├── PromotionService.cs
│   └── VipService.cs
├── Strategy/                  # Business logic strategies
│   ├── PromotionType/
│   ├── PromotionSetting/
│   ├── Affiliate/
│   └── ReferralType/
├── Factory/                   # Strategy factories
│   ├── PromotionTypeFactory.cs
│   └── PromotionSettingFactory.cs
├── Protos/                    # gRPC definitions
│   ├── model.proto
│   ├── promotion.proto
│   └── vip.proto
├── Cache/
├── Enums/
├── Exception/
└── Interceptor/
```

## Stored Procedure Naming

**Prefix:** `Promodia_`

**Format:** `Promodia_{Feature}_{Action}`

**Examples:**
```
Promodia_Promotion_GetPromotionRequestList
Promodia_Promotion_GetCompanyPromotions
Promodia_Promotion_UpdateCompanyPromotion
Promodia_Promotion_TransferToMainWallet
Promodia_UpdateVipEvent
Promodia_GetVipMemberInfo
Promodia_CalculateVipBonusMemberPerformance
Promodia_GetPlayerRedeemHistory
```

## Strategy Pattern

### Promotion Type Strategies

Base class: `PromotionTypeStrategy`

Concrete implementations:
- FirstDepositBonus
- SingleDepositBonus
- DailyDepositBonus
- RegisterBonus
- RebateOnTurnover
- RebateOnTurnoverLoseOnly
- RebateOnWinlose
- RebateOnDepositAndWithdrawal
- CouponReward
- LuckyWheelBonus

### Strategy Constructor Pattern

```csharp
public class FirstDepositBonus : PromotionTypeStrategy
{
    public FirstDepositBonus(
        IPromotionRepository promotionRepository,
        ILoggerService logger,
        PromotionSettingStrategy headSetting,
        PromotionTypeDto promotionTypeDto,
        IMainRepository mainRepository,
        IAccountRepository accountRepository,
        IAffiliateProviderService affiliateProviderService)
        : base(promotionRepository, logger, headSetting, promotionTypeDto)
    {
    }
}
```

### Factory Pattern

```csharp
public class PromotionTypeFactory
{
    public PromotionTypeStrategy Create(EnumPromotionType type)
    {
        return type switch
        {
            EnumPromotionType.FirstDeposit => new FirstDepositBonus(...),
            EnumPromotionType.SingleDeposit => new SingleDepositBonus(...),
            // ... more types
        };
    }
}
```

## gRPC Service Pattern

### Proto Definition

```protobuf
service Promotion {
    rpc GetCreatePromotionTypeList(...) returns (...);
    rpc CreateCompanyPromotion(...) returns (...);
    rpc ProceedPromotion(...) returns (...);
    rpc ApplyPromotion(...) returns (...);
    rpc GetPromotionRequestList(...) returns (...);
}
```

### Service Implementation

```csharp
public class PromotionService : Promotion.PromotionBase
{
    private readonly IPromotionRepository _promotionRepository;
    private readonly PromotionTypeFactory _factory;

    public PromotionService(
        IPromotionRepository promotionRepository,
        PromotionTypeFactory factory)
    {
        _promotionRepository = promotionRepository;
        _factory = factory;
    }

    public override async Task<ProtoGeneralResponse> ApplyPromotion(
        ProtoApplyPromotionRequest request,
        ServerCallContext context)
    {
        var strategy = _factory.Create(request.PromotionType);
        var result = await strategy.ProcessAsync(request);
        return new ProtoGeneralResponse { ... };
    }
}
```

## Repository Pattern

```csharp
public class PromotionRepository : BaseRepository, IPromotionRepository
{
    public PromotionRequestListResponseDto GetPromotionRequestList(
        GetPromotionRequestListDto request)
    {
        return GetData<PromotionRequestListResponseDto>(
            "Promodia_Promotion_GetPromotionRequestList",
            new
            {
                request.WebId,
                request.Page,
                request.RowCountPerPage,
                request.Status
            });
    }
}
```

## Promotion Request Flow

```
1. gRPC Request → PromotionService.ApplyPromotion()
                         ↓
2. PromotionTypeFactory.Create() → Select strategy
                         ↓
3. PromotionTypeStrategy.ValidateApplyRequest()
                         ↓
4. PromotionTypeStrategy.ProcessPromotion()
                         ↓
5. IPromotionRepository.ApplyPromotion() → SP call
                         ↓
6. gRPC Response
```

## Setting Strategies (91 Types)

Examples of specialized setting classes:
- MaximumTotalUsage
- RefereeLimitation
- ExcludeRebate
- SpinQualificationAccumulatedDepositTicket
- MinimumDeposit
- MaximumBonus
- TurnoverRequirement

Each handles specific validation and calculation logic.

## Program.cs Configuration

```csharp
var builder = WebApplication.CreateBuilder(args);

builder.Services.AddGrpc(options =>
{
    options.Interceptors.Add<GrpcLogInterceptor>();
    options.Interceptors.Add<GrpcGlobalExceptionHandlerInterceptor>();
});

builder.Services.AddScoped<IPromotionRepository, PromotionRepository>();
builder.Services.AddScoped<PromotionTypeFactory>();

var app = builder.Build();

app.MapGrpcService<PromotionService>();
app.MapGrpcService<VipService>();

app.Run();
```

## When to Use This Guide

Use Promodia patterns when:
- Working with promotion logic
- Implementing VIP features
- Creating new promotion types
- Adding promotion settings

Do NOT use for:
- REST API endpoints (use Coloris)
- Backoffice UI backend (use Coloris)
- Non-promotion features (use Coloris)

## Key Enums

```csharp
EnumPromotionType
EnumPromotionStatus
EnumRebateCalculation
EnumCouponType
EnumVipBonusType
EnumBonusWalletType
```

## Caching

- ArtemisEnumCache (10-minute memory cache)
- GameProviderCache
- Lock-based synchronization for updates
