# Code Generation from Patterns v3.0.0

Generate code stubs automatically from established patterns.

---

## Overview

Pattern-driven code generation:
1. Read API contract
2. Load pattern signatures
3. Generate stub code following patterns
4. Agent completes implementation

**Benefit:** 30-40% less boilerplate, consistent patterns

---

## FE Code Stubs

### Model Stub

**Input:** API Contract
**Pattern:** ForDisplay getters

```typescript
// Auto-generated from api-contract.json

export interface I{Feature}Request {
  CustomerId: number
  StartDate: string
  EndDate: string
  Page: number
  RowCountPerPage: number
}

export interface I{Feature}Response {
  Id: number
  Amount: number
  Currency: string
  Status: number
  CreatedOn: string
}

export class {Feature}Model {
  Id: number
  Amount: number
  Currency: string
  Status: number
  CreatedOn: string

  constructor(data: I{Feature}Response) {
    this.Id = data.Id
    this.Amount = data.Amount
    this.Currency = data.Currency
    this.Status = data.Status
    this.CreatedOn = data.CreatedOn
  }

  // Pattern: ForDisplay getters
  get AmountForDisplay(): string {
    return getValueForDisplay(this.Amount, this.Currency)
  }

  get StatusForDisplay(): string {
    return this.Status === 1 ? 'Active' : 'Inactive'
  }

  get CreatedOnForDisplay(): string {
    return format(new Date(this.CreatedOn), 'yyyy-MM-dd HH:mm:ss')
  }
}
```

### apiCalling Stub

**Input:** API Contract endpoints
**Pattern:** API wrapper with commented real API (use FakeAPI for dev)

```typescript
// Auto-generated from api-contract.json

// Add to apiCalling.ts
callGet{Feature}List(request: I{Feature}Request) {
  // return api.post('/api/{feature}/v2/get-list', request)
  return FakeAPI.get{Feature}List(request)
}

callCreate{Feature}(request: ICreate{Feature}Request) {
  // return api.post('/api/{feature}/v2/create', request)
  return FakeAPI.create{Feature}(request)
}

callUpdate{Feature}(request: IUpdate{Feature}Request) {
  // return api.post('/api/{feature}/v2/update', request)
  return FakeAPI.update{Feature}(request)
}

callDelete{Feature}(request: IDelete{Feature}Request) {
  // return api.post('/api/{feature}/v2/delete', request)
  return FakeAPI.delete{Feature}(request)
}
```

### apis Stub

**Input:** apiCalling methods
**Pattern:** getResponse wrapper

```typescript
// Auto-generated from apiCalling methods

// Add to apis.ts
get{Feature}List(request: I{Feature}Request) {
  return getResponse(apiCalling.callGet{Feature}List(request))
}

create{Feature}(request: ICreate{Feature}Request) {
  return getResponse(apiCalling.callCreate{Feature}(request))
}

update{Feature}(request: IUpdate{Feature}Request) {
  return getResponse(apiCalling.callUpdate{Feature}(request))
}

delete{Feature}(request: IDelete{Feature}Request) {
  return getResponse(apiCalling.callDelete{Feature}(request))
}
```

### FakeData Stub

**Input:** Response interface
**Pattern:** Promise.resolve with sample data

```typescript
// Auto-generated from I{Feature}Response

// Add to FakeAPI.ts
get{Feature}List(request: I{Feature}Request) {
  return Promise.resolve({
    ErrorCode: 0,
    ErrorMessage: 'Success',
    Data: {
      List: [
        {
          Id: 1,
          Amount: 100.00,
          Currency: 'USD',
          Status: 1,
          CreatedOn: '2025-01-08T12:00:00Z'
        },
        {
          Id: 2,
          Amount: 200.00,
          Currency: 'USD',
          Status: 1,
          CreatedOn: '2025-01-07T12:00:00Z'
        }
      ],
      TotalCount: 2,
      MaxPage: 1
    }
  })
}

create{Feature}(request: ICreate{Feature}Request) {
  return Promise.resolve({
    ErrorCode: 0,
    ErrorMessage: 'Success',
    Data: { Id: 1 }
  })
}
```

### Composable Stub

**Input:** Model + APIs
**Pattern:** useVariable, reactive filters, IColumnsDynamic

```typescript
// Auto-generated composable stub

import { ref, reactive, onMounted, watch } from 'vue'
import { format } from 'date-fns'
import apis from '@/libraries/apis'
import store from '@/store'
import EnumApiErrorCode from '@/models/enums/enumsApiErrorCode'
import EnumMessageType from '@/models/enums/enumMessageType'
import notificationHelper from '@/libraries/elementUiHelpers/notificationHelper'
import textFormatHelper from '@/libraries/formatHelpers/textFormatHelper'
import { t } from '@/libraries/vue-i18n'
import { IColumnsDynamic } from '@/models/Columns'
import { ITableRow, multipleExportExcel } from '@/libraries/excelHelper'
import useToggleTableHeaderColumn from '@/composables/useToggleTableHeader'
import { useVariable } from './useVariable'
import { I{Feature}Request, {Feature}Model } from '@/models/{module}/{feature}Model'

export default function use{Feature}() {
  const { isLoading, isProcessing, isExportable } = useVariable('module/{feature}')

  const startDate = ref(format(textFormatHelper.getCompanyTodayTimeZone().start, 'yyyy-MM-dd HH:mm:ss'))
  const endDate = ref(format(textFormatHelper.getCompanyTodayTimeZone().end, 'yyyy-MM-dd HH:mm:ss'))

  const filterFormModel = reactive({
    StartDate: startDate.value,
    EndDate: endDate.value,
    RowCountPerPage: 25,
    Page: 1,
    TotalPage: 1,
  })

  const tableData = ref<{Feature}Model[]>([])
  const dataLoading = ref(false)

  const columns = ref<IColumnsDynamic[]>([
    // TODO: Define columns based on response fields
    { label: 'no', value: 1, show: true, columnIndex: 1, isSortable: false, prop: 'RowNumber', width: 60, align: 'center' },
    { label: 'amount', value: 2, show: true, columnIndex: 2, isSortable: true, prop: 'AmountForDisplay', width: 120, align: 'right' },
    { label: 'status', value: 3, show: true, columnIndex: 3, isSortable: false, prop: 'StatusForDisplay', width: 100, align: 'center' },
    { label: 'created_on', value: 4, show: true, columnIndex: 4, isSortable: true, prop: 'CreatedOnForDisplay', width: 160, align: 'center' },
  ])

  const { storeTableColumnsToLocalStorage, setTableColumnsSettingFromStore } = useToggleTableHeaderColumn()

  const createRequest = (): I{Feature}Request => ({
    CustomerId: Number(store.state.customerId),
    StartDate: filterFormModel.StartDate,
    EndDate: filterFormModel.EndDate,
    RowCountPerPage: filterFormModel.RowCountPerPage,
    Page: filterFormModel.Page,
  })

  const getData = async () => {
    dataLoading.value = true
    try {
      const request = createRequest()
      const response = await apis.get{Feature}List(request)

      if (response.ErrorCode === EnumApiErrorCode.Success) {
        tableData.value = response.Data.List.map((item) => new {Feature}Model(item))
        filterFormModel.TotalPage = response.Data.MaxPage ?? 1
      } else {
        tableData.value = []
        notificationHelper.notification(response.Message, EnumMessageType.Error)
      }
    } catch (error) {
      console.error('getData error:', error)
      tableData.value = []
    } finally {
      dataLoading.value = false
    }
  }

  const onSearch = () => {
    filterFormModel.Page = 1
    getData()
  }

  onMounted(() => {
    setTableColumnsSettingFromStore({ columns: columns.value, pageIndex: 'X.X', isNestedPage: false })
    getData()
  })

  watch(() => columns, () => {
    storeTableColumnsToLocalStorage({ columns: columns.value, pageIndex: 'X.X' })
  }, { deep: true })

  return {
    isLoading,
    isProcessing,
    isExportable,
    dataLoading,
    filterFormModel,
    tableData,
    columns,
    onSearch,
    getData,
  }
}
```

---

## BE Code Stubs

### Model Stub

**Input:** API Contract
**Pattern:** Request/Response DTOs

```csharp
// Auto-generated from api-contract.json

// Request
public class Get{Feature}ListRequest : BaseRequest
{
    public int CustomerId { get; set; }
    public string StartDate { get; set; }
    public string EndDate { get; set; }
    public int Page { get; set; } = 1;
    public int RowCountPerPage { get; set; } = 25;

    public override ApiReturnError Validate()
    {
        if (CustomerId <= 0) return ApiReturnError.InvalidCustomerId;
        return ApiReturnError.Success;
    }
}

// Response
public class Get{Feature}ListResponse : BaseResponse
{
    public List<{Feature}Item> List { get; set; }
    public int TotalCount { get; set; }
    public int MaxPage { get; set; }
}

public class {Feature}Item
{
    public int Id { get; set; }
    public decimal Amount { get; set; }
    public string Currency { get; set; }
    public int Status { get; set; }
    public DateTime CreatedOn { get; set; }
}
```

### Repository Stub

**Input:** SP names from db-schema
**Pattern:** BaseRepository with Dapper

```csharp
// Auto-generated from db-schema.sql

public interface I{Feature}Repository
{
    Task<Get{Feature}ListResponse> GetListAsync(Get{Feature}ListRequest request);
    Task<BaseResponse> CreateAsync(Create{Feature}Request request);
    Task<BaseResponse> UpdateAsync(Update{Feature}Request request);
    Task<BaseResponse> DeleteAsync(Delete{Feature}Request request);
}

public class {Feature}Repository : BaseRepository, I{Feature}Repository
{
    public async Task<Get{Feature}ListResponse> GetListAsync(Get{Feature}ListRequest request)
    {
        var parameters = new
        {
            request.CustomerId,
            request.StartDate,
            request.EndDate,
            request.Page,
            request.RowCountPerPage
        };

        using var connection = GetConnection(DatabaseRole.MainDb);
        var result = await connection.QueryMultipleAsync(
            GetStoredProcedureName("Coloris_{Feature}_GetList"),
            parameters,
            commandType: CommandType.StoredProcedure);

        var items = result.Read<{Feature}Item>().ToList();
        var meta = result.ReadFirstOrDefault<ResponseMeta>();

        return new Get{Feature}ListResponse
        {
            ErrorCode = meta?.ErrorCode ?? 0,
            ErrorMessage = meta?.ErrorMessage ?? "Success",
            List = items,
            TotalCount = meta?.TotalCount ?? 0,
            MaxPage = meta?.MaxPage ?? 1
        };
    }
}
```

---

## DB Code Stubs

### SP GetList Stub

**Input:** Table schema, API contract
**Pattern:** Pagination, error codes

```sql
-- Auto-generated from table schema

CREATE PROCEDURE [dbo].[Coloris_{Feature}_GetList_1.0.0]
    @CustomerId INT,
    @StartDate DATETIME,
    @EndDate DATETIME,
    @Page INT = 1,
    @RowCountPerPage INT = 25
AS
BEGIN
    SET NOCOUNT ON;

    SELECT
        Id,
        Amount,
        Currency,
        Status,
        CreatedOn,
        COUNT(1) OVER() AS TotalCount
    FROM [dbo].[{Feature}]
    WHERE CustomerId = @CustomerId
        AND IsDeleted = 0
        AND CreatedOn BETWEEN @StartDate AND @EndDate
    ORDER BY CreatedOn DESC
    OFFSET (@Page - 1) * @RowCountPerPage ROWS
    FETCH NEXT @RowCountPerPage ROWS ONLY;

    SELECT 0 AS ErrorCode, 'Success' AS ErrorMessage;
END
GO
```

### InsertData Stub

**Input:** SP names
**Pattern:** sp_lookup registration

```sql
-- Auto-generated from SP names

-- GetList
INSERT [dbo].[SimpleSettings] ([Website], [IsUAT], [Type], [Id], [Value], [Remark], [LastUpdatedOn], [LastUpdatedBy])
VALUES (N'Coloris', 0, N'sp_lookup', N'Coloris_{Feature}_GetList', N'[dbo].[Coloris_{Feature}_GetList_1.0.0]', N'{Feature} GetList', GETUTCDATE(), N'System')
GO

-- Create
INSERT [dbo].[SimpleSettings] ([Website], [IsUAT], [Type], [Id], [Value], [Remark], [LastUpdatedOn], [LastUpdatedBy])
VALUES (N'Coloris', 0, N'sp_lookup', N'Coloris_{Feature}_Create', N'[dbo].[Coloris_{Feature}_Create_1.0.0]', N'{Feature} Create', GETUTCDATE(), N'System')
GO
```

---

## Generation Workflow

```markdown
1. Read .memory/features/{feature}/api-contract.json
2. Parse endpoints and schemas
3. Generate stubs:
   - FE: models, apiCalling, apis, FakeData, composable
   - BE: models, repository
   - DB: SPs, InsertData
4. Write stubs to temp files or show inline
5. Agent completes implementation using stubs
```

---

## Quick Reference

### Generate All Stubs

```markdown
From API contract, generate:

FE STUBS:
├── models/{feature}Model.ts       (interfaces + class)
├── apiCalling.ts                  (add methods)
├── apis.ts                        (add wrappers)
├── FakeAPI.ts                     (add fake data)
└── composables/use{Feature}.ts    (full composable)

BE STUBS:
├── Models/{Feature}Request.cs     (request DTOs)
├── Models/{Feature}Response.cs    (response DTOs)
└── Repository/{Feature}Repository.cs (interface + impl)

DB STUBS:
├── Coloris_{Feature}_GetList_1.0.0.sql
├── Coloris_{Feature}_Create_1.0.0.sql
└── InsertData.sql entries
```
