# Pattern Detector v3.0.0

Automatic pattern detection and validation for code quality.

---

## Overview

The pattern detector:
1. **Detects** established patterns in generated code
2. **Validates** adherence to patterns
3. **Reports** violations for correction
4. **Learns** new patterns from implementations

---

## Detection Workflow

```
┌─────────────────────────────────────────────────────────────┐
│                    IMPLEMENTATION                            │
│  Agent generates code for feature                           │
└─────────────────────────────────────────────────────────────┘
                            │
                            ▼
┌─────────────────────────────────────────────────────────────┐
│                    PATTERN SCAN                              │
│  Load pattern-signatures.json                               │
│  Scan generated files with regex patterns                   │
│  Match against known signatures                             │
└─────────────────────────────────────────────────────────────┘
                            │
                            ▼
┌─────────────────────────────────────────────────────────────┐
│                    VALIDATION                                │
│  Check for pattern violations                               │
│  Compare with expected structure                            │
│  Score pattern adherence                                    │
└─────────────────────────────────────────────────────────────┘
                            │
                            ▼
┌─────────────────────────────────────────────────────────────┐
│                    REPORT                                    │
│  List detected patterns                                     │
│  Flag violations with fix suggestions                       │
│  Update learnings.json statistics                           │
└─────────────────────────────────────────────────────────────┘
```

---

## Pattern Categories

### Frontend Patterns (Kirby)

| Pattern | Signature | Files |
|---------|-----------|-------|
| API Wrapper | `getResponse(apiCalling.call...)` | apis.ts |
| ForDisplay Getter | `get {Prop}ForDisplay()` | models/*.ts |
| Error Handling | `ErrorCode === EnumApiErrorCode.Success` | composables/*.ts |
| Dynamic Columns | `ref<IColumnsDynamic[]>` | composables/*.ts |

### Backend Patterns (Coloris)

| Pattern | Signature | Files |
|---------|-----------|-------|
| Controller | `[RoutePrefix][JwtFilter]` | Controllers/*.cs |
| Repository | `class *Repository : BaseRepository` | Repository/*.cs |
| Validation | `override ApiReturnError Validate()` | Models/*Request.cs |

### Database Patterns (Monika)

| Pattern | Signature | Files |
|---------|-----------|-------|
| SP Versioning | `{Service}_{Feature}_{Action}_X.Y.Z` | Stored Procedures/*.sql |
| SP Registration | `INSERT SimpleSettings sp_lookup` | InsertData.sql |
| Soft Delete | `IsDeleted = 1, Status = 'Deleted'` | *.sql |
| Pagination | `OFFSET/FETCH NEXT` | Stored Procedures/*.sql |

---

## Detection Commands

### Scan Files for Patterns

```markdown
For each file in implementation:
1. Determine file type (ts/cs/sql)
2. Load relevant patterns from pattern-signatures.json
3. Apply regex patterns
4. Record matches and violations
```

### Check FE Composable

```markdown
SCAN: src/composables/use{Feature}.ts

PATTERNS TO CHECK:
[ ] Uses apis.* (not apiCalling.*)
[ ] Has ErrorCode === EnumApiErrorCode.Success
[ ] Has ref<IColumnsDynamic[]>
[ ] Has useToggleTableHeaderColumn

VIOLATIONS:
- If apiCalling.call* found → "Use apis.* wrapper instead"
- If no ErrorCode check → "Add ErrorCode check before Data access"
```

### Check FE Model

```markdown
SCAN: src/models/{module}/*.ts

PATTERNS TO CHECK:
[ ] Has I{Feature}Request interface
[ ] Has I{Feature}Response interface
[ ] Has {Feature}Model class
[ ] Has ForDisplay getters

VIOLATIONS:
- If no ForDisplay getter → "Add get {Prop}ForDisplay() for formatted values"
```

### Check BE Controller

```markdown
SCAN: Controllers/{Module}/{Feature}Controller.cs

PATTERNS TO CHECK:
[ ] Has [RoutePrefix("api/{feature}")]
[ ] Has [JwtFilter, LogFilter, AdminInfoFilter]
[ ] Has v1 and v2 endpoints
[ ] Uses req.GetUserData() for v2

VIOLATIONS:
- If missing filters → "Add standard filters: JwtFilter, LogFilter, AdminInfoFilter"
```

### Check SP Naming

```markdown
SCAN: Stored Procedures/*.sql

PATTERNS TO CHECK:
[ ] Name matches: {Service}_{Feature}_{Action}_X.Y.Z
[ ] Registered in InsertData.sql

VIOLATIONS:
- If no version → "Add version suffix: _1.0.0"
- If not registered → "Add to InsertData.sql sp_lookup"
```

---

## Violation Handling

### Severity Levels

| Level | Action | Example |
|-------|--------|---------|
| **Error** | Must fix before commit | SP without version |
| **Warning** | Should fix | Missing ForDisplay getter |
| **Info** | Optional improvement | Could use newer pattern |

### Auto-Fix Suggestions

```markdown
VIOLATION: apiCalling.callGetList in composable
SEVERITY: Error
FIX: Replace with apis.getList()

VIOLATION: SP name "PlayerNote_Create" missing version
SEVERITY: Error
FIX: Rename to "Coloris_PlayerNote_Create_1.0.0"

VIOLATION: Model missing ForDisplay getter
SEVERITY: Warning
FIX: Add "get AmountForDisplay() { return getValueForDisplay(this.Amount, this.Currency) }"
```

---

## Learning Hooks

### After Implementation Complete

```markdown
1. Scan all generated files
2. Extract patterns used
3. Compare with pattern-signatures.json
4. If new pattern found (3+ occurrences):
   - Flag for review
   - Suggest adding to signatures
5. Update learnings.json statistics:
   - patterns_detected++
   - Update usage_count for each pattern
   - Record last_used timestamp
```

### Pattern Learning Entry

```json
{
  "pattern_name": "new_discovered_pattern",
  "discovered_in": "feature_implementation",
  "occurrences": 3,
  "confidence": 0.92,
  "signature": "detected regex pattern",
  "status": "pending_review"
}
```

---

## Statistics Tracking

### Update After Each Feature

```json
{
  "statistics": {
    "features_analyzed": 5,
    "patterns_detected": 8,
    "patterns_matched": {
      "fe.api_wrapper": 5,
      "fe.forDisplay_getter": 12,
      "db.sp_versioning": 8
    },
    "violations_found": 3,
    "violations_fixed": 2,
    "avg_pattern_adherence": 0.94,
    "last_scan": "2025-01-08T12:00:00Z"
  }
}
```

---

## Integration with Agents

### Pre-Implementation Check

```markdown
Before generating code:
1. Load pattern-signatures.json
2. Show agent which patterns to follow
3. Provide examples from signatures
```

### Post-Implementation Validation

```markdown
After code generation:
1. Run pattern detection
2. Check all required patterns present
3. Report violations
4. Agent fixes violations
5. Re-run until clean
```

### Checkpoint Example

```markdown
FE COMPOSABLE CHECKPOINT:
✓ Pattern: api_wrapper (detected)
✓ Pattern: error_handling (detected)
✓ Pattern: dynamic_columns (detected)
✗ Violation: apiCalling used directly (line 45)
  Fix: Replace apiCalling.callGetList with apis.getList

Score: 3/4 patterns, 1 violation
Status: NEEDS_FIX
```

---

## Quick Reference

### Run Detection

```markdown
After implementation, run:
1. "Check patterns in generated files"
2. "Validate against pattern-signatures.json"
3. "Report violations and fixes"
4. "Update learnings.json statistics"
```

### Pattern Files

| File | Purpose |
|------|---------|
| `pattern-signatures.json` | Regex patterns and examples |
| `code-generation.md` | Generate stubs from patterns |
| `.memory/learnings.json` | Pattern statistics |
