# CRUD Feature Playbook

Step-by-step guide for implementing a standard CRUD feature across all layers.

## Pre-Implementation

### 1. Gather Requirements

- [ ] Feature name (e.g., PlayerNote)
- [ ] Which layers needed (DB, BE, FE)
- [ ] Load domain knowledge from `docs/backoffice/`

### 2. Define API Contract

```json
{
  "feature": "PlayerNote",
  "endpoints": [
    { "action": "GetList", "method": "POST", "route": "/player-note/get-list" },
    { "action": "Create", "method": "POST", "route": "/player-note/create" },
    { "action": "Update", "method": "POST", "route": "/player-note/update" },
    { "action": "Delete", "method": "POST", "route": "/player-note/delete" }
  ]
}
```

---

## Database Layer (Monika)

### 1. Create Table

**Location:** `{Schema}/dbo/Tables/`

```sql
CREATE TABLE [dbo].[PlayerNote] (
  [Id] int IDENTITY(1,1) PRIMARY KEY,
  [WebId] int NOT NULL,
  [CustomerId] int NOT NULL,
  [Note] nvarchar(1000) NOT NULL,
  [Status] nvarchar(50) NOT NULL DEFAULT 'Active',
  [IsDeleted] bit NOT NULL DEFAULT 0,
  [CreatedOn] datetime NOT NULL DEFAULT GETUTCDATE(),
  [CreatedBy] nvarchar(100) NOT NULL,
  [ModifiedOn] datetime NOT NULL DEFAULT GETUTCDATE(),
  [ModifiedBy] nvarchar(100) NOT NULL
)

CREATE NONCLUSTERED INDEX IX_PlayerNote_WebId_CustomerId
ON [dbo].[PlayerNote] ([WebId], [CustomerId]) WHERE [IsDeleted] = 0
```

### 2. Create Stored Procedures

**Location:** `{Schema}/dbo/Stored Procedures/`

**Naming:** `Coloris_PlayerNote_{Action}_1.0.0`

- `Coloris_PlayerNote_GetList_1.0.0`
- `Coloris_PlayerNote_Create_1.0.0`
- `Coloris_PlayerNote_Update_1.0.0`
- `Coloris_PlayerNote_Delete_1.0.0`

### 3. Register in InsertData.sql

**Location:** `ApplicationSetting/Data/InsertData.sql`

```sql
INSERT [dbo].[SimpleSettings]
  ([Website], [IsUAT], [Type], [Id], [Value], [Decription], [PossibleValues], [Responsible], [ModifiedBy], [ModifiedOn], [IsEnabled])
VALUES
  (N'Coloris', 0, N'sp_lookup', N'Coloris_PlayerNote_GetList', N'[dbo].[Coloris_PlayerNote_GetList_1.0.0]', N'', N'', N'Developer', N'Rithy', GETDATE(), 1),
  (N'Coloris', 0, N'sp_lookup', N'Coloris_PlayerNote_Create', N'[dbo].[Coloris_PlayerNote_Create_1.0.0]', N'', N'', N'Developer', N'Rithy', GETDATE(), 1),
  (N'Coloris', 0, N'sp_lookup', N'Coloris_PlayerNote_Update', N'[dbo].[Coloris_PlayerNote_Update_1.0.0]', N'', N'', N'Developer', N'Rithy', GETDATE(), 1),
  (N'Coloris', 0, N'sp_lookup', N'Coloris_PlayerNote_Delete', N'[dbo].[Coloris_PlayerNote_Delete_1.0.0]', N'', N'', N'Developer', N'Rithy', GETDATE(), 1)
```

---

## Backend Layer (Coloris)

### 1. Create Models

**Location:** `Models/PlayerNote/`

- `PlayerNoteRequest.cs` - Request models
- `PlayerNoteResponse.cs` - Response models

### 2. Create Repository

**Location:** `Repository/`

- `IPlayerNoteRepository.cs` - Interface
- `PlayerNoteRepository.cs` - Implementation with Dapper

### 3. Create Service

**Location:** `Services/`

- `IPlayerNoteService.cs` - Interface
- `PlayerNoteService.cs` - Business logic

### 4. Create Controller

**Location:** `Controllers/PlayerNote/`

- `PlayerNoteController.cs` - API endpoints

### 5. Register in DI

**Location:** `App_Start/UnityConfig.cs`

```csharp
container.RegisterType<IPlayerNoteRepository, PlayerNoteRepository>();
container.RegisterType<IPlayerNoteService, PlayerNoteService>();
```

---

## Frontend Layer (Kirby)

### 1. Create Model

**Location:** `src/models/playerNote/PlayerNote.ts`

```typescript
export interface IPlayerNoteRequest { ... }
export interface IPlayerNoteResponse { ... }
export class IPlayerNoteItem {
  // With ForDisplay getters
}
```

### 2. Add to apiCalling.ts

**Location:** `src/libraries/apiCalling.ts`

```typescript
callGetPlayerNoteList(request): IAxiosPromise<Response> {
  if (useRealApi) {
    return api.post('/back-office/v2/player-note/get-list', request);
  }
  return FakeAPI.getPlayerNoteList(request);
},
```

### 3. Add to apis.ts

**Location:** `src/libraries/apis.ts`

```typescript
getPlayerNoteList(request): Promise<ApiResponse<Response>> {
  return getResponse(apiCalling.callGetPlayerNoteList(request));
},
```

### 4. Create FakeData (Optional)

**Location:** `src/libraries/fakeData/fakePlayerNote.ts`

### 5. Create Composable

**Location:** `src/composables/playerNote/usePlayerNote.ts`

Follow CustomerRecordReport pattern.

### 6. Create Component

**Location:** `src/views/desktop/playerNote/PlayerNoteList.vue`

### 7. Add Router Entry

**Location:** `src/router/index.ts`

```typescript
{
  path: '/membership/player-note',
  name: 'PlayerNoteList',
  component: () => import('@/views/desktop/playerNote/PlayerNoteList.vue'),
  beforeEnter: (to, from, next) => authorizeRoute('membership/player_note', next, {}, 'Membership'),
},
```

---

## Post-Implementation

### 1. Test Locally

```bash
# FE dev server
npm run serve:stg

# BE build
dotnet build
```

### 2. Update shared-memory.md

Document the feature implementation status.

### 3. Update learnings.json

Add any new patterns learned.
