# Frontend Testing Playbook v3.1.0

Test Kirby frontend components with Playwright and screenshot proof.

---

## Overview

| Tool | Purpose |
|------|---------|
| Playwright | Browser automation |
| Screenshots | Visual proof of test results |
| webapp-testing skill | Server management |

---

## Quick Start

### 1. Start Kirby Dev Server

```bash
cd /Users/rithytep_1/projects/Nuxt/kirby
npm run serve:stg
```

### 2. Run Test Script

```bash
python /Users/rithytep_1/.claude/skills/backoffice-fullstack/scripts/fe-test-template.py \
  --url "http://localhost:8080/v2/feature-page" \
  --output "/tmp/fe-tests"
```

---

## Test Patterns

### Pattern 1: Page Load Test

Verify page loads correctly with data.

```python
from playwright.sync_api import sync_playwright

def test_page_load():
    with sync_playwright() as p:
        browser = p.chromium.launch(headless=True)
        page = browser.new_page()

        # Navigate to page
        page.goto('http://localhost:8080/v2/feature-page')
        page.wait_for_load_state('networkidle')

        # Screenshot proof
        page.screenshot(path='/tmp/page-load.png', full_page=True)

        # Verify elements exist
        assert page.locator('.el-table').is_visible()
        assert page.locator('.filter-form').is_visible()

        print('Page load test PASSED')
        browser.close()

test_page_load()
```

### Pattern 2: Filter Form Test

Test filter form submission.

```python
from playwright.sync_api import sync_playwright

def test_filter_form():
    with sync_playwright() as p:
        browser = p.chromium.launch(headless=True)
        page = browser.new_page()

        page.goto('http://localhost:8080/v2/feature-page')
        page.wait_for_load_state('networkidle')

        # Screenshot before
        page.screenshot(path='/tmp/filter-before.png')

        # Fill filter form
        page.fill('input[placeholder="Username"]', 'testuser')
        page.click('button:has-text("Search")')

        # Wait for table to update
        page.wait_for_load_state('networkidle')

        # Screenshot after
        page.screenshot(path='/tmp/filter-after.png')

        print('Filter form test PASSED')
        browser.close()

test_filter_form()
```

### Pattern 3: Table Data Test

Verify table displays data correctly.

```python
from playwright.sync_api import sync_playwright

def test_table_data():
    with sync_playwright() as p:
        browser = p.chromium.launch(headless=True)
        page = browser.new_page()

        page.goto('http://localhost:8080/v2/feature-page')
        page.wait_for_load_state('networkidle')

        # Get table rows
        rows = page.locator('.el-table__body-wrapper tr').all()
        row_count = len(rows)

        # Screenshot table
        page.screenshot(path='/tmp/table-data.png')

        # Verify data loaded
        assert row_count > 0, f'Expected rows, got {row_count}'
        print(f'Table data test PASSED - {row_count} rows')

        browser.close()

test_table_data()
```

### Pattern 4: Column Toggle Test

Test dynamic column visibility.

```python
from playwright.sync_api import sync_playwright

def test_column_toggle():
    with sync_playwright() as p:
        browser = p.chromium.launch(headless=True)
        page = browser.new_page()

        page.goto('http://localhost:8080/v2/feature-page')
        page.wait_for_load_state('networkidle')

        # Screenshot before toggle
        page.screenshot(path='/tmp/columns-before.png')

        # Click column toggle button
        page.click('.column-toggle-btn')
        page.wait_for_selector('.column-toggle-dropdown')

        # Toggle a column off
        page.click('.column-toggle-dropdown .el-checkbox:first-child')

        # Screenshot after toggle
        page.screenshot(path='/tmp/columns-after.png')

        print('Column toggle test PASSED')
        browser.close()

test_column_toggle()
```

### Pattern 5: Export Test

Test data export functionality.

```python
from playwright.sync_api import sync_playwright
import os

def test_export():
    with sync_playwright() as p:
        browser = p.chromium.launch(headless=True)
        page = browser.new_page()

        page.goto('http://localhost:8080/v2/feature-page')
        page.wait_for_load_state('networkidle')

        # Screenshot before export
        page.screenshot(path='/tmp/export-before.png')

        # Click export button
        with page.expect_download() as download_info:
            page.click('button:has-text("Export")')

        download = download_info.value
        download.save_as(f'/tmp/{download.suggested_filename}')

        # Verify file exists
        assert os.path.exists(f'/tmp/{download.suggested_filename}')

        print(f'Export test PASSED - saved {download.suggested_filename}')
        browser.close()

test_export()
```

### Pattern 6: Form Validation Test

Test form validation messages.

```python
from playwright.sync_api import sync_playwright

def test_form_validation():
    with sync_playwright() as p:
        browser = p.chromium.launch(headless=True)
        page = browser.new_page()

        page.goto('http://localhost:8080/v2/feature-form')
        page.wait_for_load_state('networkidle')

        # Submit empty form
        page.click('button:has-text("Submit")')

        # Wait for validation messages
        page.wait_for_selector('.el-form-item__error')

        # Screenshot validation errors
        page.screenshot(path='/tmp/validation-errors.png')

        # Verify error messages
        errors = page.locator('.el-form-item__error').all()
        assert len(errors) > 0, 'Expected validation errors'

        print(f'Form validation test PASSED - {len(errors)} errors shown')
        browser.close()

test_form_validation()
```

---

## Element Plus Selectors

Common selectors for Kirby components:

| Component | Selector |
|-----------|----------|
| Table | `.el-table` |
| Table Row | `.el-table__body-wrapper tr` |
| Table Cell | `.el-table__body-wrapper td` |
| Button | `.el-button` or `button:has-text("Label")` |
| Input | `.el-input__inner` |
| Select | `.el-select` |
| Dropdown | `.el-select-dropdown__item` |
| Date Picker | `.el-date-editor` |
| Checkbox | `.el-checkbox` |
| Pagination | `.el-pagination` |
| Dialog | `.el-dialog` |
| Form Error | `.el-form-item__error` |
| Loading | `.el-loading-mask` |

---

## Screenshot Locations

Save screenshots to `/tmp/` or skill memory:

```python
# Temp location
page.screenshot(path='/tmp/test-result.png')

# Skill memory (persisted)
page.screenshot(path='/Users/rithytep_1/.claude/skills/backoffice-fullstack/.memory/screenshots/test.png')
```

---

## Test Report Template

After running tests, generate report:

```markdown
## FE Test Report - {Feature Name}

**Date:** {timestamp}
**URL:** {page_url}
**Status:** PASSED / FAILED

### Tests Run
| Test | Result | Screenshot |
|------|--------|------------|
| Page Load | PASSED | [page-load.png] |
| Filter Form | PASSED | [filter-after.png] |
| Table Data | PASSED (25 rows) | [table-data.png] |
| Export | PASSED | [export.xlsx downloaded] |

### Screenshots
- `/tmp/page-load.png`
- `/tmp/filter-after.png`
- `/tmp/table-data.png`
```

---

## Integration with Parallel Execution

Add FE testing as Phase 4 after implementation:

```
Phase 1: DB (Monika)
Phase 2: BE + FE API (Parallel)
Phase 3: FE Integration (Kirby)
Phase 4: FE Testing (Playwright) ← NEW
```

### Status Update

```json
{
  "fe": {
    "component": "completed",
    "router": "completed",
    "testing": "in_progress"
  }
}
```

---

## Quick Commands

```bash
# Run single test
python -c "
from playwright.sync_api import sync_playwright
with sync_playwright() as p:
    browser = p.chromium.launch(headless=True)
    page = browser.new_page()
    page.goto('http://localhost:8080/v2/feature-page')
    page.wait_for_load_state('networkidle')
    page.screenshot(path='/tmp/quick-test.png', full_page=True)
    print('Screenshot saved to /tmp/quick-test.png')
    browser.close()
"

# View screenshot (macOS)
open /tmp/quick-test.png
```
