# Report Feature Playbook

## Overview

This playbook covers implementing report pages in the WL Backoffice system. Reports typically display paginated data with filters, date ranges, and export capabilities.

## Architecture

```
┌─────────────┐     ┌─────────────┐     ┌─────────────┐     ┌─────────────┐
│   Kirby     │────▶│   Coloris   │────▶│   Monika    │────▶│   Data      │
│   (FE)      │     │   (BE API)  │     │   (SQL DB)  │     │   (Tables)  │
└─────────────┘     └─────────────┘     └─────────────┘     └─────────────┘
     │                    │                   │
     ▼                    ▼                   ▼
  Vue Page            Controller            SP GetList
  + Composable        + Service             + Pagination
```

## Implementation Steps

### 1. Database Layer (Monika)

**Create GetList Stored Procedure:**

```sql
-- File: Coloris_{Feature}Report_GetList_1.0.0.sql
CREATE PROCEDURE [dbo].[Coloris_{Feature}Report_GetList_1.0.0]
    @webId INT,
    @operatorId INT,
    @page INT,
    @rowCountPerPage INT,
    @dateFrom DATETIME,
    @dateTo DATETIME,
    @keyword NVARCHAR(100) = NULL,
    @status NVARCHAR(20) = NULL
AS
BEGIN
    SET NOCOUNT ON;

    -- Validate operator
    -- Build result with filters
    -- Return paginated data with TotalCount

    SELECT 0 AS ErrorCode

    SELECT
        [Columns],
        TotalCount = COUNT(1) OVER(),
        MaxPage = CASE
            WHEN COUNT(1) OVER() % @rowCountPerPage = 0
            THEN COUNT(1) OVER() / @rowCountPerPage
            ELSE COUNT(1) OVER() / @rowCountPerPage + 1
        END
    FROM [dbo].[{Feature}] WITH (NOLOCK)
    WHERE [WebId] = @webId
    AND [CreatedOn] BETWEEN @dateFrom AND @dateTo
    ORDER BY [CreatedOn] DESC
    OFFSET (@page - 1) * @rowCountPerPage ROWS
    FETCH NEXT @rowCountPerPage ROWS ONLY
END
```

**Register in InsertData.sql:**

```sql
,(N'Coloris', 0, N'sp_lookup', N'Coloris_{Feature}Report_GetList', N'[dbo].[Coloris_{Feature}Report_GetList_1.0.0]', N'', N'', N'Developer', N'{Author}', GETDATE(), 1)
```

### 2. Backend Layer (Coloris)

**Models (Models/{Feature}Report/):**

```csharp
// Request
public class {Feature}ReportListRequest : IBoRequest
{
    public int WebId { get; set; }
    public int OperatorId { get; set; }
    public int SimulateId { get; set; }
    public int PageIndex { get; set; } = 1;
    public int PageSize { get; set; } = 25;
    public DateTime DateFrom { get; set; }
    public DateTime DateTo { get; set; }
    public string Keyword { get; set; }
    public string Status { get; set; }
}

// Response
public class {Feature}ReportListResponse : BaseResponse
{
    public List<{Feature}ReportItem> Data { get; set; }
    public int TotalCount { get; set; }
}
```

**Repository:**

```csharp
public {Feature}ReportListResponse GetList({Feature}ReportListRequest req)
{
    return GetData<{Feature}ReportListResponse>("Coloris_{Feature}Report_GetList", new
    {
        req.WebId,
        req.OperatorId,
        Page = req.PageIndex,
        RowCountPerPage = req.PageSize,
        req.DateFrom,
        req.DateTo,
        req.Keyword,
        req.Status
    }).FirstOrDefault();
}
```

**Controller:**

```csharp
[RoutePrefix("api/back-office/{feature}-report"), LogFilter, JwtFilter, AdminInfoFilter]
public class {Feature}ReportController : BaseApiController
{
    [HttpPost]
    [Route("get-list")]
    [Route("v2/get-list")]
    public ApiResponse<{Feature}ReportListResponse> GetList({Feature}ReportListRequest req)
    {
        // v2 JWT population
        // Validation
        // Return service result
    }
}
```

### 3. Frontend Layer (Kirby)

**API Definition (apis/{feature}Report.ts):**

```typescript
import apiCalling from './apiCalling'

const BASE_URL = '/api/back-office/{feature}-report'

export const {feature}ReportApi = {
  getList: (params: {Feature}ReportListRequest) =>
    apiCalling<{Feature}ReportListResponse>({
      url: `${BASE_URL}/v2/get-list`,
      method: 'POST',
      data: params
    })
}
```

**Composable (composables/use{Feature}Report.ts):**

```typescript
import { ref, reactive } from 'vue'
import { {feature}ReportApi } from '@/apis/{feature}Report'

export function use{Feature}Report() {
  const loading = ref(false)
  const data = ref<{Feature}ReportItem[]>([])
  const pagination = reactive({
    pageIndex: 1,
    pageSize: 25,
    totalCount: 0
  })

  const filters = reactive({
    dateFrom: new Date(),
    dateTo: new Date(),
    keyword: '',
    status: ''
  })

  async function fetchList() {
    loading.value = true
    try {
      const response = await {feature}ReportApi.getList({
        pageIndex: pagination.pageIndex,
        pageSize: pagination.pageSize,
        ...filters
      })
      if (response.errorCode === 0) {
        data.value = response.data
        pagination.totalCount = response.totalCount
      }
    } finally {
      loading.value = false
    }
  }

  function handlePageChange(page: number) {
    pagination.pageIndex = page
    fetchList()
  }

  return {
    loading,
    data,
    pagination,
    filters,
    fetchList,
    handlePageChange
  }
}
```

**Page Component (pages/{feature}Report/index.vue):**

```vue
<template>
  <CustomerRecordReport
    :loading="loading"
    :columns="columns"
    :data="data"
    :pagination="pagination"
    @page-change="handlePageChange"
  >
    <template #filters>
      <DateRangePicker v-model="filters.dateRange" />
      <SearchInput v-model="filters.keyword" @search="fetchList" />
      <StatusFilter v-model="filters.status" @change="fetchList" />
    </template>

    <template #actions>
      <ExportButton @click="handleExport" />
    </template>
  </CustomerRecordReport>
</template>

<script setup lang="ts">
import { onMounted } from 'vue'
import { use{Feature}Report } from '@/composables/use{Feature}Report'

const {
  loading,
  data,
  pagination,
  filters,
  fetchList,
  handlePageChange
} = use{Feature}Report()

onMounted(() => {
  fetchList()
})
</script>
```

## Common Report Components

| Component | Usage |
|-----------|-------|
| `CustomerRecordReport` | Main report layout with table |
| `DateRangePicker` | Date range filter |
| `SearchInput` | Keyword search |
| `StatusFilter` | Status dropdown |
| `ExportButton` | CSV/Excel export |
| `Pagination` | Page navigation |

## Best Practices

1. **Date Filters**: Always require date range to limit data
2. **Pagination**: Default 25 rows, max 100
3. **Indexes**: Add indexes for filter columns in DB
4. **Caching**: Consider caching for frequently accessed reports
5. **Export**: Limit export rows (e.g., max 10,000)

## Testing Checklist

- [ ] API returns correct ErrorCode
- [ ] Pagination works correctly
- [ ] Date filters work
- [ ] Keyword search works
- [ ] Status filter works
- [ ] Empty state displays properly
- [ ] Loading state shows spinner
- [ ] Export downloads correctly
