# Settings Feature Playbook

## Overview

This playbook covers implementing settings/configuration pages in the WL Backoffice system. Settings features typically involve viewing and updating configuration values for a website.

## Architecture

```
┌─────────────┐     ┌─────────────┐     ┌─────────────┐
│   Kirby     │────▶│   Coloris   │────▶│   Monika    │
│   (FE)      │     │   (BE API)  │     │   (SQL DB)  │
└─────────────┘     └─────────────┘     └─────────────┘
     │                    │                   │
     ▼                    ▼                   ▼
  Vue Form            Controller          SP Get/Upsert
  + Validation        + Service           + Audit Trail
```

## Implementation Steps

### 1. Database Layer (Monika)

**Settings Table (if new):**

```sql
CREATE TABLE [dbo].[{Feature}Settings]
(
    [Id] INT IDENTITY(1, 1) NOT NULL,
    [WebId] INT NOT NULL,
    [SettingKey] NVARCHAR(100) NOT NULL,
    [SettingValue] NVARCHAR(MAX) NULL,
    [IsEnabled] BIT NOT NULL DEFAULT 1,
    [CreatedBy] NVARCHAR(50) NOT NULL,
    [CreatedOn] DATETIME NOT NULL DEFAULT GETDATE(),
    [ModifiedBy] NVARCHAR(50) NOT NULL,
    [ModifiedOn] DATETIME NOT NULL DEFAULT GETDATE(),
    CONSTRAINT [PK_{Feature}Settings] PRIMARY KEY CLUSTERED ([Id] ASC)
)

CREATE UNIQUE INDEX [IX_{Feature}Settings_WebId_Key]
ON [dbo].[{Feature}Settings] ([WebId], [SettingKey])
```

**Get Settings SP:**

```sql
CREATE PROCEDURE [dbo].[Coloris_{Feature}Settings_Get_1.0.0]
    @webId INT
AS
BEGIN
    SET NOCOUNT ON;

    SELECT 0 AS ErrorCode

    SELECT
        [Id],
        [WebId],
        [SettingKey],
        [SettingValue],
        [IsEnabled],
        [ModifiedOn],
        [ModifiedBy]
    FROM [dbo].[{Feature}Settings] WITH (NOLOCK)
    WHERE [WebId] = @webId
END
```

**Upsert Settings SP (Insert or Update):**

```sql
CREATE PROCEDURE [dbo].[Coloris_{Feature}Settings_Upsert_1.0.0]
    @webId INT,
    @operatorId INT,
    @settingKey NVARCHAR(100),
    @settingValue NVARCHAR(MAX),
    @isEnabled BIT = 1
AS
BEGIN
    SET NOCOUNT ON;

    -- Validate operator
    DECLARE @operator NVARCHAR(50) = ''
    SELECT @operator = Username FROM Customer WITH (NOLOCK)
    WHERE WebId = @webId AND CustomerId = @operatorId
    AND (AccountType = 7 OR AccountType = 8) AND AgtId = 0

    IF (@operator = '')
    BEGIN
        SELECT 108 AS ErrorCode, 'Invalid Operator Id' AS ErrorMessage
        RETURN
    END

    -- Upsert pattern
    IF EXISTS (
        SELECT 1 FROM [dbo].[{Feature}Settings]
        WHERE [WebId] = @webId AND [SettingKey] = @settingKey
    )
    BEGIN
        UPDATE [dbo].[{Feature}Settings]
        SET [SettingValue] = @settingValue,
            [IsEnabled] = @isEnabled,
            [ModifiedBy] = @operator,
            [ModifiedOn] = GETDATE()
        WHERE [WebId] = @webId AND [SettingKey] = @settingKey
    END
    ELSE
    BEGIN
        INSERT INTO [dbo].[{Feature}Settings]
        ([WebId], [SettingKey], [SettingValue], [IsEnabled], [CreatedBy], [CreatedOn], [ModifiedBy], [ModifiedOn])
        VALUES
        (@webId, @settingKey, @settingValue, @isEnabled, @operator, GETDATE(), @operator, GETDATE())
    END

    SELECT 0 AS ErrorCode, 'Success' AS ErrorMessage
END
```

### 2. Backend Layer (Coloris)

**Models:**

```csharp
// Get Request
public class {Feature}SettingsGetRequest : IBoRequest
{
    public int WebId { get; set; }
    public int OperatorId { get; set; }
    public int SimulateId { get; set; }
}

// Get Response
public class {Feature}SettingsResponse : BaseResponse
{
    public {Feature}SettingsData Data { get; set; }
}

public class {Feature}SettingsData
{
    public bool IsEnabled { get; set; }
    public string Option1 { get; set; }
    public string Option2 { get; set; }
    public int MaxLimit { get; set; }
    // Add settings fields...
}

// Update Request
public class {Feature}SettingsUpdateRequest : IBoRequest
{
    public int WebId { get; set; }
    public int OperatorId { get; set; }
    public int SimulateId { get; set; }
    public bool IsEnabled { get; set; }
    public string Option1 { get; set; }
    public string Option2 { get; set; }
    public int MaxLimit { get; set; }
}
```

**Service:**

```csharp
public class {Feature}SettingsService : I{Feature}SettingsService
{
    public {Feature}SettingsResponse Get({Feature}SettingsGetRequest req)
    {
        var settings = _repository.GetSettings(req.WebId);

        // Map database rows to DTO
        return new {Feature}SettingsResponse
        {
            ErrorCode = 0,
            Data = new {Feature}SettingsData
            {
                IsEnabled = GetBoolSetting(settings, "IsEnabled", true),
                Option1 = GetStringSetting(settings, "Option1", ""),
                MaxLimit = GetIntSetting(settings, "MaxLimit", 100)
            }
        };
    }

    public BaseResponse Update({Feature}SettingsUpdateRequest req)
    {
        // Save each setting
        _repository.UpsertSetting(req.WebId, req.OperatorId, "IsEnabled", req.IsEnabled.ToString());
        _repository.UpsertSetting(req.WebId, req.OperatorId, "Option1", req.Option1);
        _repository.UpsertSetting(req.WebId, req.OperatorId, "MaxLimit", req.MaxLimit.ToString());

        return new BaseResponse(0);
    }
}
```

**Controller:**

```csharp
[RoutePrefix("api/back-office/{feature}-settings"), LogFilter, JwtFilter, AdminInfoFilter]
public class {Feature}SettingsController : BaseApiController
{
    [HttpPost]
    [Route("get")]
    [Route("v2/get")]
    public ApiResponse<{Feature}SettingsResponse> Get({Feature}SettingsGetRequest req)
    {
        // Populate WebId from JWT for v2
        return new ApiResponse<{Feature}SettingsResponse>(_service.Get(req));
    }

    [HttpPost]
    [Route("update")]
    [Route("v2/update")]
    public ApiResponse<BaseResponse> Update({Feature}SettingsUpdateRequest req)
    {
        // Validate and update
        return new ApiResponse<BaseResponse>(_service.Update(req));
    }
}
```

### 3. Frontend Layer (Kirby)

**API:**

```typescript
export const {feature}SettingsApi = {
  get: () =>
    apiCalling<{Feature}SettingsResponse>({
      url: '/api/back-office/{feature}-settings/v2/get',
      method: 'POST'
    }),

  update: (data: {Feature}SettingsUpdateRequest) =>
    apiCalling<BaseResponse>({
      url: '/api/back-office/{feature}-settings/v2/update',
      method: 'POST',
      data
    })
}
```

**Composable:**

```typescript
export function use{Feature}Settings() {
  const loading = ref(false)
  const saving = ref(false)
  const settings = reactive<{Feature}SettingsData>({
    isEnabled: true,
    option1: '',
    option2: '',
    maxLimit: 100
  })
  const originalSettings = ref<{Feature}SettingsData | null>(null)

  async function fetchSettings() {
    loading.value = true
    try {
      const response = await {feature}SettingsApi.get()
      if (response.errorCode === 0) {
        Object.assign(settings, response.data)
        originalSettings.value = { ...response.data }
      }
    } finally {
      loading.value = false
    }
  }

  async function saveSettings() {
    saving.value = true
    try {
      const response = await {feature}SettingsApi.update(settings)
      if (response.errorCode === 0) {
        ElMessage.success('Settings saved successfully')
        originalSettings.value = { ...settings }
      } else {
        ElMessage.error(response.errorMessage)
      }
    } finally {
      saving.value = false
    }
  }

  const hasChanges = computed(() => {
    if (!originalSettings.value) return false
    return JSON.stringify(settings) !== JSON.stringify(originalSettings.value)
  })

  function resetChanges() {
    if (originalSettings.value) {
      Object.assign(settings, originalSettings.value)
    }
  }

  return {
    loading,
    saving,
    settings,
    hasChanges,
    fetchSettings,
    saveSettings,
    resetChanges
  }
}
```

**Page Component:**

```vue
<template>
  <div class="settings-page">
    <PageHeader title="{Feature} Settings">
      <template #actions>
        <el-button @click="resetChanges" :disabled="!hasChanges">
          Reset
        </el-button>
        <el-button type="primary" @click="saveSettings" :loading="saving" :disabled="!hasChanges">
          Save Changes
        </el-button>
      </template>
    </PageHeader>

    <el-card v-loading="loading">
      <el-form :model="settings" label-width="200px">
        <el-form-item label="Enable Feature">
          <el-switch v-model="settings.isEnabled" />
        </el-form-item>

        <el-form-item label="Option 1">
          <el-input v-model="settings.option1" />
        </el-form-item>

        <el-form-item label="Max Limit">
          <el-input-number v-model="settings.maxLimit" :min="1" :max="1000" />
        </el-form-item>
      </el-form>
    </el-card>
  </div>
</template>

<script setup lang="ts">
import { onMounted } from 'vue'
import { use{Feature}Settings } from '@/composables/use{Feature}Settings'

const {
  loading,
  saving,
  settings,
  hasChanges,
  fetchSettings,
  saveSettings,
  resetChanges
} = use{Feature}Settings()

onMounted(() => {
  fetchSettings()
})
</script>
```

## Common Patterns

### Key-Value Storage Pattern

Store settings as key-value pairs for flexibility:

```typescript
// Multiple settings in one table
interface SettingRow {
  key: string
  value: string
}

// Mapped to typed DTO in service layer
interface SettingsDto {
  isEnabled: boolean    // key: "IsEnabled", value: "true"
  maxLimit: number      // key: "MaxLimit", value: "100"
  option: string        // key: "Option", value: "value"
}
```

### JSON Storage Pattern

Store complex settings as JSON:

```csharp
public class ComplexSettings
{
    public List<string> AllowedValues { get; set; }
    public Dictionary<string, int> Limits { get; set; }
}

// Store as JSON string in database
// Parse/serialize in service layer
```

## Best Practices

1. **Validation**: Validate settings before save (min/max, required, format)
2. **Audit Trail**: Track who changed what and when
3. **Defaults**: Always have sensible defaults
4. **Caching**: Consider caching settings (invalidate on update)
5. **Permissions**: Check user can modify settings
6. **Change Detection**: Show unsaved changes warning

## Testing Checklist

- [ ] Settings load correctly on page mount
- [ ] Default values shown for new websites
- [ ] Save button disabled when no changes
- [ ] Validation errors display correctly
- [ ] Save success message shows
- [ ] Reset button reverts changes
- [ ] Settings persist after page refresh
- [ ] Audit columns updated correctly
