# Validation Playbook

Complete guide for implementing validation across FE (Kirby), BE (Coloris), and DB (Monika).

## Validation Flow

```
User Input
    ↓
FE: Element Plus Form Rules → formHelper.getRules()
    ↓
API Call
    ↓
BE: [Required] annotations → Model.Validate() → ValidationHelper
    ↓
DB: SP validation → Status checks → Error codes
    ↓
Response with ErrorCode
    ↓
FE: Error display via notificationHelper
```

---

## FE Validation (Kirby)

### formHelper Library

**File:** `src/libraries/elementUiHelpers/formHelper.ts`

### Basic Rule Pattern

```typescript
import formHelper, { IRule } from '@/libraries/elementUiHelpers/formHelper'

const rules: Record<string, IRule> = {
  username: {
    required: true,
    isTriggerByChange: false,
  },
  amount: {
    required: true,
    isTriggerByChange: true,
    customRule: () => {
      if (formModel.amount <= 0) {
        return t('amount_must_be_positive')
      }
      return ''
    },
  },
}

const formRules = formHelper.getRules(rules)
const onSubmit = formHelper.getSubmitFunction(handleSubmit)
```

### IRule Interface

```typescript
interface IRule {
  required?: boolean
  isTriggerByChange?: boolean
  customRule?: () => string
}
```

### Built-in Validators

| Validator | Usage |
|-----------|-------|
| `checkDecimalInput()` | Prevents decimal values |
| `checkNumberInput()` | Integer-only, max 20 digits |
| `checkNumberWithTwoDecimalPoint()` | Currency format |
| `validateDate()` | Date range with today check |
| `checkUserPasswordValidation()` | Complex password rules |
| `checkPositiveValue()` | Positive number only |
| `checkConfirmPassword()` | Password confirmation |
| `checkNotRequiredEmail()` | Optional email format |
| `checkImageExtensionValidator()` | File type validation |
| `checkIllegalCharacters()` | Blocks `<>` characters |
| `checkIntegerValueValidator()` | Range 1-1000 integers |

### Form Component Integration

```vue
<template>
  <el-form
    ref="ruleFormRef"
    :model="formModel"
    :rules="formRules"
    label-width="158px"
  >
    <el-form-item :label="t('username')" prop="username">
      <el-input v-model="formModel.username" />
    </el-form-item>
    <el-form-item :label="t('amount')" prop="amount">
      <el-input-number v-model="formModel.amount" :min="0" />
    </el-form-item>
    <el-form-item>
      <el-button type="primary" @click="onSubmit(ruleFormRef)">
        {{ t('submit') }}
      </el-button>
    </el-form-item>
  </el-form>
</template>

<script setup lang="ts">
import { ref, reactive } from 'vue'
import type { FormInstance } from 'element-plus'
import formHelper, { IRule } from '@/libraries/elementUiHelpers/formHelper'
import { t } from '@/libraries/vue-i18n'

const ruleFormRef = ref<FormInstance>()
const formModel = reactive({
  username: '',
  amount: 0,
})

const rules: Record<string, IRule> = {
  username: { required: true },
  amount: {
    required: true,
    isTriggerByChange: true,
    customRule: () => formModel.amount <= 0 ? t('amount_must_be_positive') : '',
  },
}

const formRules = formHelper.getRules(rules)

const handleSubmit = async () => {
  // API call
}

const onSubmit = formHelper.getSubmitFunction(handleSubmit)
</script>
```

### Computed Validation

```typescript
const getDisabledDateOptions = computed(() => {
  const { FilterGameProvider, FilterRefNo } = filterFormModel
  if (!FilterGameProvider?.includes('All') || FilterRefNo) {
    return [AvialableValidation.MaxPassed3Months, AvialableValidation.Max60Days]
  }
  return [AvialableValidation.Only7Days]
})

watch(getDisabledDateOptions, () => {
  keyDateRange.value += 1
})
```

---

## BE Validation (Coloris)

### Data Annotations

```csharp
public class CreateFeatureRequest : IBoRequest
{
    public int WebId { get; set; }
    public int OperatorId { get; set; }

    [Required]
    [MaxLength(100)]
    public string Name { get; set; }

    [Required]
    [Range(0.01, double.MaxValue, ErrorMessage = "Amount must be positive")]
    public decimal Amount { get; set; }

    [EmailAddress]
    public string Email { get; set; }
}
```

### Model Validate Override

```csharp
public class WithdrawalRequest : BaseApiRequest
{
    [Required]
    public string Username { get; set; }

    [Required]
    public decimal Amount { get; set; }

    public override ApiReturnError Validate()
    {
        if (Amount <= 0)
            return ApiReturnError.AmountCannotBeZero;

        if (WebId < 0)
            return ApiReturnError.InvalidWebId;

        if (!Username.Trim().IsValidPlayerUsername())
            return ApiReturnError.InvalidUsernameFormat;

        if (IsContainInvalidCharacter(PlayerBankName))
            return ApiReturnError.ContainInvalidCharacter;

        return ApiReturnError.Success;
    }
}
```

### ValidationHelper Class

```csharp
public static class ValidationHelper
{
    private const int MaxUserNameLength = 20;
    private const int MinPasswordLength = 6;

    private static readonly Regex UserNameRegex = new Regex(@"^[0-9A-Za-z]+$");

    public static ApiReturnError BaseValidation(int webId)
    {
        return webId < 0 ? ApiReturnError.InvalidParameter : ApiReturnError.Success;
    }

    public static bool IsAmountGreaterThanZero(decimal amount)
    {
        return amount > 0;
    }

    public static bool IsValidEmail(string email)
    {
        string pattern = @"^[\w-\.]+@([\w-]+\.)+[\w-]{2,4}$";
        return new Regex(pattern).IsMatch(email);
    }

    public static bool IsValidPhoneNumber(string phoneNumber)
    {
        string pattern = @"^\+?[0-9\s-]+$";
        return new Regex(pattern).IsMatch(phoneNumber)
            && phoneNumber.Length >= 7
            && phoneNumber.Length <= 15;
    }
}
```

### Service Layer Validation

```csharp
public class FeatureService : IFeatureService
{
    public async Task<BaseResponse> CreateFeatureAsync(CreateFeatureRequest request)
    {
        var validationError = request.Validate();
        if (validationError != ApiReturnError.Success)
        {
            return new BaseResponse((int)validationError);
        }

        if (await _repository.ExistsAsync(request.Name))
        {
            return new BaseResponse((int)ApiReturnError.DuplicateExists);
        }

        return await _repository.CreateAsync(request);
    }
}
```

### Common Error Codes

| Code | Enum | Message |
|------|------|---------|
| 0 | Success | Success |
| 101 | InvalidParameter | Invalid Request Parameter |
| 102 | InvalidPasswordFormat | Password does not meet requirements |
| 104 | InvalidUsernameFormat | Invalid Username Format |
| 124 | ContainInvalidCharacter | Contain Invalid Character |
| 200 | WrongUserNameOrPassword | Wrong Username Or Password |
| 309 | OnlyAllowPositiveAmount | Only positive amount allowed |
| 312 | TransactionNotFound | No transaction found |
| 330 | WaitingOrIncompletePromotionExists | Promotion exists |
| 1001 | UserNotFound | User Not Found |
| 1005 | DuplicateUsername | Duplicate Username |
| 1006 | InsufficientBalance | Insufficient Balance |

---

## DB Validation (Monika)

### Lock-based Concurrency

```sql
DECLARE @resName NVARCHAR(100)
DECLARE @res INT
SET @resName = 'lockProcess' + CAST(@transactionId AS NVARCHAR(50))

EXEC @res = sp_getapplock
    @Resource = @resName,
    @LockMode = 'Exclusive',
    @LockOwner = 'Transaction',
    @LockTimeout = 5000

IF @res NOT IN (0, 1)
BEGIN
    SELECT -1 AS ErrorCode, 'Lock Error' AS ErrorMessage
    ROLLBACK
    RETURN
END
```

### Existence Check

```sql
IF NOT EXISTS(
    SELECT TOP 1 1
    FROM TransactionRequest WITH (NOLOCK)
    WHERE WebId = @webId
      AND Id = @transactionId
      AND TransactionStatus = @fromStatus)
BEGIN
    SELECT 312 AS ErrorCode,
           FORMATMESSAGE('No transaction found: %d', @transactionId) AS ErrorMessage
    ROLLBACK
    RETURN
END
```

### Status Validation

```sql
DECLARE @waiting INT = 1
DECLARE @verified INT = 2
DECLARE @approved INT = 3
DECLARE @rejected INT = 4

IF @fromStatus = @waiting AND @toStatus NOT IN (@verified, @rejected)
BEGIN
    SELECT 314 AS ErrorCode, 'Invalid status transition' AS ErrorMessage
    RETURN
END
```

### Amount Validation

```sql
IF @amount > @requestedAmount
BEGIN
    SELECT 316 AS ErrorCode,
           'Verified amount exceeds requested' AS ErrorMessage
    ROLLBACK
    RETURN
END

IF @amount <= 0
BEGIN
    SELECT 14008 AS ErrorCode,
           'Amount cannot be zero or negative' AS ErrorMessage
    RETURN
END
```

### Error Handling Pattern

```sql
BEGIN TRANSACTION

UPDATE [dbo].[Feature]
SET ModifiedOn = GETDATE(),
    Status = @toStatus
WHERE Id = @id AND WebId = @webId

IF @@ERROR <> 0 OR @@ROWCOUNT = 0
BEGIN
    ROLLBACK
    SELECT -1 AS ErrorCode, 'Update failed' AS ErrorMessage
    RETURN
END

COMMIT TRANSACTION
SELECT 0 AS ErrorCode, 'Success' AS ErrorMessage
```

### Duplicate Check

```sql
IF EXISTS(
    SELECT TOP 1 1
    FROM Feature WITH (NOLOCK)
    WHERE WebId = @webId AND Name = @name AND Id != @id)
BEGIN
    SELECT 330 AS ErrorCode, 'Duplicate name exists' AS ErrorMessage
    RETURN
END
```

---

## Error Code Mapping

### FE Error Display

```typescript
import EnumApiErrorCode from '@/models/enums/enumsApiErrorCode'
import notificationHelper from '@/libraries/elementUiHelpers/notificationHelper'
import EnumMessageType from '@/models/enums/enumMessageType'

const handleResponse = (response: any) => {
  if (response.ErrorCode === EnumApiErrorCode.Success) {
    notificationHelper.notification(t('success'), EnumMessageType.Success)
  } else {
    notificationHelper.notification(
      response.ErrorMessageForDisplay || t('error_occurred'),
      EnumMessageType.Error
    )
  }
}
```

### BE Error Response

```csharp
return new BaseResponse
{
    ErrorCode = (int)ApiReturnError.InvalidParameter,
    ErrorMessage = ApiReturnError.InvalidParameter.GetDescription(),
    ErrorMessageForDisplay = "Invalid request parameters"
};
```

---

## Validation Checklist

### FE
- [ ] Form rules with formHelper.getRules()
- [ ] Required fields marked
- [ ] Custom validation for business logic
- [ ] Error messages translated (i18n)
- [ ] Submit via formHelper.getSubmitFunction()

### BE
- [ ] Data annotations on request model
- [ ] Validate() override for complex rules
- [ ] Service layer validation before DB call
- [ ] Return appropriate ApiReturnError codes

### DB
- [ ] Lock mechanism for concurrent operations
- [ ] Existence checks before updates
- [ ] Status transition validation
- [ ] Amount/range validation
- [ ] Duplicate checks
- [ ] @@ERROR and @@ROWCOUNT verification
- [ ] Proper ROLLBACK on failure
