#!/usr/bin/env python3
"""
Backend API Test Template for Coloris
Tests API endpoints with JWT authentication and response validation.

Usage:
  python be-test-template.py --url "http://localhost/api/feature/v2/get-list" --method POST
  python be-test-template.py --url "http://localhost/api/feature/v2/create" --method POST --data '{"Name": "Test"}'
  python be-test-template.py --config /path/to/test-config.json
"""

import argparse
import json
import os
import sys
import time
from dataclasses import dataclass, asdict
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional, Any
import requests
from requests.exceptions import RequestException


@dataclass
class TestCase:
    name: str
    url: str
    method: str
    headers: Dict[str, str]
    body: Dict[str, Any]
    expected_status: int
    expected_error_code: int
    validate_response: Dict[str, Any]


@dataclass
class TestResult:
    name: str
    passed: bool
    status_code: int
    response_time_ms: float
    error_code: Optional[int]
    message: str
    response_body: Optional[Dict]


class BeApiTester:
    def __init__(self, base_url: str, jwt_token: Optional[str] = None):
        self.base_url = base_url.rstrip('/')
        self.jwt_token = jwt_token
        self.results: List[TestResult] = []

    def _get_headers(self, custom_headers: Optional[Dict] = None) -> Dict:
        headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json'
        }
        if self.jwt_token:
            headers['Authorization'] = f'Bearer {self.jwt_token}'
        if custom_headers:
            headers.update(custom_headers)
        return headers

    def test_endpoint(
        self,
        name: str,
        endpoint: str,
        method: str = 'POST',
        body: Optional[Dict] = None,
        expected_status: int = 200,
        expected_error_code: int = 0,
        validate_fields: Optional[Dict] = None,
        headers: Optional[Dict] = None
    ) -> TestResult:
        url = f'{self.base_url}{endpoint}'
        request_headers = self._get_headers(headers)

        start_time = time.time()
        try:
            if method.upper() == 'GET':
                response = requests.get(url, headers=request_headers, params=body, timeout=30)
            elif method.upper() == 'POST':
                response = requests.post(url, headers=request_headers, json=body, timeout=30)
            elif method.upper() == 'PUT':
                response = requests.put(url, headers=request_headers, json=body, timeout=30)
            elif method.upper() == 'DELETE':
                response = requests.delete(url, headers=request_headers, json=body, timeout=30)
            else:
                raise ValueError(f'Unsupported method: {method}')

            response_time = (time.time() - start_time) * 1000

            try:
                response_body = response.json()
            except json.JSONDecodeError:
                response_body = {'raw': response.text[:500]}

            error_code = response_body.get('ErrorCode')
            error_message = response_body.get('ErrorMessage', response_body.get('Message', ''))

            passed = True
            messages = []

            if response.status_code != expected_status:
                passed = False
                messages.append(f'Status: expected {expected_status}, got {response.status_code}')

            if error_code != expected_error_code:
                passed = False
                messages.append(f'ErrorCode: expected {expected_error_code}, got {error_code}')

            if validate_fields:
                for field, expected_value in validate_fields.items():
                    actual_value = self._get_nested_value(response_body, field)
                    if actual_value != expected_value:
                        passed = False
                        messages.append(f'{field}: expected {expected_value}, got {actual_value}')

            result = TestResult(
                name=name,
                passed=passed,
                status_code=response.status_code,
                response_time_ms=round(response_time, 2),
                error_code=error_code,
                message='; '.join(messages) if messages else 'OK',
                response_body=response_body
            )

        except RequestException as e:
            response_time = (time.time() - start_time) * 1000
            result = TestResult(
                name=name,
                passed=False,
                status_code=0,
                response_time_ms=round(response_time, 2),
                error_code=None,
                message=f'Request failed: {str(e)}',
                response_body=None
            )

        self.results.append(result)
        return result

    def _get_nested_value(self, data: Dict, path: str) -> Any:
        keys = path.split('.')
        value = data
        for key in keys:
            if isinstance(value, dict) and key in value:
                value = value[key]
            elif isinstance(value, list) and key.isdigit():
                value = value[int(key)]
            else:
                return None
        return value

    def run_test_suite(self, tests: List[TestCase]) -> List[TestResult]:
        for test in tests:
            self.test_endpoint(
                name=test.name,
                endpoint=test.url,
                method=test.method,
                body=test.body,
                expected_status=test.expected_status,
                expected_error_code=test.expected_error_code,
                validate_fields=test.validate_response,
                headers=test.headers
            )
        return self.results

    def print_results(self, verbose: bool = False) -> bool:
        print(f"\n{'='*60}")
        print(f"Backend API Test Results")
        print(f"{'='*60}")
        print(f"Base URL: {self.base_url}")
        print(f"Total Tests: {len(self.results)}")

        passed = sum(1 for r in self.results if r.passed)
        failed = len(self.results) - passed

        print(f"Passed: {passed}")
        print(f"Failed: {failed}")
        print(f"{'-'*60}")

        for result in self.results:
            status = "PASS" if result.passed else "FAIL"
            print(f"\n[{status}] {result.name}")
            print(f"  Status: {result.status_code} | Time: {result.response_time_ms}ms")
            if result.error_code is not None:
                print(f"  ErrorCode: {result.error_code}")
            print(f"  Message: {result.message}")

            if verbose and result.response_body:
                print(f"  Response: {json.dumps(result.response_body, indent=2)[:500]}")

        print(f"\n{'='*60}")
        status = "PASSED" if failed == 0 else "FAILED"
        print(f"Status: {status}")
        print(f"{'='*60}\n")

        return failed == 0

    def generate_report(self, output_path: str):
        timestamp = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        passed = sum(1 for r in self.results if r.passed)
        failed = len(self.results) - passed
        status = "PASSED" if failed == 0 else "FAILED"

        report = f"""# Backend API Test Report

**Generated:** {timestamp}
**Base URL:** {self.base_url}
**Status:** {'✅' if status == 'PASSED' else '❌'} {status}

---

## Summary

| Metric | Value |
|--------|-------|
| Total Tests | {len(self.results)} |
| Passed | {passed} |
| Failed | {failed} |
| Avg Response Time | {sum(r.response_time_ms for r in self.results) / len(self.results):.2f}ms |

---

## Test Results

| Test | Status | Time | ErrorCode | Message |
|------|--------|------|-----------|---------|
"""
        for r in self.results:
            status_icon = "✅" if r.passed else "❌"
            report += f"| {r.name} | {status_icon} | {r.response_time_ms}ms | {r.error_code} | {r.message[:50]} |\n"

        if failed > 0:
            report += "\n---\n\n## Failed Tests Details\n\n"
            for r in self.results:
                if not r.passed:
                    report += f"""### {r.name}

- **Status Code:** {r.status_code}
- **Error Code:** {r.error_code}
- **Message:** {r.message}
- **Response:** `{json.dumps(r.response_body)[:200] if r.response_body else 'N/A'}`

"""

        report += f"\n---\n\n*Generated by Backoffice Fullstack Skill v3.2.0*\n"

        os.makedirs(os.path.dirname(output_path) if os.path.dirname(output_path) else '.', exist_ok=True)
        with open(output_path, 'w') as f:
            f.write(report)


def create_default_tests(feature: str) -> List[TestCase]:
    return [
        TestCase(
            name=f"GetList - Valid Request",
            url=f"/api/{feature}/v2/get-list",
            method="POST",
            headers={},
            body={
                "WebId": 1,
                "Page": 1,
                "RowCountPerPage": 25
            },
            expected_status=200,
            expected_error_code=0,
            validate_response={}
        ),
        TestCase(
            name=f"GetList - Invalid WebId",
            url=f"/api/{feature}/v2/get-list",
            method="POST",
            headers={},
            body={
                "WebId": 0,
                "Page": 1,
                "RowCountPerPage": 25
            },
            expected_status=200,
            expected_error_code=400,
            validate_response={}
        ),
        TestCase(
            name=f"Create - Valid Request",
            url=f"/api/{feature}/v2/create",
            method="POST",
            headers={},
            body={
                "WebId": 1,
                "Name": f"Test {feature}",
                "Amount": 100.00,
                "Status": 1
            },
            expected_status=200,
            expected_error_code=0,
            validate_response={}
        )
    ]


def main():
    parser = argparse.ArgumentParser(description='Backend API Test Template for Coloris')
    parser.add_argument('--url', '-u', help='Full URL to test (single endpoint)')
    parser.add_argument('--base-url', '-b', default='http://localhost', help='Base URL for API')
    parser.add_argument('--method', '-m', default='POST', help='HTTP method')
    parser.add_argument('--data', '-d', help='Request body (JSON string)')
    parser.add_argument('--token', '-t', help='JWT token for authentication')
    parser.add_argument('--config', '-c', help='Path to test config JSON file')
    parser.add_argument('--feature', '-f', help='Feature name for default tests')
    parser.add_argument('--output', '-o', default='/tmp/be-test-report.md', help='Output path for report')
    parser.add_argument('--verbose', '-v', action='store_true', help='Show response bodies')

    args = parser.parse_args()

    tester = BeApiTester(args.base_url, args.token)

    if args.config:
        with open(args.config, 'r') as f:
            config = json.load(f)
        tests = [TestCase(**t) for t in config.get('tests', [])]
        tester.run_test_suite(tests)
    elif args.feature:
        tests = create_default_tests(args.feature)
        tester.run_test_suite(tests)
    elif args.url:
        body = json.loads(args.data) if args.data else {}
        tester.test_endpoint(
            name="Single Test",
            endpoint=args.url.replace(args.base_url, ''),
            method=args.method,
            body=body
        )
    else:
        print("Error: Provide --url, --feature, or --config")
        sys.exit(1)

    success = tester.print_results(args.verbose)
    tester.generate_report(args.output)
    print(f"Report saved to: {args.output}")

    sys.exit(0 if success else 1)


if __name__ == '__main__':
    main()
