#!/usr/bin/env python3
"""
Frontend Test Template for Kirby
Usage: python fe-test-template.py --url "http://localhost:8080/v2/page" --output "/tmp/screenshots"
"""

import argparse
import os
from datetime import datetime
from playwright.sync_api import sync_playwright


def run_tests(url: str, output_dir: str, headless: bool = True):
    """Run FE tests with screenshot proof."""

    os.makedirs(output_dir, exist_ok=True)
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    results = []

    with sync_playwright() as p:
        browser = p.chromium.launch(headless=headless)
        page = browser.new_page(viewport={'width': 1920, 'height': 1080})

        # Test 1: Page Load
        print('Test 1: Page Load...')
        try:
            page.goto(url)
            page.wait_for_load_state('networkidle')
            screenshot_path = f'{output_dir}/01_page_load_{timestamp}.png'
            page.screenshot(path=screenshot_path, full_page=True)
            results.append(('Page Load', 'PASSED', screenshot_path))
            print(f'  PASSED - Screenshot: {screenshot_path}')
        except Exception as e:
            results.append(('Page Load', 'FAILED', str(e)))
            print(f'  FAILED - {e}')

        # Test 2: Table Exists
        print('Test 2: Table Exists...')
        try:
            table = page.locator('.el-table')
            if table.is_visible():
                screenshot_path = f'{output_dir}/02_table_{timestamp}.png'
                table.screenshot(path=screenshot_path)
                row_count = len(page.locator('.el-table__body-wrapper tr').all())
                results.append(('Table Exists', f'PASSED ({row_count} rows)', screenshot_path))
                print(f'  PASSED - {row_count} rows - Screenshot: {screenshot_path}')
            else:
                results.append(('Table Exists', 'FAILED', 'Table not visible'))
                print('  FAILED - Table not visible')
        except Exception as e:
            results.append(('Table Exists', 'FAILED', str(e)))
            print(f'  FAILED - {e}')

        # Test 3: Filter Form
        print('Test 3: Filter Form...')
        try:
            filter_form = page.locator('.filter-form, .el-form').first
            if filter_form.is_visible():
                screenshot_path = f'{output_dir}/03_filter_form_{timestamp}.png'
                filter_form.screenshot(path=screenshot_path)
                results.append(('Filter Form', 'PASSED', screenshot_path))
                print(f'  PASSED - Screenshot: {screenshot_path}')
            else:
                results.append(('Filter Form', 'SKIPPED', 'No filter form found'))
                print('  SKIPPED - No filter form found')
        except Exception as e:
            results.append(('Filter Form', 'SKIPPED', str(e)))
            print(f'  SKIPPED - {e}')

        # Test 4: Pagination
        print('Test 4: Pagination...')
        try:
            pagination = page.locator('.el-pagination')
            if pagination.is_visible():
                screenshot_path = f'{output_dir}/04_pagination_{timestamp}.png'
                pagination.screenshot(path=screenshot_path)
                results.append(('Pagination', 'PASSED', screenshot_path))
                print(f'  PASSED - Screenshot: {screenshot_path}')
            else:
                results.append(('Pagination', 'SKIPPED', 'No pagination found'))
                print('  SKIPPED - No pagination found')
        except Exception as e:
            results.append(('Pagination', 'SKIPPED', str(e)))
            print(f'  SKIPPED - {e}')

        # Test 5: Full Page Screenshot
        print('Test 5: Full Page Screenshot...')
        try:
            screenshot_path = f'{output_dir}/05_full_page_{timestamp}.png'
            page.screenshot(path=screenshot_path, full_page=True)
            results.append(('Full Page', 'PASSED', screenshot_path))
            print(f'  PASSED - Screenshot: {screenshot_path}')
        except Exception as e:
            results.append(('Full Page', 'FAILED', str(e)))
            print(f'  FAILED - {e}')

        browser.close()

    # Generate Report
    report_path = f'{output_dir}/test_report_{timestamp}.md'
    generate_report(url, results, report_path, timestamp)
    print(f'\nReport saved to: {report_path}')

    return results


def generate_report(url: str, results: list, output_path: str, timestamp: str):
    """Generate markdown test report."""

    passed = sum(1 for r in results if 'PASSED' in r[1])
    failed = sum(1 for r in results if r[1] == 'FAILED')
    skipped = sum(1 for r in results if 'SKIPPED' in r[1])

    report = f"""# FE Test Report

**Date:** {timestamp}
**URL:** {url}
**Status:** {'PASSED' if failed == 0 else 'FAILED'}

## Summary
- Passed: {passed}
- Failed: {failed}
- Skipped: {skipped}

## Test Results

| Test | Result | Screenshot/Details |
|------|--------|-------------------|
"""

    for test_name, result, details in results:
        if 'PASSED' in result:
            report += f'| {test_name} | {result} | `{details}` |\n'
        else:
            report += f'| {test_name} | {result} | {details} |\n'

    report += """
## Screenshots

"""
    for test_name, result, details in results:
        if 'PASSED' in result and details.endswith('.png'):
            report += f'### {test_name}\n![{test_name}]({details})\n\n'

    with open(output_path, 'w') as f:
        f.write(report)


def main():
    parser = argparse.ArgumentParser(description='Frontend Test Template for Kirby')
    parser.add_argument('--url', required=True, help='Page URL to test (e.g., http://localhost:8080/v2/page)')
    parser.add_argument('--output', default='/tmp/fe-tests', help='Output directory for screenshots')
    parser.add_argument('--headed', action='store_true', help='Run in headed mode (visible browser)')

    args = parser.parse_args()

    print(f'Running FE tests for: {args.url}')
    print(f'Output directory: {args.output}')
    print('-' * 50)

    run_tests(args.url, args.output, headless=not args.headed)


if __name__ == '__main__':
    main()
