#!/usr/bin/env python3
"""
API Mock Server for Backoffice Fullstack Skill
Serves FakeAPI responses for FE development without BE.

Usage:
  python mock-server.py
  python mock-server.py --port 3000
  python mock-server.py --config /path/to/mock-config.json
"""

import argparse
import json
import os
import sys
from datetime import datetime
from pathlib import Path
from typing import Dict, Any, List

try:
    from flask import Flask, request, jsonify
    from flask_cors import CORS
except ImportError:
    print("Error: Flask and flask-cors required. Install with:")
    print("  pip install flask flask-cors")
    sys.exit(1)


app = Flask(__name__)
CORS(app)

# Default mock data storage
mock_data: Dict[str, List[Dict]] = {}
mock_config: Dict[str, Any] = {}


def load_mock_config(config_path: str) -> Dict:
    if os.path.exists(config_path):
        with open(config_path, 'r') as f:
            return json.load(f)
    return {}


def create_response(error_code: int = 0, message: str = "Success", data: Any = None) -> Dict:
    return {
        "ErrorCode": error_code,
        "ErrorMessage": message,
        "Message": message,
        "Data": data
    }


def create_list_response(items: List, page: int = 1, page_size: int = 25) -> Dict:
    total = len(items)
    start = (page - 1) * page_size
    end = start + page_size
    paged_items = items[start:end]

    return create_response(data={
        "List": paged_items,
        "TotalCount": total,
        "MaxPage": (total + page_size - 1) // page_size if total > 0 else 1
    })


# Generic CRUD endpoints
@app.route('/api/<feature>/v2/get-list', methods=['POST'])
def get_list(feature: str):
    data = request.get_json() or {}
    page = data.get('Page', 1)
    page_size = data.get('RowCountPerPage', 25)

    items = mock_data.get(feature, [])

    # Apply basic filtering
    if 'Status' in data and data['Status'] is not None:
        items = [i for i in items if i.get('Status') == data['Status']]

    if 'StartDate' in data and data['StartDate']:
        items = [i for i in items if i.get('CreatedOn', '') >= data['StartDate']]

    if 'EndDate' in data and data['EndDate']:
        items = [i for i in items if i.get('CreatedOn', '') <= data['EndDate']]

    return jsonify(create_list_response(items, page, page_size))


@app.route('/api/<feature>/v2/get-by-id', methods=['POST'])
def get_by_id(feature: str):
    data = request.get_json() or {}
    item_id = data.get('Id')

    items = mock_data.get(feature, [])
    item = next((i for i in items if i.get('Id') == item_id), None)

    if item:
        return jsonify(create_response(data=item))
    else:
        return jsonify(create_response(404, "Record not found"))


@app.route('/api/<feature>/v2/create', methods=['POST'])
def create(feature: str):
    data = request.get_json() or {}

    if feature not in mock_data:
        mock_data[feature] = []

    # Generate new ID
    existing_ids = [i.get('Id', 0) for i in mock_data[feature]]
    new_id = max(existing_ids) + 1 if existing_ids else 1

    new_item = {
        'Id': new_id,
        'WebId': data.get('WebId', 1),
        'Name': data.get('Name', ''),
        'Amount': data.get('Amount', 0),
        'Currency': data.get('Currency', 'USD'),
        'Status': data.get('Status', 1),
        'Remark': data.get('Remark', ''),
        'CreatedOn': datetime.utcnow().isoformat() + 'Z',
        'ModifiedOn': datetime.utcnow().isoformat() + 'Z'
    }

    mock_data[feature].append(new_item)

    return jsonify(create_response(data={'Id': new_id}))


@app.route('/api/<feature>/v2/update', methods=['POST'])
def update(feature: str):
    data = request.get_json() or {}
    item_id = data.get('Id')

    items = mock_data.get(feature, [])
    item = next((i for i in items if i.get('Id') == item_id), None)

    if not item:
        return jsonify(create_response(404, "Record not found"))

    # Update fields
    for key in ['Name', 'Amount', 'Currency', 'Status', 'Remark']:
        if key in data:
            item[key] = data[key]
    item['ModifiedOn'] = datetime.utcnow().isoformat() + 'Z'

    return jsonify(create_response())


@app.route('/api/<feature>/v2/delete', methods=['POST'])
def delete(feature: str):
    data = request.get_json() or {}
    item_id = data.get('Id')

    if feature not in mock_data:
        return jsonify(create_response(404, "Record not found"))

    items = mock_data[feature]
    original_len = len(items)
    mock_data[feature] = [i for i in items if i.get('Id') != item_id]

    if len(mock_data[feature]) == original_len:
        return jsonify(create_response(404, "Record not found"))

    return jsonify(create_response())


# Health check
@app.route('/health', methods=['GET'])
def health():
    return jsonify({
        'status': 'healthy',
        'timestamp': datetime.utcnow().isoformat(),
        'features': list(mock_data.keys())
    })


# List all mock data
@app.route('/mock/list', methods=['GET'])
def list_mock():
    return jsonify({
        feature: len(items) for feature, items in mock_data.items()
    })


# Add mock data
@app.route('/mock/<feature>/seed', methods=['POST'])
def seed_data(feature: str):
    data = request.get_json() or {}
    items = data.get('items', [])

    if feature not in mock_data:
        mock_data[feature] = []

    for item in items:
        if 'Id' not in item:
            existing_ids = [i.get('Id', 0) for i in mock_data[feature]]
            item['Id'] = max(existing_ids) + 1 if existing_ids else 1
        if 'CreatedOn' not in item:
            item['CreatedOn'] = datetime.utcnow().isoformat() + 'Z'
        mock_data[feature].append(item)

    return jsonify(create_response(data={'added': len(items)}))


# Clear mock data
@app.route('/mock/<feature>/clear', methods=['POST'])
def clear_data(feature: str):
    if feature in mock_data:
        count = len(mock_data[feature])
        mock_data[feature] = []
        return jsonify(create_response(data={'cleared': count}))
    return jsonify(create_response(404, "Feature not found"))


def seed_default_data():
    """Seed default mock data for common features."""
    mock_data['player-note'] = [
        {
            'Id': 1, 'WebId': 1, 'Name': 'VIP Player Note',
            'Amount': 1000.00, 'Currency': 'USD', 'Status': 1,
            'Remark': 'High value player', 'CreatedOn': '2025-01-08T10:00:00Z'
        },
        {
            'Id': 2, 'WebId': 1, 'Name': 'Regular Player Note',
            'Amount': 100.00, 'Currency': 'USD', 'Status': 1,
            'Remark': 'Regular activity', 'CreatedOn': '2025-01-07T10:00:00Z'
        }
    ]

    mock_data['customer-record'] = [
        {
            'Id': 1, 'WebId': 1, 'Username': 'user001',
            'Amount': 500.00, 'Currency': 'USD', 'Status': 1,
            'Type': 'Deposit', 'CreatedOn': '2025-01-08T10:00:00Z'
        }
    ]


def main():
    parser = argparse.ArgumentParser(description='API Mock Server')
    parser.add_argument('--port', '-p', type=int, default=3000, help='Port to run server')
    parser.add_argument('--host', '-H', default='0.0.0.0', help='Host to bind')
    parser.add_argument('--config', '-c', help='Path to mock config JSON')
    parser.add_argument('--no-seed', action='store_true', help='Skip default data seeding')
    parser.add_argument('--debug', '-d', action='store_true', help='Enable debug mode')

    args = parser.parse_args()

    if args.config:
        global mock_config
        mock_config = load_mock_config(args.config)

    if not args.no_seed:
        seed_default_data()

    print(f"\n{'='*60}")
    print("API Mock Server for Backoffice Fullstack")
    print(f"{'='*60}")
    print(f"Host: {args.host}")
    print(f"Port: {args.port}")
    print(f"Debug: {args.debug}")
    print(f"{'-'*60}")
    print("Endpoints:")
    print("  POST /api/{feature}/v2/get-list")
    print("  POST /api/{feature}/v2/get-by-id")
    print("  POST /api/{feature}/v2/create")
    print("  POST /api/{feature}/v2/update")
    print("  POST /api/{feature}/v2/delete")
    print("  GET  /health")
    print("  GET  /mock/list")
    print("  POST /mock/{feature}/seed")
    print("  POST /mock/{feature}/clear")
    print(f"{'='*60}\n")

    app.run(host=args.host, port=args.port, debug=args.debug)


if __name__ == '__main__':
    main()
