#!/usr/bin/env python3
"""
Pattern Report Generator for Backoffice Fullstack Skill
Generates markdown reports from pattern validation results.

Usage:
  python pattern-report.py --path /path/to/feature --layer all --output /tmp/report.md
"""

import argparse
import os
import sys
from datetime import datetime
from pathlib import Path


def generate_report(result, output_path: str):
    """Generate a markdown report from ValidationResult."""

    timestamp = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
    status = "PASSED" if result.violations_found == 0 else "FAILED"
    status_emoji = "✅" if status == "PASSED" else "❌"

    report = f"""# Pattern Validation Report

**Generated:** {timestamp}
**Layer:** {result.layer.upper()}
**Status:** {status_emoji} {status}

---

## Summary

| Metric | Value |
|--------|-------|
| Files Scanned | {result.files_scanned}/{result.total_files} |
| Patterns Found | {result.patterns_found} |
| Violations Found | {result.violations_found} |

---

"""

    if result.violations:
        report += """## Violations

| Pattern | File | Line | Issue |
|---------|------|------|-------|
"""
        for v in result.violations:
            file_short = Path(v.file_path).name
            report += f"| {v.pattern_name} | `{file_short}` | {v.line_number} | {v.message} |\n"

        report += "\n### Violation Details\n\n"
        for i, v in enumerate(result.violations, 1):
            report += f"""#### {i}. {v.pattern_name}

- **File:** `{v.file_path}:{v.line_number}`
- **Line:** `{v.line_content}`
- **Issue:** {v.message}

"""

    report += """---

## Patterns Detected

"""

    if result.matches:
        pattern_counts = {}
        pattern_files = {}
        for m in result.matches:
            pattern_counts[m.pattern_name] = pattern_counts.get(m.pattern_name, 0) + 1
            if m.pattern_name not in pattern_files:
                pattern_files[m.pattern_name] = set()
            pattern_files[m.pattern_name].add(Path(m.file_path).name)

        report += "| Pattern | Count | Files |\n"
        report += "|---------|-------|-------|\n"
        for pattern, count in sorted(pattern_counts.items()):
            files = ', '.join(list(pattern_files[pattern])[:3])
            if len(pattern_files[pattern]) > 3:
                files += f" (+{len(pattern_files[pattern]) - 3} more)"
            report += f"| {pattern} | {count} | {files} |\n"
    else:
        report += "*No patterns detected*\n"

    report += f"""
---

## Recommendations

"""

    if result.violations_found > 0:
        report += """### Fix Required

The following violations must be addressed before merging:

"""
        violation_patterns = set(v.pattern_name for v in result.violations)
        for pattern in violation_patterns:
            count = sum(1 for v in result.violations if v.pattern_name == pattern)
            report += f"- [ ] Fix **{pattern}** ({count} violation{'s' if count > 1 else ''})\n"
    else:
        report += "No violations found. Code follows all established patterns.\n"

    report += f"""
---

*Report generated by Backoffice Fullstack Skill v3.2.0*
"""

    os.makedirs(os.path.dirname(output_path) if os.path.dirname(output_path) else '.', exist_ok=True)
    with open(output_path, 'w') as f:
        f.write(report)

    return report


def main():
    parser = argparse.ArgumentParser(
        description='Pattern Report Generator for Backoffice Fullstack Skill'
    )
    parser.add_argument(
        '--path', '-p',
        required=True,
        help='Path to scan (file or directory)'
    )
    parser.add_argument(
        '--layer', '-l',
        choices=['fe', 'be', 'db', 'all'],
        default='all',
        help='Layer to validate (fe, be, db, or all)'
    )
    parser.add_argument(
        '--signatures', '-s',
        default=None,
        help='Path to pattern-signatures.json'
    )
    parser.add_argument(
        '--output', '-o',
        default='/tmp/pattern-report.md',
        help='Output path for report'
    )

    args = parser.parse_args()

    script_dir = Path(__file__).parent
    sys.path.insert(0, str(script_dir))

    from pattern_validator import PatternValidator

    signatures_path = args.signatures or str(script_dir.parent / 'pattern-detection' / 'pattern-signatures.json')

    if not os.path.exists(signatures_path):
        print(f"Error: Signatures file not found: {signatures_path}")
        sys.exit(1)

    if not os.path.exists(args.path):
        print(f"Error: Path not found: {args.path}")
        sys.exit(1)

    validator = PatternValidator(signatures_path)
    result = validator.validate(args.path, args.layer)

    report = generate_report(result, args.output)
    print(f"Report generated: {args.output}")

    print(f"\nSummary:")
    print(f"  Files: {result.files_scanned}")
    print(f"  Patterns: {result.patterns_found}")
    print(f"  Violations: {result.violations_found}")

    sys.exit(0 if result.violations_found == 0 else 1)


if __name__ == '__main__':
    main()
