#!/usr/bin/env python3
"""
Pattern Validator for Backoffice Fullstack Skill
Scans code against pattern-signatures.json and reports violations.

Usage:
  python pattern-validator.py --path /path/to/feature --layer fe
  python pattern-validator.py --path /path/to/feature --layer all
  python pattern-validator.py --path /path/to/feature --layer be --report
"""

import argparse
import json
import os
import re
import sys
from dataclasses import dataclass
from pathlib import Path
from typing import Dict, List, Optional
from datetime import datetime


@dataclass
class PatternMatch:
    pattern_name: str
    file_path: str
    line_number: int
    line_content: str
    match_type: str  # 'valid' or 'violation'
    message: str


@dataclass
class ValidationResult:
    layer: str
    total_files: int
    files_scanned: int
    patterns_found: int
    violations_found: int
    matches: List[PatternMatch]
    violations: List[PatternMatch]


class PatternValidator:
    def __init__(self, signatures_path: str):
        self.signatures_path = signatures_path
        self.signatures = self._load_signatures()

    def _load_signatures(self) -> dict:
        with open(self.signatures_path, 'r') as f:
            return json.load(f)

    def _get_file_extensions(self, layer: str) -> List[str]:
        extensions = {
            'fe': ['.ts', '.vue', '.tsx'],
            'be': ['.cs'],
            'db': ['.sql']
        }
        if layer == 'all':
            return [ext for exts in extensions.values() for ext in exts]
        return extensions.get(layer, [])

    def _get_patterns_for_layer(self, layer: str) -> Dict:
        layer_map = {
            'fe': 'frontend',
            'be': 'backend',
            'db': 'database'
        }
        patterns = self.signatures.get('patterns', {})
        if layer == 'all':
            return patterns
        layer_key = layer_map.get(layer)
        if layer_key and layer_key in patterns:
            return {layer_key: patterns[layer_key]}
        return {}

    def _scan_file(self, file_path: str, patterns: Dict) -> tuple:
        matches = []
        violations = []

        try:
            with open(file_path, 'r', encoding='utf-8', errors='ignore') as f:
                content = f.read()
                lines = content.split('\n')
        except Exception as e:
            print(f"Warning: Could not read {file_path}: {e}")
            return matches, violations

        for layer_name, layer_patterns in patterns.items():
            for pattern_id, pattern_info in layer_patterns.items():
                if not pattern_info.get('auto_detectable', False):
                    continue

                regex = pattern_info.get('regex', '')
                if not regex:
                    continue

                try:
                    compiled = re.compile(regex, re.MULTILINE | re.DOTALL)
                except re.error as e:
                    print(f"Warning: Invalid regex for {pattern_id}: {e}")
                    continue

                for match in compiled.finditer(content):
                    start_pos = match.start()
                    line_number = content[:start_pos].count('\n') + 1
                    line_content = lines[line_number - 1] if line_number <= len(lines) else ''

                    matches.append(PatternMatch(
                        pattern_name=pattern_info.get('name', pattern_id),
                        file_path=file_path,
                        line_number=line_number,
                        line_content=line_content.strip()[:100],
                        match_type='valid',
                        message=pattern_info.get('description', '')
                    ))

                violation_info = pattern_info.get('violation', {})
                if violation_info:
                    violation_regex = violation_info.get('pattern', '')
                    if violation_regex:
                        try:
                            violation_compiled = re.compile(violation_regex, re.MULTILINE)
                            for match in violation_compiled.finditer(content):
                                start_pos = match.start()
                                line_number = content[:start_pos].count('\n') + 1
                                line_content = lines[line_number - 1] if line_number <= len(lines) else ''

                                violations.append(PatternMatch(
                                    pattern_name=pattern_info.get('name', pattern_id),
                                    file_path=file_path,
                                    line_number=line_number,
                                    line_content=line_content.strip()[:100],
                                    match_type='violation',
                                    message=violation_info.get('message', 'Pattern violation detected')
                                ))
                        except re.error:
                            pass

        return matches, violations

    def validate(self, path: str, layer: str) -> ValidationResult:
        patterns = self._get_patterns_for_layer(layer)
        extensions = self._get_file_extensions(layer)

        all_matches = []
        all_violations = []
        files_scanned = 0
        total_files = 0

        path_obj = Path(path)
        if path_obj.is_file():
            files = [path_obj]
        else:
            files = []
            for ext in extensions:
                files.extend(path_obj.rglob(f'*{ext}'))

        total_files = len(files)

        for file_path in files:
            file_str = str(file_path)
            if any(skip in file_str for skip in ['node_modules', '.git', 'dist', 'build', 'bin', 'obj']):
                continue

            matches, violations = self._scan_file(file_str, patterns)
            all_matches.extend(matches)
            all_violations.extend(violations)
            files_scanned += 1

        return ValidationResult(
            layer=layer,
            total_files=total_files,
            files_scanned=files_scanned,
            patterns_found=len(all_matches),
            violations_found=len(all_violations),
            matches=all_matches,
            violations=all_violations
        )


def print_results(result: ValidationResult, verbose: bool = False):
    print(f"\n{'='*60}")
    print(f"Pattern Validation Report - Layer: {result.layer.upper()}")
    print(f"{'='*60}")
    print(f"Files scanned: {result.files_scanned}/{result.total_files}")
    print(f"Patterns found: {result.patterns_found}")
    print(f"Violations found: {result.violations_found}")

    if result.violations:
        print(f"\n{'!'*60}")
        print("VIOLATIONS:")
        print(f"{'!'*60}")
        for v in result.violations:
            print(f"\n[{v.pattern_name}]")
            print(f"  File: {v.file_path}:{v.line_number}")
            print(f"  Line: {v.line_content}")
            print(f"  Issue: {v.message}")

    if verbose and result.matches:
        print(f"\n{'-'*60}")
        print("PATTERNS FOUND:")
        print(f"{'-'*60}")
        pattern_counts = {}
        for m in result.matches:
            pattern_counts[m.pattern_name] = pattern_counts.get(m.pattern_name, 0) + 1

        for pattern, count in sorted(pattern_counts.items()):
            print(f"  {pattern}: {count} occurrences")

    status = "PASSED" if result.violations_found == 0 else "FAILED"
    print(f"\n{'='*60}")
    print(f"Status: {status}")
    print(f"{'='*60}\n")

    return result.violations_found == 0


def main():
    parser = argparse.ArgumentParser(
        description='Pattern Validator for Backoffice Fullstack Skill'
    )
    parser.add_argument(
        '--path', '-p',
        required=True,
        help='Path to scan (file or directory)'
    )
    parser.add_argument(
        '--layer', '-l',
        choices=['fe', 'be', 'db', 'all'],
        default='all',
        help='Layer to validate (fe, be, db, or all)'
    )
    parser.add_argument(
        '--signatures', '-s',
        default=None,
        help='Path to pattern-signatures.json'
    )
    parser.add_argument(
        '--verbose', '-v',
        action='store_true',
        help='Show detailed pattern matches'
    )
    parser.add_argument(
        '--report', '-r',
        action='store_true',
        help='Generate markdown report'
    )
    parser.add_argument(
        '--output', '-o',
        default='/tmp/pattern-report.md',
        help='Output path for report (with --report)'
    )

    args = parser.parse_args()

    script_dir = Path(__file__).parent.parent
    signatures_path = args.signatures or str(script_dir / 'pattern-detection' / 'pattern-signatures.json')

    if not os.path.exists(signatures_path):
        print(f"Error: Signatures file not found: {signatures_path}")
        sys.exit(1)

    if not os.path.exists(args.path):
        print(f"Error: Path not found: {args.path}")
        sys.exit(1)

    validator = PatternValidator(signatures_path)
    result = validator.validate(args.path, args.layer)

    success = print_results(result, args.verbose)

    if args.report:
        from pattern_report import generate_report
        generate_report(result, args.output)
        print(f"Report saved to: {args.output}")

    sys.exit(0 if success else 1)


if __name__ == '__main__':
    main()
