#!/usr/bin/env python3
"""
Feature Scaffolder for Backoffice Fullstack Skill
Creates complete feature folder structure with all files from templates.

Usage:
  python scaffold-feature.py --feature PlayerNote --module membership
  python scaffold-feature.py --feature PlayerNote --output /tmp/feature
  python scaffold-feature.py --feature PlayerNote --layers fe,be,db
"""

import argparse
import json
import os
import re
import shutil
import sys
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional


class FeatureScaffolder:
    def __init__(self, templates_dir: str, output_base: str):
        self.templates_dir = Path(templates_dir)
        self.output_base = Path(output_base)
        self.generated_files: List[str] = []

    def scaffold(
        self,
        feature: str,
        module: str,
        service: str = 'Coloris',
        layers: List[str] = None
    ) -> Dict:
        layers = layers or ['fe', 'be', 'db']

        # Create output directory structure
        feature_dir = self.output_base / feature.lower()
        os.makedirs(feature_dir, exist_ok=True)

        status = {
            'feature': feature,
            'module': module,
            'service': service,
            'created_at': datetime.utcnow().isoformat(),
            'layers': {}
        }

        if 'fe' in layers:
            status['layers']['fe'] = self._scaffold_fe(feature, module, feature_dir)

        if 'be' in layers:
            status['layers']['be'] = self._scaffold_be(feature, module, service, feature_dir)

        if 'db' in layers:
            status['layers']['db'] = self._scaffold_db(feature, service, feature_dir)

        # Create status.json
        status_path = feature_dir / 'status.json'
        with open(status_path, 'w') as f:
            json.dump(status, f, indent=2)
        self.generated_files.append(str(status_path))

        # Create API contract
        self._create_api_contract(feature, module, service, feature_dir)

        # Create README
        self._create_readme(feature, module, feature_dir)

        return status

    def _replace_placeholders(self, content: str, feature: str, module: str, service: str = 'Coloris') -> str:
        replacements = {
            '{Feature}': feature,
            '{feature}': feature.lower(),
            '{Module}': module.capitalize(),
            '{module}': module.lower(),
            '{Service}': service,
            '{service}': service.lower(),
            '{Action}': 'Action',
            '{Major}': '1',
            '{Minor}': '0',
            '{Patch}': '0'
        }
        for placeholder, value in replacements.items():
            content = content.replace(placeholder, value)
        return content

    def _copy_and_transform(
        self,
        template_name: str,
        output_path: Path,
        feature: str,
        module: str,
        service: str = 'Coloris'
    ) -> Optional[str]:
        template_path = self.templates_dir / template_name
        if not template_path.exists():
            return None

        with open(template_path, 'r') as f:
            content = f.read()

        content = self._replace_placeholders(content, feature, module, service)

        os.makedirs(output_path.parent, exist_ok=True)
        with open(output_path, 'w') as f:
            f.write(content)

        self.generated_files.append(str(output_path))
        return str(output_path)

    def _scaffold_fe(self, feature: str, module: str, feature_dir: Path) -> Dict:
        fe_dir = feature_dir / 'fe'
        os.makedirs(fe_dir, exist_ok=True)

        status = {
            'models': 'pending',
            'api_calling': 'pending',
            'apis': 'pending',
            'fake_api': 'pending',
            'composable': 'pending',
            'component': 'pending',
            'dialog': 'pending',
            'filter': 'pending',
            'validation': 'pending'
        }

        # Models
        if self._copy_and_transform(
            'fe-model-template.ts',
            fe_dir / 'models' / f'{feature.lower()}Model.ts',
            feature, module
        ):
            status['models'] = 'generated'

        # API Calling
        if self._copy_and_transform(
            'fe-api-calling-template.ts',
            fe_dir / 'api' / f'apiCalling-{feature.lower()}.ts',
            feature, module
        ):
            status['api_calling'] = 'generated'

        # Composable
        if self._copy_and_transform(
            'fe-composable-template.ts',
            fe_dir / 'composables' / f'use{feature}.ts',
            feature, module
        ):
            status['composable'] = 'generated'

        # Component
        if self._copy_and_transform(
            'fe-component-template.vue',
            fe_dir / 'views' / f'{feature}.vue',
            feature, module
        ):
            status['component'] = 'generated'

        # Dialog
        if self._copy_and_transform(
            'fe-dialog-template.vue',
            fe_dir / 'components' / f'{feature}Dialog.vue',
            feature, module
        ):
            status['dialog'] = 'generated'

        # Filter Form
        if self._copy_and_transform(
            'fe-filter-form-template.vue',
            fe_dir / 'components' / f'{feature}FilterForm.vue',
            feature, module
        ):
            status['filter'] = 'generated'

        # Form Validation
        if self._copy_and_transform(
            'fe-form-validation-template.ts',
            fe_dir / 'validation' / f'{feature.lower()}Validation.ts',
            feature, module
        ):
            status['validation'] = 'generated'

        return status

    def _scaffold_be(self, feature: str, module: str, service: str, feature_dir: Path) -> Dict:
        be_dir = feature_dir / 'be'
        os.makedirs(be_dir, exist_ok=True)

        status = {
            'model': 'pending',
            'repository': 'pending',
            'service': 'pending',
            'controller': 'pending',
            'test': 'pending'
        }

        # Model
        if self._copy_and_transform(
            'be-model-template.cs',
            be_dir / 'Models' / f'{feature}Model.cs',
            feature, module, service
        ):
            status['model'] = 'generated'

        # Repository
        if self._copy_and_transform(
            'be-repository-template.cs',
            be_dir / 'Repository' / f'{feature}Repository.cs',
            feature, module, service
        ):
            status['repository'] = 'generated'

        # Service
        if self._copy_and_transform(
            'be-service-template.cs',
            be_dir / 'Services' / f'{feature}Service.cs',
            feature, module, service
        ):
            status['service'] = 'generated'

        # Controller
        if self._copy_and_transform(
            'be-controller-template.cs',
            be_dir / 'Controllers' / f'{feature}Controller.cs',
            feature, module, service
        ):
            status['controller'] = 'generated'

        # Test
        if self._copy_and_transform(
            'be-test-template.cs',
            be_dir / 'Tests' / f'{feature}ControllerTests.cs',
            feature, module, service
        ):
            status['test'] = 'generated'

        return status

    def _scaffold_db(self, feature: str, service: str, feature_dir: Path) -> Dict:
        db_dir = feature_dir / 'db'
        os.makedirs(db_dir, exist_ok=True)

        status = {
            'table': 'pending',
            'sp_getlist': 'pending',
            'sp_getbyid': 'pending',
            'sp_create': 'pending',
            'sp_update': 'pending',
            'sp_delete': 'pending',
            'sp_search': 'pending',
            'sp_bulk': 'pending',
            'insertdata': 'pending'
        }

        sp_templates = [
            ('db-table-template.sql', 'Tables' / f'{feature}.sql', 'table'),
            ('db-sp-getlist-template.sql', 'StoredProcedures' / f'{service}_{feature}_GetList_1.0.0.sql', 'sp_getlist'),
            ('db-sp-getbyid-template.sql', 'StoredProcedures' / f'{service}_{feature}_GetById_1.0.0.sql', 'sp_getbyid'),
            ('db-sp-create-template.sql', 'StoredProcedures' / f'{service}_{feature}_Create_1.0.0.sql', 'sp_create'),
            ('db-sp-update-template.sql', 'StoredProcedures' / f'{service}_{feature}_Update_1.0.0.sql', 'sp_update'),
            ('db-sp-delete-template.sql', 'StoredProcedures' / f'{service}_{feature}_Delete_1.0.0.sql', 'sp_delete'),
            ('db-sp-search-template.sql', 'StoredProcedures' / f'{service}_{feature}_Search_1.0.0.sql', 'sp_search'),
            ('db-sp-bulk-template.sql', 'StoredProcedures' / f'{service}_{feature}_Bulk_1.0.0.sql', 'sp_bulk'),
            ('db-insertdata-template.sql', f'InsertData-{feature}.sql', 'insertdata'),
        ]

        for template, output, key in sp_templates:
            if self._copy_and_transform(
                template,
                db_dir / output,
                feature, feature.lower(), service
            ):
                status[key] = 'generated'

        return status

    def _create_api_contract(self, feature: str, module: str, service: str, feature_dir: Path):
        contract = {
            'feature': feature,
            'module': module,
            'service': service,
            'version': '1.0.0',
            'created_at': datetime.utcnow().isoformat(),
            'endpoints': [
                {
                    'name': f'Get{feature}List',
                    'method': 'POST',
                    'path': f'/api/{feature.lower()}/v2/get-list',
                    'request': {
                        'WebId': 'int',
                        'Page': 'int',
                        'RowCountPerPage': 'int',
                        'StartDate': 'string?',
                        'EndDate': 'string?'
                    },
                    'response': {
                        'List': f'List<{feature}Item>',
                        'TotalCount': 'int',
                        'MaxPage': 'int'
                    }
                },
                {
                    'name': f'Create{feature}',
                    'method': 'POST',
                    'path': f'/api/{feature.lower()}/v2/create',
                    'request': {
                        'Name': 'string',
                        'Amount': 'decimal',
                        'Status': 'int'
                    },
                    'response': {
                        'Id': 'int'
                    }
                }
            ]
        }

        contract_path = feature_dir / 'api-contract.json'
        with open(contract_path, 'w') as f:
            json.dump(contract, f, indent=2)
        self.generated_files.append(str(contract_path))

    def _create_readme(self, feature: str, module: str, feature_dir: Path):
        readme = f"""# {feature} Feature

**Module:** {module}
**Created:** {datetime.utcnow().strftime('%Y-%m-%d %H:%M')}

## Structure

```
{feature.lower()}/
├── fe/
│   ├── models/
│   ├── api/
│   ├── composables/
│   ├── views/
│   ├── components/
│   └── validation/
├── be/
│   ├── Models/
│   ├── Repository/
│   ├── Services/
│   ├── Controllers/
│   └── Tests/
├── db/
│   ├── Tables/
│   └── StoredProcedures/
├── api-contract.json
├── status.json
└── README.md
```

## Development Steps

1. **DB Layer**
   - Review and execute table creation
   - Review and execute stored procedures
   - Run InsertData entries

2. **BE Layer**
   - Copy Models to Coloris/Models
   - Copy Repository to Coloris/Repository
   - Copy Service to Coloris/Services
   - Copy Controller to Coloris/Controllers
   - Register DI in Startup.cs

3. **FE Layer**
   - Copy models to kirby/src/models/{module}/
   - Add methods to apiCalling.ts
   - Add wrappers to apis.ts
   - Copy composable to kirby/src/composables/
   - Copy component to kirby/src/views/{module}/
   - Add route in router/index.ts

## Testing

```bash
# Test BE API
python scripts/be-test-template.py --feature {feature.lower()} --base-url http://localhost

# Test FE
python scripts/fe-test-template.py --url http://localhost:8080/v2/{feature.lower()}
```
"""
        readme_path = feature_dir / 'README.md'
        with open(readme_path, 'w') as f:
            f.write(readme)
        self.generated_files.append(str(readme_path))


def main():
    parser = argparse.ArgumentParser(description='Feature Scaffolder')
    parser.add_argument('--feature', '-f', required=True, help='Feature name (PascalCase)')
    parser.add_argument('--module', '-m', default=None, help='Module name (lowercase)')
    parser.add_argument('--service', '-s', default='Coloris', help='Service name')
    parser.add_argument('--output', '-o', default='/tmp/scaffold', help='Output base directory')
    parser.add_argument('--layers', '-l', default='fe,be,db', help='Layers to scaffold (comma-separated)')

    args = parser.parse_args()

    script_dir = Path(__file__).parent.parent
    templates_dir = str(script_dir / 'templates')

    if not os.path.exists(templates_dir):
        print(f"Error: Templates directory not found: {templates_dir}")
        sys.exit(1)

    module = args.module or args.feature.lower()
    layers = [l.strip() for l in args.layers.split(',')]

    scaffolder = FeatureScaffolder(templates_dir, args.output)
    status = scaffolder.scaffold(args.feature, module, args.service, layers)

    print(f"\n{'='*60}")
    print(f"Feature Scaffold Complete")
    print(f"{'='*60}")
    print(f"Feature: {args.feature}")
    print(f"Module: {module}")
    print(f"Service: {args.service}")
    print(f"Output: {args.output}/{args.feature.lower()}")
    print(f"Files Generated: {len(scaffolder.generated_files)}")
    print(f"{'-'*60}")

    for layer, layer_status in status.get('layers', {}).items():
        print(f"\n{layer.upper()} Layer:")
        for component, state in layer_status.items():
            icon = "✓" if state == 'generated' else "○"
            print(f"  [{icon}] {component}: {state}")

    print(f"\n{'='*60}")
    print(f"Next Steps:")
    print(f"  1. Review generated files")
    print(f"  2. Copy to respective project directories")
    print(f"  3. Update status.json as you complete each step")
    print(f"{'='*60}\n")


if __name__ == '__main__':
    main()
