#!/usr/bin/env python3
"""
Stored Procedure Validator for Backoffice Fullstack Skill
Validates SQL stored procedures against naming conventions and patterns.

Usage:
  python sp-validator.py --path /path/to/sp.sql
  python sp-validator.py --path /path/to/stored-procedures --recursive
"""

import argparse
import os
import re
import sys
from dataclasses import dataclass
from pathlib import Path
from typing import List, Optional, Tuple


@dataclass
class SPValidationResult:
    file_path: str
    sp_name: Optional[str]
    is_valid: bool
    version: Optional[str]
    errors: List[str]
    warnings: List[str]


class SPValidator:
    # Pattern: Service_Feature_Action_Major.Minor.Patch
    SP_NAME_PATTERN = re.compile(
        r'CREATE\s+PROCEDURE\s+\[?dbo\]?\.\[?([A-Za-z]+_[A-Za-z]+_[A-Za-z]+_\d+\.\d+\.\d+)\]?',
        re.IGNORECASE
    )

    VERSION_PATTERN = re.compile(r'_(\d+\.\d+\.\d+)$')

    REQUIRED_PATTERNS = [
        (r'SET\s+NOCOUNT\s+ON', 'Missing SET NOCOUNT ON'),
        (r'BEGIN\s+TRY|TRY\s*\.\.\.\s*CATCH', 'Missing error handling (TRY...CATCH)'),
        (r'@WebId\s+INT|@WebID\s+INT', 'Missing @WebId parameter'),
    ]

    FORBIDDEN_PATTERNS = [
        (r'SELECT\s+\*\s+FROM', 'Avoid SELECT * - specify columns'),
        (r'EXEC\s*\(', 'Avoid dynamic SQL with EXEC()'),
        (r'PRINT\s+', 'Remove PRINT statements'),
    ]

    RECOMMENDED_PATTERNS = [
        (r'WITH\s*\(\s*NOLOCK\s*\)', 'Consider using NOLOCK for read operations'),
        (r'OFFSET.*FETCH\s+NEXT', 'Pagination pattern detected'),
        (r'ErrorCode.*ErrorMessage', 'Standard error response pattern'),
    ]

    def __init__(self):
        self.results: List[SPValidationResult] = []

    def validate_file(self, file_path: str) -> SPValidationResult:
        errors = []
        warnings = []
        sp_name = None
        version = None

        try:
            with open(file_path, 'r', encoding='utf-8') as f:
                content = f.read()
        except Exception as e:
            return SPValidationResult(
                file_path=file_path,
                sp_name=None,
                is_valid=False,
                version=None,
                errors=[f'Could not read file: {e}'],
                warnings=[]
            )

        # Extract SP name
        match = self.SP_NAME_PATTERN.search(content)
        if match:
            sp_name = match.group(1)
            version_match = self.VERSION_PATTERN.search(sp_name)
            if version_match:
                version = version_match.group(1)
            else:
                errors.append(f"SP name missing version suffix: {sp_name}")
        else:
            # Check if it's just missing version
            simple_pattern = re.search(
                r'CREATE\s+PROCEDURE\s+\[?dbo\]?\.\[?([A-Za-z_]+)\]?',
                content, re.IGNORECASE
            )
            if simple_pattern:
                found_name = simple_pattern.group(1)
                errors.append(f"SP name '{found_name}' doesn't follow pattern: Service_Feature_Action_X.X.X")
            else:
                errors.append("Could not find CREATE PROCEDURE statement")

        # Check required patterns
        for pattern, message in self.REQUIRED_PATTERNS:
            if not re.search(pattern, content, re.IGNORECASE):
                warnings.append(message)

        # Check forbidden patterns
        for pattern, message in self.FORBIDDEN_PATTERNS:
            if re.search(pattern, content, re.IGNORECASE):
                errors.append(message)

        # Check recommended patterns (informational)
        found_recommendations = []
        for pattern, message in self.RECOMMENDED_PATTERNS:
            if re.search(pattern, content, re.IGNORECASE):
                found_recommendations.append(message)

        # Validate parameter naming
        params = re.findall(r'@(\w+)\s+', content)
        for param in params:
            if not param[0].isupper():
                warnings.append(f"Parameter @{param} should start with uppercase")

        # Check for proper transaction handling
        has_transaction = bool(re.search(r'BEGIN\s+TRANSACTION', content, re.IGNORECASE))
        has_commit = bool(re.search(r'COMMIT\s+TRANSACTION', content, re.IGNORECASE))
        has_rollback = bool(re.search(r'ROLLBACK\s+TRANSACTION', content, re.IGNORECASE))

        if has_transaction and not (has_commit and has_rollback):
            warnings.append("Transaction started but missing COMMIT or ROLLBACK")

        # Check for IsDeleted handling (soft delete pattern)
        if 'DELETE' in content.upper():
            if not re.search(r'IsDeleted\s*=\s*1', content, re.IGNORECASE):
                warnings.append("Consider using soft delete (IsDeleted = 1) instead of hard delete")

        result = SPValidationResult(
            file_path=file_path,
            sp_name=sp_name,
            is_valid=len(errors) == 0,
            version=version,
            errors=errors,
            warnings=warnings
        )

        self.results.append(result)
        return result

    def validate_directory(self, dir_path: str, recursive: bool = True) -> List[SPValidationResult]:
        path = Path(dir_path)
        if recursive:
            files = list(path.rglob('*.sql'))
        else:
            files = list(path.glob('*.sql'))

        for file in files:
            self.validate_file(str(file))

        return self.results

    def print_results(self) -> bool:
        print(f"\n{'='*60}")
        print("Stored Procedure Validation Results")
        print(f"{'='*60}")
        print(f"Total Files: {len(self.results)}")

        valid = sum(1 for r in self.results if r.is_valid)
        invalid = len(self.results) - valid

        print(f"Valid: {valid}")
        print(f"Invalid: {invalid}")
        print(f"{'-'*60}")

        for result in self.results:
            file_name = Path(result.file_path).name
            status = "VALID" if result.is_valid else "INVALID"
            print(f"\n[{status}] {file_name}")

            if result.sp_name:
                print(f"  SP Name: {result.sp_name}")
            if result.version:
                print(f"  Version: {result.version}")

            if result.errors:
                print(f"  Errors:")
                for err in result.errors:
                    print(f"    - {err}")

            if result.warnings:
                print(f"  Warnings:")
                for warn in result.warnings:
                    print(f"    - {warn}")

        print(f"\n{'='*60}")
        status = "PASSED" if invalid == 0 else "FAILED"
        print(f"Status: {status}")
        print(f"{'='*60}\n")

        return invalid == 0

    def generate_report(self, output_path: str):
        valid = sum(1 for r in self.results if r.is_valid)
        invalid = len(self.results) - valid

        report = f"""# Stored Procedure Validation Report

**Status:** {'✅ PASSED' if invalid == 0 else '❌ FAILED'}

---

## Summary

| Metric | Value |
|--------|-------|
| Total SPs | {len(self.results)} |
| Valid | {valid} |
| Invalid | {invalid} |

---

## Naming Convention

```
Pattern: {{Service}}_{{Feature}}_{{Action}}_{{Major}}.{{Minor}}.{{Patch}}
Example: Coloris_PlayerNote_GetList_1.0.0
```

---

## Results

| File | SP Name | Version | Status |
|------|---------|---------|--------|
"""
        for r in self.results:
            file_name = Path(r.file_path).name
            status = "✅" if r.is_valid else "❌"
            sp_name = r.sp_name or "N/A"
            version = r.version or "N/A"
            report += f"| {file_name} | {sp_name[:30]} | {version} | {status} |\n"

        if invalid > 0:
            report += "\n---\n\n## Errors\n\n"
            for r in self.results:
                if r.errors:
                    file_name = Path(r.file_path).name
                    report += f"### {file_name}\n\n"
                    for err in r.errors:
                        report += f"- ❌ {err}\n"
                    report += "\n"

        os.makedirs(os.path.dirname(output_path) if os.path.dirname(output_path) else '.', exist_ok=True)
        with open(output_path, 'w') as f:
            f.write(report)


def main():
    parser = argparse.ArgumentParser(description='Stored Procedure Validator')
    parser.add_argument('--path', '-p', required=True, help='Path to SP file or directory')
    parser.add_argument('--recursive', '-r', action='store_true', help='Scan subdirectories')
    parser.add_argument('--output', '-o', default='/tmp/sp-validation-report.md', help='Output path for report')

    args = parser.parse_args()

    if not os.path.exists(args.path):
        print(f"Error: Path not found: {args.path}")
        sys.exit(1)

    validator = SPValidator()

    if os.path.isfile(args.path):
        validator.validate_file(args.path)
    else:
        validator.validate_directory(args.path, args.recursive)

    success = validator.print_results()
    validator.generate_report(args.output)
    print(f"Report saved to: {args.output}")

    sys.exit(0 if success else 1)


if __name__ == '__main__':
    main()
