#!/usr/bin/env python3
"""
Code Stub Generator for Backoffice Fullstack Skill
Generates code stubs from API contract JSON.

Usage:
  python stubgen.py --contract /path/to/api-contract.json --output /tmp/stubs
  python stubgen.py --contract /path/to/api-contract.json --layer fe --output /tmp/stubs
  python stubgen.py --feature PlayerNote --output /tmp/stubs
"""

import argparse
import json
import os
import re
import sys
from dataclasses import dataclass
from pathlib import Path
from typing import Dict, List, Optional
from datetime import datetime


@dataclass
class Endpoint:
    name: str
    method: str
    path: str
    request_fields: Dict[str, str]
    response_fields: Dict[str, str]


@dataclass
class ApiContract:
    feature: str
    module: str
    service: str
    endpoints: List[Endpoint]


class StubGenerator:
    def __init__(self, templates_dir: str):
        self.templates_dir = Path(templates_dir)
        self.generated_files: List[str] = []

    def load_contract(self, contract_path: str) -> ApiContract:
        with open(contract_path, 'r') as f:
            data = json.load(f)

        endpoints = []
        for ep in data.get('endpoints', []):
            endpoints.append(Endpoint(
                name=ep.get('name', ''),
                method=ep.get('method', 'POST'),
                path=ep.get('path', ''),
                request_fields=ep.get('request', {}),
                response_fields=ep.get('response', {})
            ))

        return ApiContract(
            feature=data.get('feature', 'Feature'),
            module=data.get('module', 'module'),
            service=data.get('service', 'Coloris'),
            endpoints=endpoints
        )

    def generate_all(self, contract: ApiContract, output_dir: str) -> List[str]:
        os.makedirs(output_dir, exist_ok=True)

        # Generate FE stubs
        self._generate_fe_model(contract, output_dir)
        self._generate_fe_api_calling(contract, output_dir)
        self._generate_fe_apis(contract, output_dir)
        self._generate_fe_fake_api(contract, output_dir)

        # Generate BE stubs
        self._generate_be_model(contract, output_dir)
        self._generate_be_repository(contract, output_dir)

        # Generate DB stubs
        self._generate_db_sp_getlist(contract, output_dir)
        self._generate_db_insertdata(contract, output_dir)

        return self.generated_files

    def _replace_placeholders(self, content: str, contract: ApiContract) -> str:
        replacements = {
            '{Feature}': contract.feature,
            '{feature}': contract.feature.lower(),
            '{Module}': contract.module,
            '{module}': contract.module.lower(),
            '{Service}': contract.service,
            '{service}': contract.service.lower(),
        }
        for placeholder, value in replacements.items():
            content = content.replace(placeholder, value)
        return content

    def _generate_fe_model(self, contract: ApiContract, output_dir: str):
        content = f"""// Auto-generated by stubgen.py - {datetime.now().strftime('%Y-%m-%d %H:%M')}

import {{ format }} from 'date-fns'
import {{ getValueForDisplay }} from '@/libraries/formatHelpers/numberFormatHelper'

export interface I{contract.feature}Request {{
  CustomerId: number
  StartDate: string
  EndDate: string
  Page: number
  RowCountPerPage: number
}}

export interface I{contract.feature}Response {{
  Id: number
  Name: string
  Amount: number
  Currency: string
  Status: number
  CreatedOn: string
}}

export class {contract.feature}Model {{
  Id: number
  Name: string
  Amount: number
  Currency: string
  Status: number
  CreatedOn: string

  constructor(data: I{contract.feature}Response) {{
    this.Id = data.Id
    this.Name = data.Name
    this.Amount = data.Amount
    this.Currency = data.Currency
    this.Status = data.Status
    this.CreatedOn = data.CreatedOn
  }}

  get AmountForDisplay(): string {{
    return getValueForDisplay(this.Amount, this.Currency)
  }}

  get StatusForDisplay(): string {{
    return this.Status === 1 ? 'Active' : 'Inactive'
  }}

  get CreatedOnForDisplay(): string {{
    return format(new Date(this.CreatedOn), 'yyyy-MM-dd HH:mm:ss')
  }}
}}

export interface ICreate{contract.feature}Request {{
  Name: string
  Amount: number
  Status: number
  Remark?: string
}}

export interface IUpdate{contract.feature}Request {{
  Id: number
  Name: string
  Amount: number
  Status: number
  Remark?: string
}}

export interface IDelete{contract.feature}Request {{
  Id: number
}}
"""
        file_path = os.path.join(output_dir, f'{contract.feature.lower()}Model.ts')
        with open(file_path, 'w') as f:
            f.write(content)
        self.generated_files.append(file_path)

    def _generate_fe_api_calling(self, contract: ApiContract, output_dir: str):
        feature_lower = contract.feature[0].lower() + contract.feature[1:]
        content = f"""// Auto-generated by stubgen.py - Add to apiCalling.ts

// {contract.feature} API Methods
callGet{contract.feature}List(request: I{contract.feature}Request) {{
  // return api.post('/api/{feature_lower}/v2/get-list', request)
  return FakeAPI.get{contract.feature}List(request)
}}

callGet{contract.feature}ById(request: {{ Id: number }}) {{
  // return api.post('/api/{feature_lower}/v2/get-by-id', request)
  return FakeAPI.get{contract.feature}ById(request)
}}

callCreate{contract.feature}(request: ICreate{contract.feature}Request) {{
  // return api.post('/api/{feature_lower}/v2/create', request)
  return FakeAPI.create{contract.feature}(request)
}}

callUpdate{contract.feature}(request: IUpdate{contract.feature}Request) {{
  // return api.post('/api/{feature_lower}/v2/update', request)
  return FakeAPI.update{contract.feature}(request)
}}

callDelete{contract.feature}(request: IDelete{contract.feature}Request) {{
  // return api.post('/api/{feature_lower}/v2/delete', request)
  return FakeAPI.delete{contract.feature}(request)
}}
"""
        file_path = os.path.join(output_dir, f'apiCalling-{contract.feature.lower()}.ts')
        with open(file_path, 'w') as f:
            f.write(content)
        self.generated_files.append(file_path)

    def _generate_fe_apis(self, contract: ApiContract, output_dir: str):
        content = f"""// Auto-generated by stubgen.py - Add to apis.ts

// {contract.feature} API Wrappers
get{contract.feature}List(request: I{contract.feature}Request) {{
  return getResponse(apiCalling.callGet{contract.feature}List(request))
}}

get{contract.feature}ById(request: {{ Id: number }}) {{
  return getResponse(apiCalling.callGet{contract.feature}ById(request))
}}

create{contract.feature}(request: ICreate{contract.feature}Request) {{
  return getResponse(apiCalling.callCreate{contract.feature}(request))
}}

update{contract.feature}(request: IUpdate{contract.feature}Request) {{
  return getResponse(apiCalling.callUpdate{contract.feature}(request))
}}

delete{contract.feature}(request: IDelete{contract.feature}Request) {{
  return getResponse(apiCalling.callDelete{contract.feature}(request))
}}
"""
        file_path = os.path.join(output_dir, f'apis-{contract.feature.lower()}.ts')
        with open(file_path, 'w') as f:
            f.write(content)
        self.generated_files.append(file_path)

    def _generate_fe_fake_api(self, contract: ApiContract, output_dir: str):
        content = f"""// Auto-generated by stubgen.py - Add to FakeAPI.ts

// {contract.feature} Fake Data
get{contract.feature}List(request: I{contract.feature}Request) {{
  return Promise.resolve({{
    ErrorCode: 0,
    ErrorMessage: 'Success',
    Data: {{
      List: [
        {{
          Id: 1,
          Name: 'Sample {contract.feature} 1',
          Amount: 100.00,
          Currency: 'USD',
          Status: 1,
          CreatedOn: new Date().toISOString()
        }},
        {{
          Id: 2,
          Name: 'Sample {contract.feature} 2',
          Amount: 200.00,
          Currency: 'USD',
          Status: 1,
          CreatedOn: new Date().toISOString()
        }}
      ],
      TotalCount: 2,
      MaxPage: 1
    }}
  }})
}}

get{contract.feature}ById(request: {{ Id: number }}) {{
  return Promise.resolve({{
    ErrorCode: 0,
    ErrorMessage: 'Success',
    Data: {{
      Id: request.Id,
      Name: 'Sample {contract.feature}',
      Amount: 100.00,
      Currency: 'USD',
      Status: 1,
      CreatedOn: new Date().toISOString()
    }}
  }})
}}

create{contract.feature}(request: ICreate{contract.feature}Request) {{
  return Promise.resolve({{
    ErrorCode: 0,
    ErrorMessage: 'Success',
    Data: {{ Id: 1 }}
  }})
}}

update{contract.feature}(request: IUpdate{contract.feature}Request) {{
  return Promise.resolve({{
    ErrorCode: 0,
    ErrorMessage: 'Success'
  }})
}}

delete{contract.feature}(request: IDelete{contract.feature}Request) {{
  return Promise.resolve({{
    ErrorCode: 0,
    ErrorMessage: 'Success'
  }})
}}
"""
        file_path = os.path.join(output_dir, f'FakeAPI-{contract.feature.lower()}.ts')
        with open(file_path, 'w') as f:
            f.write(content)
        self.generated_files.append(file_path)

    def _generate_be_model(self, contract: ApiContract, output_dir: str):
        content = f"""// Auto-generated by stubgen.py - {datetime.now().strftime('%Y-%m-%d %H:%M')}

namespace Coloris.Models.{contract.feature}
{{
    public class Get{contract.feature}ListRequest : BaseRequest
    {{
        public int CustomerId {{ get; set; }}
        public string StartDate {{ get; set; }}
        public string EndDate {{ get; set; }}
        public int Page {{ get; set; }} = 1;
        public int RowCountPerPage {{ get; set; }} = 25;

        public override ApiReturnError Validate()
        {{
            if (CustomerId <= 0) return ApiReturnError.InvalidCustomerId;
            return ApiReturnError.Success;
        }}
    }}

    public class Get{contract.feature}ListResponse : BaseResponse
    {{
        public List<{contract.feature}Item> List {{ get; set; }}
        public int TotalCount {{ get; set; }}
        public int MaxPage {{ get; set; }}
    }}

    public class {contract.feature}Item
    {{
        public int Id {{ get; set; }}
        public string Name {{ get; set; }}
        public decimal Amount {{ get; set; }}
        public string Currency {{ get; set; }}
        public int Status {{ get; set; }}
        public DateTime CreatedOn {{ get; set; }}
    }}

    public class Create{contract.feature}Request : BaseRequest
    {{
        public string Name {{ get; set; }}
        public decimal Amount {{ get; set; }}
        public int Status {{ get; set; }}
        public string Remark {{ get; set; }}

        public override ApiReturnError Validate()
        {{
            if (string.IsNullOrWhiteSpace(Name)) return ApiReturnError.NameRequired;
            if (Amount < 0) return ApiReturnError.InvalidAmount;
            return ApiReturnError.Success;
        }}
    }}

    public class Update{contract.feature}Request : Create{contract.feature}Request
    {{
        public int Id {{ get; set; }}

        public override ApiReturnError Validate()
        {{
            if (Id <= 0) return ApiReturnError.InvalidId;
            return base.Validate();
        }}
    }}

    public class Delete{contract.feature}Request : BaseRequest
    {{
        public int Id {{ get; set; }}

        public override ApiReturnError Validate()
        {{
            if (Id <= 0) return ApiReturnError.InvalidId;
            return ApiReturnError.Success;
        }}
    }}
}}
"""
        file_path = os.path.join(output_dir, f'{contract.feature}Model.cs')
        with open(file_path, 'w') as f:
            f.write(content)
        self.generated_files.append(file_path)

    def _generate_be_repository(self, contract: ApiContract, output_dir: str):
        content = f"""// Auto-generated by stubgen.py - {datetime.now().strftime('%Y-%m-%d %H:%M')}

using System.Threading.Tasks;
using System.Data;
using Dapper;
using Coloris.Models.{contract.feature};

namespace Coloris.Repository
{{
    public interface I{contract.feature}Repository
    {{
        Task<Get{contract.feature}ListResponse> GetListAsync(Get{contract.feature}ListRequest request);
        Task<BaseResponse> CreateAsync(Create{contract.feature}Request request);
        Task<BaseResponse> UpdateAsync(Update{contract.feature}Request request);
        Task<BaseResponse> DeleteAsync(Delete{contract.feature}Request request);
    }}

    public class {contract.feature}Repository : BaseRepository, I{contract.feature}Repository
    {{
        public async Task<Get{contract.feature}ListResponse> GetListAsync(Get{contract.feature}ListRequest request)
        {{
            var parameters = new
            {{
                request.WebId,
                request.CustomerId,
                request.StartDate,
                request.EndDate,
                request.Page,
                request.RowCountPerPage
            }};

            using var connection = GetConnection(DatabaseRole.MainDb);
            var result = await connection.QueryMultipleAsync(
                GetStoredProcedureName("{contract.service}_{contract.feature}_GetList"),
                parameters,
                commandType: CommandType.StoredProcedure);

            var items = result.Read<{contract.feature}Item>().ToList();
            var meta = result.ReadFirstOrDefault<ResponseMeta>();

            return new Get{contract.feature}ListResponse
            {{
                ErrorCode = meta?.ErrorCode ?? 0,
                ErrorMessage = meta?.ErrorMessage ?? "Success",
                List = items,
                TotalCount = meta?.TotalCount ?? 0,
                MaxPage = meta?.MaxPage ?? 1
            }};
        }}

        public async Task<BaseResponse> CreateAsync(Create{contract.feature}Request request)
        {{
            // Implementation
            throw new NotImplementedException();
        }}

        public async Task<BaseResponse> UpdateAsync(Update{contract.feature}Request request)
        {{
            // Implementation
            throw new NotImplementedException();
        }}

        public async Task<BaseResponse> DeleteAsync(Delete{contract.feature}Request request)
        {{
            // Implementation
            throw new NotImplementedException();
        }}
    }}
}}
"""
        file_path = os.path.join(output_dir, f'{contract.feature}Repository.cs')
        with open(file_path, 'w') as f:
            f.write(content)
        self.generated_files.append(file_path)

    def _generate_db_sp_getlist(self, contract: ApiContract, output_dir: str):
        content = f"""-- Auto-generated by stubgen.py - {datetime.now().strftime('%Y-%m-%d %H:%M')}

CREATE PROCEDURE [dbo].[{contract.service}_{contract.feature}_GetList_1.0.0]
    @WebId INT,
    @CustomerId INT = NULL,
    @StartDate DATETIME = NULL,
    @EndDate DATETIME = NULL,
    @Page INT = 1,
    @RowCountPerPage INT = 25
AS
BEGIN
    SET NOCOUNT ON;

    SELECT
        Id,
        Name,
        Amount,
        Currency,
        Status,
        CreatedOn,
        COUNT(1) OVER() AS TotalCount
    FROM [dbo].[{contract.feature}] WITH (NOLOCK)
    WHERE WebId = @WebId
        AND IsDeleted = 0
        AND (@CustomerId IS NULL OR CustomerId = @CustomerId)
        AND (@StartDate IS NULL OR CreatedOn >= @StartDate)
        AND (@EndDate IS NULL OR CreatedOn <= @EndDate)
    ORDER BY CreatedOn DESC
    OFFSET (@Page - 1) * @RowCountPerPage ROWS
    FETCH NEXT @RowCountPerPage ROWS ONLY;

    SELECT 0 AS ErrorCode, 'Success' AS ErrorMessage;
END
GO
"""
        file_path = os.path.join(output_dir, f'{contract.service}_{contract.feature}_GetList_1.0.0.sql')
        with open(file_path, 'w') as f:
            f.write(content)
        self.generated_files.append(file_path)

    def _generate_db_insertdata(self, contract: ApiContract, output_dir: str):
        content = f"""-- Auto-generated by stubgen.py - Add to InsertData.sql

-- {contract.feature} SP Registration
INSERT [dbo].[SimpleSettings]
    ([Website], [IsUAT], [Type], [Id], [Value], [Remark], [LastUpdatedOn], [LastUpdatedBy])
VALUES
    (N'{contract.service}', 0, N'sp_lookup', N'{contract.service}_{contract.feature}_GetList',
     N'[dbo].[{contract.service}_{contract.feature}_GetList_1.0.0]', N'{contract.feature} GetList', GETUTCDATE(), N'System')
GO

INSERT [dbo].[SimpleSettings]
    ([Website], [IsUAT], [Type], [Id], [Value], [Remark], [LastUpdatedOn], [LastUpdatedBy])
VALUES
    (N'{contract.service}', 0, N'sp_lookup', N'{contract.service}_{contract.feature}_Create',
     N'[dbo].[{contract.service}_{contract.feature}_Create_1.0.0]', N'{contract.feature} Create', GETUTCDATE(), N'System')
GO

INSERT [dbo].[SimpleSettings]
    ([Website], [IsUAT], [Type], [Id], [Value], [Remark], [LastUpdatedOn], [LastUpdatedBy])
VALUES
    (N'{contract.service}', 0, N'sp_lookup', N'{contract.service}_{contract.feature}_Update',
     N'[dbo].[{contract.service}_{contract.feature}_Update_1.0.0]', N'{contract.feature} Update', GETUTCDATE(), N'System')
GO

INSERT [dbo].[SimpleSettings]
    ([Website], [IsUAT], [Type], [Id], [Value], [Remark], [LastUpdatedOn], [LastUpdatedBy])
VALUES
    (N'{contract.service}', 0, N'sp_lookup', N'{contract.service}_{contract.feature}_Delete',
     N'[dbo].[{contract.service}_{contract.feature}_Delete_1.0.0]', N'{contract.feature} Delete', GETUTCDATE(), N'System')
GO
"""
        file_path = os.path.join(output_dir, f'InsertData-{contract.feature}.sql')
        with open(file_path, 'w') as f:
            f.write(content)
        self.generated_files.append(file_path)


def main():
    parser = argparse.ArgumentParser(description='Code Stub Generator')
    parser.add_argument('--contract', '-c', help='Path to API contract JSON')
    parser.add_argument('--feature', '-f', help='Feature name (generates default contract)')
    parser.add_argument('--output', '-o', default='/tmp/stubs', help='Output directory')
    parser.add_argument('--layer', '-l', choices=['fe', 'be', 'db', 'all'], default='all', help='Layer to generate')

    args = parser.parse_args()

    script_dir = Path(__file__).parent.parent
    templates_dir = str(script_dir / 'templates')

    generator = StubGenerator(templates_dir)

    if args.contract:
        contract = generator.load_contract(args.contract)
    elif args.feature:
        contract = ApiContract(
            feature=args.feature,
            module=args.feature.lower(),
            service='Coloris',
            endpoints=[]
        )
    else:
        print("Error: Provide --contract or --feature")
        sys.exit(1)

    files = generator.generate_all(contract, args.output)

    print(f"\n{'='*60}")
    print(f"Stub Generation Complete")
    print(f"{'='*60}")
    print(f"Feature: {contract.feature}")
    print(f"Output: {args.output}")
    print(f"Files Generated: {len(files)}")
    print(f"{'-'*60}")
    for f in files:
        print(f"  - {Path(f).name}")
    print(f"{'='*60}\n")


if __name__ == '__main__':
    main()
