#!/usr/bin/env python3
"""
Template Validation Script for Backoffice Fullstack Skill
Validates template syntax and placeholder consistency.

Usage:
  python validate-templates.py
  python validate-templates.py --templates /path/to/templates
  python validate-templates.py --fix  # Auto-fix common issues
"""

import argparse
import os
import re
import sys
from dataclasses import dataclass
from pathlib import Path
from typing import Dict, List, Set, Tuple


@dataclass
class ValidationResult:
    file_path: str
    is_valid: bool
    errors: List[str]
    warnings: List[str]
    placeholders: Set[str]


class TemplateValidator:
    PLACEHOLDER_PATTERN = re.compile(r'\{([A-Za-z]+)\}')

    EXPECTED_PLACEHOLDERS = {
        'Feature', 'feature', 'Module', 'module', 'Action', 'action',
        'Service', 'Major', 'Minor', 'Patch'
    }

    EXTENSION_RULES = {
        '.ts': {
            'required_patterns': [
                (r'export', 'Missing export statement'),
            ],
            'forbidden_patterns': [
                (r'console\.log\(', 'Remove console.log in production'),
            ]
        },
        '.vue': {
            'required_patterns': [
                (r'<template>', 'Missing <template> tag'),
                (r'<script', 'Missing <script> tag'),
            ],
            'forbidden_patterns': [
                (r'console\.log\(', 'Remove console.log in production'),
            ]
        },
        '.cs': {
            'required_patterns': [
                (r'namespace|using', 'Missing namespace or using statements'),
                (r'public class|public interface', 'Missing public class/interface'),
            ],
            'forbidden_patterns': [
                (r'Console\.WriteLine', 'Remove Console.WriteLine in production'),
            ]
        },
        '.sql': {
            'required_patterns': [
                (r'CREATE PROCEDURE|ALTER PROCEDURE', 'Missing procedure definition'),
                (r'SET NOCOUNT ON', 'Missing SET NOCOUNT ON'),
            ],
            'forbidden_patterns': [
                (r'SELECT \*', 'Avoid SELECT * - specify columns'),
            ]
        }
    }

    def __init__(self, templates_dir: str):
        self.templates_dir = Path(templates_dir)
        self.results: List[ValidationResult] = []

    def validate_all(self) -> List[ValidationResult]:
        template_files = []
        for ext in ['.ts', '.vue', '.cs', '.sql', '.json']:
            template_files.extend(self.templates_dir.glob(f'*{ext}'))

        for file_path in template_files:
            result = self.validate_file(str(file_path))
            self.results.append(result)

        return self.results

    def validate_file(self, file_path: str) -> ValidationResult:
        errors = []
        warnings = []
        placeholders = set()

        try:
            with open(file_path, 'r', encoding='utf-8') as f:
                content = f.read()
        except Exception as e:
            return ValidationResult(
                file_path=file_path,
                is_valid=False,
                errors=[f'Could not read file: {e}'],
                warnings=[],
                placeholders=set()
            )

        # Extract placeholders
        found_placeholders = self.PLACEHOLDER_PATTERN.findall(content)
        placeholders = set(found_placeholders)

        # Check for unknown placeholders
        unknown = placeholders - self.EXPECTED_PLACEHOLDERS
        if unknown:
            warnings.append(f"Unknown placeholders: {unknown}")

        # Check for unclosed placeholders
        if content.count('{') != content.count('}'):
            errors.append("Mismatched braces - possible unclosed placeholder")

        # Check file-specific rules
        ext = Path(file_path).suffix
        if ext in self.EXTENSION_RULES:
            rules = self.EXTENSION_RULES[ext]

            for pattern, message in rules.get('required_patterns', []):
                if not re.search(pattern, content):
                    errors.append(message)

            for pattern, message in rules.get('forbidden_patterns', []):
                if re.search(pattern, content):
                    warnings.append(message)

        # Check for common issues
        if '\t' in content and '    ' in content:
            warnings.append("Mixed tabs and spaces")

        if content.endswith('\n\n'):
            warnings.append("Trailing blank lines")

        if not content.endswith('\n'):
            warnings.append("File should end with newline")

        # Check for TODO/FIXME
        if 'TODO' in content or 'FIXME' in content:
            warnings.append("Contains TODO/FIXME comments")

        return ValidationResult(
            file_path=file_path,
            is_valid=len(errors) == 0,
            errors=errors,
            warnings=warnings,
            placeholders=placeholders
        )

    def print_results(self) -> bool:
        print(f"\n{'='*60}")
        print("Template Validation Results")
        print(f"{'='*60}")
        print(f"Templates Directory: {self.templates_dir}")
        print(f"Total Files: {len(self.results)}")

        valid = sum(1 for r in self.results if r.is_valid)
        invalid = len(self.results) - valid
        warnings_count = sum(len(r.warnings) for r in self.results)

        print(f"Valid: {valid}")
        print(f"Invalid: {invalid}")
        print(f"Warnings: {warnings_count}")
        print(f"{'-'*60}")

        for result in self.results:
            file_name = Path(result.file_path).name
            status = "VALID" if result.is_valid else "INVALID"
            print(f"\n[{status}] {file_name}")

            if result.placeholders:
                print(f"  Placeholders: {', '.join(sorted(result.placeholders))}")

            if result.errors:
                print(f"  Errors:")
                for err in result.errors:
                    print(f"    - {err}")

            if result.warnings:
                print(f"  Warnings:")
                for warn in result.warnings:
                    print(f"    - {warn}")

        print(f"\n{'='*60}")
        status = "PASSED" if invalid == 0 else "FAILED"
        print(f"Status: {status}")
        print(f"{'='*60}\n")

        return invalid == 0

    def generate_report(self, output_path: str):
        valid = sum(1 for r in self.results if r.is_valid)
        invalid = len(self.results) - valid

        report = f"""# Template Validation Report

**Templates:** {self.templates_dir}
**Status:** {'✅ PASSED' if invalid == 0 else '❌ FAILED'}

---

## Summary

| Metric | Value |
|--------|-------|
| Total Templates | {len(self.results)} |
| Valid | {valid} |
| Invalid | {invalid} |

---

## Results

| File | Status | Placeholders | Issues |
|------|--------|--------------|--------|
"""
        for r in self.results:
            file_name = Path(r.file_path).name
            status = "✅" if r.is_valid else "❌"
            placeholders = ', '.join(sorted(r.placeholders)[:3])
            if len(r.placeholders) > 3:
                placeholders += f" +{len(r.placeholders) - 3}"
            issues = len(r.errors) + len(r.warnings)
            report += f"| {file_name} | {status} | {placeholders} | {issues} |\n"

        if invalid > 0:
            report += "\n---\n\n## Errors\n\n"
            for r in self.results:
                if r.errors:
                    file_name = Path(r.file_path).name
                    report += f"### {file_name}\n\n"
                    for err in r.errors:
                        report += f"- ❌ {err}\n"
                    report += "\n"

        os.makedirs(os.path.dirname(output_path) if os.path.dirname(output_path) else '.', exist_ok=True)
        with open(output_path, 'w') as f:
            f.write(report)


def main():
    parser = argparse.ArgumentParser(description='Template Validation Script')
    parser.add_argument(
        '--templates', '-t',
        default=None,
        help='Path to templates directory'
    )
    parser.add_argument(
        '--output', '-o',
        default='/tmp/template-validation-report.md',
        help='Output path for report'
    )
    parser.add_argument(
        '--fix',
        action='store_true',
        help='Auto-fix common issues (trailing whitespace, etc.)'
    )

    args = parser.parse_args()

    script_dir = Path(__file__).parent.parent
    templates_dir = args.templates or str(script_dir / 'templates')

    if not os.path.exists(templates_dir):
        print(f"Error: Templates directory not found: {templates_dir}")
        sys.exit(1)

    validator = TemplateValidator(templates_dir)
    validator.validate_all()
    success = validator.print_results()
    validator.generate_report(args.output)
    print(f"Report saved to: {args.output}")

    sys.exit(0 if success else 1)


if __name__ == '__main__':
    main()
