/**
 * Backend Test Template for Coloris
 *
 * Framework: xUnit with Moq
 * Pattern: Arrange-Act-Assert
 *
 * Dependencies:
 *   - xUnit
 *   - Moq
 *   - FluentAssertions
 */

using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using Moq;
using Xunit;
using FluentAssertions;
using Coloris.Controllers;
using Coloris.Services;
using Coloris.Repository;
using Coloris.Models;
using Coloris.Models.Request;
using Coloris.Models.Response;

namespace Coloris.Tests.Controllers
{
    public class {Feature}ControllerTests
    {
        private readonly Mock<I{Feature}Service> _mockService;
        private readonly Mock<I{Feature}Repository> _mockRepository;
        private readonly {Feature}Controller _controller;

        public {Feature}ControllerTests()
        {
            _mockService = new Mock<I{Feature}Service>();
            _mockRepository = new Mock<I{Feature}Repository>();
            _controller = new {Feature}Controller(_mockService.Object);
        }

        #region GetList Tests

        [Fact]
        public async Task GetList_WithValidRequest_ReturnsSuccessResponse()
        {
            // Arrange
            var request = new Get{Feature}ListRequest
            {
                WebId = 1,
                Page = 1,
                RowCountPerPage = 25,
                StartDate = DateTime.UtcNow.AddDays(-7).ToString("yyyy-MM-dd"),
                EndDate = DateTime.UtcNow.ToString("yyyy-MM-dd")
            };

            var expectedData = new List<{Feature}Item>
            {
                new {Feature}Item { Id = 1, Name = "Test 1", Status = 1 },
                new {Feature}Item { Id = 2, Name = "Test 2", Status = 1 }
            };

            var expectedResponse = new Get{Feature}ListResponse
            {
                ErrorCode = 0,
                ErrorMessage = "Success",
                List = expectedData,
                TotalCount = 2,
                MaxPage = 1
            };

            _mockService
                .Setup(s => s.GetListAsync(It.IsAny<Get{Feature}ListRequest>()))
                .ReturnsAsync(expectedResponse);

            // Act
            var result = await _controller.GetList(request);

            // Assert
            result.Should().NotBeNull();
            result.ErrorCode.Should().Be(0);
            result.List.Should().HaveCount(2);
            result.TotalCount.Should().Be(2);
        }

        [Fact]
        public async Task GetList_WithInvalidPage_ReturnsValidationError()
        {
            // Arrange
            var request = new Get{Feature}ListRequest
            {
                WebId = 1,
                Page = 0, // Invalid
                RowCountPerPage = 25
            };

            // Act
            var result = await _controller.GetList(request);

            // Assert
            result.ErrorCode.Should().NotBe(0);
        }

        [Fact]
        public async Task GetList_WithEmptyResult_ReturnsEmptyList()
        {
            // Arrange
            var request = new Get{Feature}ListRequest
            {
                WebId = 1,
                Page = 1,
                RowCountPerPage = 25
            };

            var expectedResponse = new Get{Feature}ListResponse
            {
                ErrorCode = 0,
                ErrorMessage = "Success",
                List = new List<{Feature}Item>(),
                TotalCount = 0,
                MaxPage = 0
            };

            _mockService
                .Setup(s => s.GetListAsync(It.IsAny<Get{Feature}ListRequest>()))
                .ReturnsAsync(expectedResponse);

            // Act
            var result = await _controller.GetList(request);

            // Assert
            result.Should().NotBeNull();
            result.ErrorCode.Should().Be(0);
            result.List.Should().BeEmpty();
        }

        #endregion

        #region Create Tests

        [Fact]
        public async Task Create_WithValidRequest_ReturnsSuccessResponse()
        {
            // Arrange
            var request = new Create{Feature}Request
            {
                WebId = 1,
                Name = "New Feature",
                Amount = 100.00m,
                Status = 1
            };

            var expectedResponse = new Create{Feature}Response
            {
                ErrorCode = 0,
                ErrorMessage = "Success",
                Id = 1
            };

            _mockService
                .Setup(s => s.CreateAsync(It.IsAny<Create{Feature}Request>()))
                .ReturnsAsync(expectedResponse);

            // Act
            var result = await _controller.Create(request);

            // Assert
            result.Should().NotBeNull();
            result.ErrorCode.Should().Be(0);
            result.Id.Should().Be(1);
        }

        [Fact]
        public async Task Create_WithDuplicateName_ReturnsDuplicateError()
        {
            // Arrange
            var request = new Create{Feature}Request
            {
                WebId = 1,
                Name = "Existing Name",
                Amount = 100.00m,
                Status = 1
            };

            var expectedResponse = new Create{Feature}Response
            {
                ErrorCode = 330, // Duplicate exists
                ErrorMessage = "Duplicate entry exists"
            };

            _mockService
                .Setup(s => s.CreateAsync(It.IsAny<Create{Feature}Request>()))
                .ReturnsAsync(expectedResponse);

            // Act
            var result = await _controller.Create(request);

            // Assert
            result.ErrorCode.Should().Be(330);
        }

        [Fact]
        public async Task Create_WithEmptyName_ReturnsValidationError()
        {
            // Arrange
            var request = new Create{Feature}Request
            {
                WebId = 1,
                Name = "", // Invalid
                Amount = 100.00m,
                Status = 1
            };

            // Act
            var result = await _controller.Create(request);

            // Assert
            result.ErrorCode.Should().NotBe(0);
        }

        #endregion

        #region Update Tests

        [Fact]
        public async Task Update_WithValidRequest_ReturnsSuccessResponse()
        {
            // Arrange
            var request = new Update{Feature}Request
            {
                Id = 1,
                WebId = 1,
                Name = "Updated Name",
                Amount = 200.00m,
                Status = 1
            };

            var expectedResponse = new BaseResponse
            {
                ErrorCode = 0,
                ErrorMessage = "Success"
            };

            _mockService
                .Setup(s => s.UpdateAsync(It.IsAny<Update{Feature}Request>()))
                .ReturnsAsync(expectedResponse);

            // Act
            var result = await _controller.Update(request);

            // Assert
            result.ErrorCode.Should().Be(0);
        }

        [Fact]
        public async Task Update_WithNonExistentId_ReturnsNotFoundError()
        {
            // Arrange
            var request = new Update{Feature}Request
            {
                Id = 9999, // Non-existent
                WebId = 1,
                Name = "Updated Name"
            };

            var expectedResponse = new BaseResponse
            {
                ErrorCode = 404,
                ErrorMessage = "Record not found"
            };

            _mockService
                .Setup(s => s.UpdateAsync(It.IsAny<Update{Feature}Request>()))
                .ReturnsAsync(expectedResponse);

            // Act
            var result = await _controller.Update(request);

            // Assert
            result.ErrorCode.Should().Be(404);
        }

        #endregion

        #region Delete Tests

        [Fact]
        public async Task Delete_WithValidId_ReturnsSuccessResponse()
        {
            // Arrange
            var request = new Delete{Feature}Request
            {
                Id = 1,
                WebId = 1
            };

            var expectedResponse = new BaseResponse
            {
                ErrorCode = 0,
                ErrorMessage = "Success"
            };

            _mockService
                .Setup(s => s.DeleteAsync(It.IsAny<Delete{Feature}Request>()))
                .ReturnsAsync(expectedResponse);

            // Act
            var result = await _controller.Delete(request);

            // Assert
            result.ErrorCode.Should().Be(0);
        }

        [Fact]
        public async Task Delete_WithNonExistentId_ReturnsNotFoundError()
        {
            // Arrange
            var request = new Delete{Feature}Request
            {
                Id = 9999,
                WebId = 1
            };

            var expectedResponse = new BaseResponse
            {
                ErrorCode = 404,
                ErrorMessage = "Record not found"
            };

            _mockService
                .Setup(s => s.DeleteAsync(It.IsAny<Delete{Feature}Request>()))
                .ReturnsAsync(expectedResponse);

            // Act
            var result = await _controller.Delete(request);

            // Assert
            result.ErrorCode.Should().Be(404);
        }

        #endregion

        #region Repository Tests

        [Fact]
        public async Task Repository_GetList_CallsStoredProcedure()
        {
            // Arrange
            var request = new Get{Feature}ListRequest
            {
                WebId = 1,
                Page = 1,
                RowCountPerPage = 25
            };

            // Act & Assert
            _mockRepository
                .Setup(r => r.GetListAsync(It.IsAny<Get{Feature}ListRequest>()))
                .ReturnsAsync(new Get{Feature}ListResponse())
                .Verifiable();

            await _mockRepository.Object.GetListAsync(request);

            _mockRepository.Verify();
        }

        #endregion
    }

    #region Integration Tests

    public class {Feature}IntegrationTests : IClassFixture<TestWebApplicationFactory>
    {
        private readonly HttpClient _client;

        public {Feature}IntegrationTests(TestWebApplicationFactory factory)
        {
            _client = factory.CreateClient();
        }

        [Fact]
        public async Task GetList_Endpoint_ReturnsOk()
        {
            // Arrange
            var request = new Get{Feature}ListRequest
            {
                WebId = 1,
                Page = 1,
                RowCountPerPage = 25
            };

            // Act
            var response = await _client.PostAsJsonAsync("/api/{feature}/v2/get-list", request);

            // Assert
            response.EnsureSuccessStatusCode();
        }
    }

    #endregion
}

/**
 * Test Helper Classes
 */

// public class TestWebApplicationFactory : WebApplicationFactory<Program>
// {
//     protected override void ConfigureWebHost(IWebHostBuilder builder)
//     {
//         builder.UseEnvironment("Testing");
//         builder.ConfigureServices(services =>
//         {
//             // Replace real services with mocks for integration tests
//         });
//     }
// }
