/**
 * Database Stored Procedure Template - Bulk Operations
 *
 * Pattern: {Service}_{Feature}_Bulk{Action}_{Major}.{Minor}.{Patch}
 * Example: Coloris_Player_BulkUpdate_1.0.0
 *
 * Features:
 *   - Batch insert/update/delete
 *   - Table-valued parameter support
 *   - Transaction handling
 *   - Error reporting per record
 *   - Performance optimized
 */

-- =============================================
-- Create Table Type for Bulk Operations
-- =============================================

IF NOT EXISTS (SELECT * FROM sys.types WHERE name = '{Feature}BulkType')
BEGIN
    CREATE TYPE [dbo].[{Feature}BulkType] AS TABLE
    (
        [Id] INT NULL,
        [Name] NVARCHAR(100) NOT NULL,
        [Description] NVARCHAR(500) NULL,
        [Amount] DECIMAL(18, 2) NOT NULL DEFAULT 0,
        [Currency] NVARCHAR(10) NOT NULL DEFAULT 'USD',
        [Status] INT NOT NULL DEFAULT 1,
        [Remark] NVARCHAR(500) NULL,
        [RowIndex] INT NOT NULL  -- For error reporting
    )
END
GO

-- =============================================
-- Bulk Insert Procedure
-- =============================================

IF EXISTS (SELECT * FROM sys.objects WHERE type = 'P' AND name = 'Coloris_{Feature}_BulkCreate_1.0.0')
    DROP PROCEDURE [dbo].[Coloris_{Feature}_BulkCreate_1.0.0]
GO

CREATE PROCEDURE [dbo].[Coloris_{Feature}_BulkCreate_1.0.0]
    @WebId INT,
    @OperatorId INT,
    @OperatorName NVARCHAR(100),
    @Items [dbo].[{Feature}BulkType] READONLY,
    @SkipDuplicates BIT = 0  -- 0: Fail on duplicate, 1: Skip duplicates
AS
BEGIN
    SET NOCOUNT ON;
    SET XACT_ABORT ON;

    -- Validate inputs
    IF @WebId IS NULL OR @WebId <= 0
    BEGIN
        SELECT 400 AS ErrorCode, 'Invalid WebId' AS ErrorMessage, 0 AS SuccessCount, 0 AS FailedCount;
        RETURN;
    END

    IF NOT EXISTS (SELECT 1 FROM @Items)
    BEGIN
        SELECT 400 AS ErrorCode, 'No items provided' AS ErrorMessage, 0 AS SuccessCount, 0 AS FailedCount;
        RETURN;
    END

    DECLARE @SuccessCount INT = 0;
    DECLARE @FailedCount INT = 0;
    DECLARE @ErrorItems TABLE (RowIndex INT, ErrorMessage NVARCHAR(500));
    DECLARE @InsertedIds TABLE (Id INT, RowIndex INT);

    BEGIN TRY
        BEGIN TRANSACTION;

        -- Check for duplicates within input
        INSERT INTO @ErrorItems (RowIndex, ErrorMessage)
        SELECT i.RowIndex, 'Duplicate name within batch: ' + i.Name
        FROM @Items i
        INNER JOIN (
            SELECT Name
            FROM @Items
            GROUP BY Name
            HAVING COUNT(*) > 1
        ) dup ON i.Name = dup.Name;

        -- Check for existing records
        IF @SkipDuplicates = 0
        BEGIN
            INSERT INTO @ErrorItems (RowIndex, ErrorMessage)
            SELECT i.RowIndex, 'Name already exists: ' + i.Name
            FROM @Items i
            INNER JOIN [dbo].[{Feature}] t WITH (NOLOCK) ON t.Name = i.Name AND t.WebId = @WebId AND t.IsDeleted = 0
            WHERE i.RowIndex NOT IN (SELECT RowIndex FROM @ErrorItems);
        END

        -- Insert valid records
        INSERT INTO [dbo].[{Feature}]
            ([WebId], [Name], [Description], [Amount], [Currency], [Status], [Remark],
             [CreatedBy], [CreatedOn], [ModifiedBy], [ModifiedOn], [IsDeleted])
        OUTPUT inserted.Id, i.RowIndex INTO @InsertedIds
        SELECT
            @WebId,
            i.Name,
            i.Description,
            i.Amount,
            i.Currency,
            i.Status,
            i.Remark,
            @OperatorName,
            GETUTCDATE(),
            @OperatorName,
            GETUTCDATE(),
            0
        FROM @Items i
        WHERE i.RowIndex NOT IN (SELECT RowIndex FROM @ErrorItems)
            AND NOT EXISTS (
                SELECT 1 FROM [dbo].[{Feature}] t WITH (NOLOCK)
                WHERE t.Name = i.Name AND t.WebId = @WebId AND t.IsDeleted = 0 AND @SkipDuplicates = 1
            );

        SET @SuccessCount = @@ROWCOUNT;
        SET @FailedCount = (SELECT COUNT(*) FROM @ErrorItems);

        COMMIT TRANSACTION;

        -- Return summary
        SELECT 0 AS ErrorCode, 'Bulk insert completed' AS ErrorMessage,
               @SuccessCount AS SuccessCount, @FailedCount AS FailedCount;

        -- Return inserted IDs
        SELECT Id, RowIndex FROM @InsertedIds ORDER BY RowIndex;

        -- Return errors if any
        IF @FailedCount > 0
        BEGIN
            SELECT RowIndex, ErrorMessage FROM @ErrorItems ORDER BY RowIndex;
        END

    END TRY
    BEGIN CATCH
        IF @@TRANCOUNT > 0
            ROLLBACK TRANSACTION;

        SELECT
            ERROR_NUMBER() AS ErrorCode,
            ERROR_MESSAGE() AS ErrorMessage,
            0 AS SuccessCount,
            (SELECT COUNT(*) FROM @Items) AS FailedCount;
    END CATCH
END
GO

-- =============================================
-- Bulk Update Procedure
-- =============================================

IF EXISTS (SELECT * FROM sys.objects WHERE type = 'P' AND name = 'Coloris_{Feature}_BulkUpdate_1.0.0')
    DROP PROCEDURE [dbo].[Coloris_{Feature}_BulkUpdate_1.0.0]
GO

CREATE PROCEDURE [dbo].[Coloris_{Feature}_BulkUpdate_1.0.0]
    @WebId INT,
    @OperatorId INT,
    @OperatorName NVARCHAR(100),
    @Items [dbo].[{Feature}BulkType] READONLY
AS
BEGIN
    SET NOCOUNT ON;
    SET XACT_ABORT ON;

    DECLARE @SuccessCount INT = 0;
    DECLARE @FailedCount INT = 0;
    DECLARE @ErrorItems TABLE (RowIndex INT, ErrorMessage NVARCHAR(500));

    BEGIN TRY
        BEGIN TRANSACTION;

        -- Check for non-existent records
        INSERT INTO @ErrorItems (RowIndex, ErrorMessage)
        SELECT i.RowIndex, 'Record not found: Id=' + CAST(i.Id AS VARCHAR(20))
        FROM @Items i
        LEFT JOIN [dbo].[{Feature}] t WITH (NOLOCK) ON t.Id = i.Id AND t.WebId = @WebId AND t.IsDeleted = 0
        WHERE t.Id IS NULL AND i.Id IS NOT NULL;

        -- Check for duplicate names
        INSERT INTO @ErrorItems (RowIndex, ErrorMessage)
        SELECT i.RowIndex, 'Name already exists: ' + i.Name
        FROM @Items i
        INNER JOIN [dbo].[{Feature}] t WITH (NOLOCK) ON t.Name = i.Name AND t.WebId = @WebId AND t.IsDeleted = 0
        WHERE t.Id != i.Id
            AND i.RowIndex NOT IN (SELECT RowIndex FROM @ErrorItems);

        -- Update valid records
        UPDATE t
        SET
            t.Name = i.Name,
            t.Description = i.Description,
            t.Amount = i.Amount,
            t.Currency = i.Currency,
            t.Status = i.Status,
            t.Remark = i.Remark,
            t.ModifiedBy = @OperatorName,
            t.ModifiedOn = GETUTCDATE()
        FROM [dbo].[{Feature}] t
        INNER JOIN @Items i ON t.Id = i.Id
        WHERE t.WebId = @WebId
            AND t.IsDeleted = 0
            AND i.RowIndex NOT IN (SELECT RowIndex FROM @ErrorItems);

        SET @SuccessCount = @@ROWCOUNT;
        SET @FailedCount = (SELECT COUNT(*) FROM @ErrorItems);

        COMMIT TRANSACTION;

        SELECT 0 AS ErrorCode, 'Bulk update completed' AS ErrorMessage,
               @SuccessCount AS SuccessCount, @FailedCount AS FailedCount;

        IF @FailedCount > 0
        BEGIN
            SELECT RowIndex, ErrorMessage FROM @ErrorItems ORDER BY RowIndex;
        END

    END TRY
    BEGIN CATCH
        IF @@TRANCOUNT > 0
            ROLLBACK TRANSACTION;

        SELECT
            ERROR_NUMBER() AS ErrorCode,
            ERROR_MESSAGE() AS ErrorMessage,
            0 AS SuccessCount,
            (SELECT COUNT(*) FROM @Items) AS FailedCount;
    END CATCH
END
GO

-- =============================================
-- Bulk Delete Procedure (Soft Delete)
-- =============================================

IF EXISTS (SELECT * FROM sys.objects WHERE type = 'P' AND name = 'Coloris_{Feature}_BulkDelete_1.0.0')
    DROP PROCEDURE [dbo].[Coloris_{Feature}_BulkDelete_1.0.0]
GO

CREATE PROCEDURE [dbo].[Coloris_{Feature}_BulkDelete_1.0.0]
    @WebId INT,
    @OperatorId INT,
    @OperatorName NVARCHAR(100),
    @Ids NVARCHAR(MAX)  -- Comma-separated IDs: '1,2,3,4,5'
AS
BEGIN
    SET NOCOUNT ON;
    SET XACT_ABORT ON;

    DECLARE @SuccessCount INT = 0;
    DECLARE @FailedCount INT = 0;
    DECLARE @IdTable TABLE (Id INT);

    -- Parse comma-separated IDs
    INSERT INTO @IdTable (Id)
    SELECT CAST(value AS INT)
    FROM STRING_SPLIT(@Ids, ',')
    WHERE LEN(LTRIM(RTRIM(value))) > 0;

    IF NOT EXISTS (SELECT 1 FROM @IdTable)
    BEGIN
        SELECT 400 AS ErrorCode, 'No IDs provided' AS ErrorMessage, 0 AS SuccessCount, 0 AS FailedCount;
        RETURN;
    END

    BEGIN TRY
        BEGIN TRANSACTION;

        -- Soft delete records
        UPDATE t
        SET
            t.IsDeleted = 1,
            t.Status = 0,
            t.ModifiedBy = @OperatorName,
            t.ModifiedOn = GETUTCDATE()
        FROM [dbo].[{Feature}] t
        INNER JOIN @IdTable i ON t.Id = i.Id
        WHERE t.WebId = @WebId
            AND t.IsDeleted = 0;

        SET @SuccessCount = @@ROWCOUNT;
        SET @FailedCount = (SELECT COUNT(*) FROM @IdTable) - @SuccessCount;

        COMMIT TRANSACTION;

        SELECT 0 AS ErrorCode, 'Bulk delete completed' AS ErrorMessage,
               @SuccessCount AS SuccessCount, @FailedCount AS FailedCount;

    END TRY
    BEGIN CATCH
        IF @@TRANCOUNT > 0
            ROLLBACK TRANSACTION;

        SELECT
            ERROR_NUMBER() AS ErrorCode,
            ERROR_MESSAGE() AS ErrorMessage,
            0 AS SuccessCount,
            (SELECT COUNT(*) FROM @IdTable) AS FailedCount;
    END CATCH
END
GO

-- Grant execute permissions
GRANT EXECUTE ON [dbo].[Coloris_{Feature}_BulkCreate_1.0.0] TO [ColorisUser]
GRANT EXECUTE ON [dbo].[Coloris_{Feature}_BulkUpdate_1.0.0] TO [ColorisUser]
GRANT EXECUTE ON [dbo].[Coloris_{Feature}_BulkDelete_1.0.0] TO [ColorisUser]
GO

/**
 * InsertData.sql Registration:
 *
 * INSERT [dbo].[SimpleSettings] ([Website], [IsUAT], [Type], [Id], [Value], [Remark], [LastUpdatedOn], [LastUpdatedBy])
 * VALUES (N'Coloris', 0, N'sp_lookup', N'Coloris_{Feature}_BulkCreate', N'[dbo].[Coloris_{Feature}_BulkCreate_1.0.0]', N'{Feature} Bulk Create', GETUTCDATE(), N'System')
 * GO
 * INSERT [dbo].[SimpleSettings] ([Website], [IsUAT], [Type], [Id], [Value], [Remark], [LastUpdatedOn], [LastUpdatedBy])
 * VALUES (N'Coloris', 0, N'sp_lookup', N'Coloris_{Feature}_BulkUpdate', N'[dbo].[Coloris_{Feature}_BulkUpdate_1.0.0]', N'{Feature} Bulk Update', GETUTCDATE(), N'System')
 * GO
 * INSERT [dbo].[SimpleSettings] ([Website], [IsUAT], [Type], [Id], [Value], [Remark], [LastUpdatedOn], [LastUpdatedBy])
 * VALUES (N'Coloris', 0, N'sp_lookup', N'Coloris_{Feature}_BulkDelete', N'[dbo].[Coloris_{Feature}_BulkDelete_1.0.0]', N'{Feature} Bulk Delete', GETUTCDATE(), N'System')
 * GO
 */

/**
 * C# Usage Example:
 *
 * // Create DataTable for bulk type
 * var dataTable = new DataTable();
 * dataTable.Columns.Add("Id", typeof(int));
 * dataTable.Columns.Add("Name", typeof(string));
 * dataTable.Columns.Add("Description", typeof(string));
 * dataTable.Columns.Add("Amount", typeof(decimal));
 * dataTable.Columns.Add("Currency", typeof(string));
 * dataTable.Columns.Add("Status", typeof(int));
 * dataTable.Columns.Add("Remark", typeof(string));
 * dataTable.Columns.Add("RowIndex", typeof(int));
 *
 * // Add rows
 * for (int i = 0; i < items.Count; i++)
 * {
 *     dataTable.Rows.Add(null, items[i].Name, items[i].Description,
 *                        items[i].Amount, items[i].Currency, items[i].Status,
 *                        items[i].Remark, i + 1);
 * }
 *
 * // Execute SP
 * var parameters = new DynamicParameters();
 * parameters.Add("@WebId", webId);
 * parameters.Add("@OperatorId", operatorId);
 * parameters.Add("@OperatorName", operatorName);
 * parameters.Add("@Items", dataTable.AsTableValuedParameter("{Feature}BulkType"));
 *
 * var result = await connection.QueryMultipleAsync("Coloris_{Feature}_BulkCreate_1.0.0", parameters, commandType: CommandType.StoredProcedure);
 */
