/**
 * Database Stored Procedure Template - Full-Text Search
 *
 * Pattern: {Service}_{Feature}_Search_{Major}.{Minor}.{Patch}
 * Example: Coloris_Player_Search_1.0.0
 *
 * Features:
 *   - Multi-column search
 *   - Fuzzy matching with LIKE
 *   - Optional exact match mode
 *   - Weighted relevance scoring
 *   - Pagination support
 */

-- Drop existing procedure if exists
IF EXISTS (SELECT * FROM sys.objects WHERE type = 'P' AND name = 'Coloris_{Feature}_Search_1.0.0')
    DROP PROCEDURE [dbo].[Coloris_{Feature}_Search_1.0.0]
GO

CREATE PROCEDURE [dbo].[Coloris_{Feature}_Search_1.0.0]
    @WebId INT,
    @SearchTerm NVARCHAR(100),
    @SearchType INT = 0,              -- 0: Contains, 1: StartsWith, 2: ExactMatch
    @SearchColumns NVARCHAR(500) = '', -- Comma-separated: 'Name,Description,Remark'
    @Status INT = NULL,
    @StartDate DATETIME = NULL,
    @EndDate DATETIME = NULL,
    @Page INT = 1,
    @RowCountPerPage INT = 25
AS
BEGIN
    SET NOCOUNT ON;

    -- Validate inputs
    IF @WebId IS NULL OR @WebId <= 0
    BEGIN
        SELECT 400 AS ErrorCode, 'Invalid WebId' AS ErrorMessage;
        RETURN;
    END

    IF @SearchTerm IS NULL OR LEN(LTRIM(RTRIM(@SearchTerm))) = 0
    BEGIN
        SELECT 400 AS ErrorCode, 'Search term is required' AS ErrorMessage;
        RETURN;
    END

    -- Prepare search pattern based on search type
    DECLARE @SearchPattern NVARCHAR(200);
    SET @SearchTerm = LTRIM(RTRIM(@SearchTerm));

    IF @SearchType = 0 -- Contains
        SET @SearchPattern = '%' + @SearchTerm + '%';
    ELSE IF @SearchType = 1 -- StartsWith
        SET @SearchPattern = @SearchTerm + '%';
    ELSE -- ExactMatch
        SET @SearchPattern = @SearchTerm;

    -- Calculate relevance score and return results
    ;WITH SearchResults AS (
        SELECT
            t.Id,
            t.WebId,
            t.Name,
            t.Description,
            t.Amount,
            t.Currency,
            t.Status,
            t.Remark,
            t.CreatedBy,
            t.CreatedOn,
            t.ModifiedBy,
            t.ModifiedOn,
            -- Calculate relevance score (higher = more relevant)
            CASE
                WHEN t.Name = @SearchTerm THEN 100                  -- Exact name match
                WHEN t.Name LIKE @SearchTerm + '%' THEN 80          -- Name starts with
                WHEN t.Name LIKE '%' + @SearchTerm + '%' THEN 60    -- Name contains
                WHEN t.Description LIKE @SearchTerm + '%' THEN 40   -- Description starts with
                WHEN t.Description LIKE '%' + @SearchTerm + '%' THEN 20 -- Description contains
                ELSE 10
            END AS RelevanceScore
        FROM [dbo].[{Feature}] t WITH (NOLOCK)
        WHERE t.WebId = @WebId
            AND t.IsDeleted = 0
            AND (
                -- Search in Name
                (CHARINDEX('Name', @SearchColumns) > 0 OR @SearchColumns = '')
                AND t.Name LIKE @SearchPattern
                OR
                -- Search in Description
                (CHARINDEX('Description', @SearchColumns) > 0 OR @SearchColumns = '')
                AND t.Description LIKE @SearchPattern
                OR
                -- Search in Remark
                (CHARINDEX('Remark', @SearchColumns) > 0)
                AND t.Remark LIKE @SearchPattern
            )
            AND (@Status IS NULL OR t.Status = @Status)
            AND (@StartDate IS NULL OR t.CreatedOn >= @StartDate)
            AND (@EndDate IS NULL OR t.CreatedOn <= @EndDate)
    ),
    TotalCount AS (
        SELECT COUNT(1) AS Total FROM SearchResults
    )
    SELECT
        sr.Id,
        sr.WebId,
        sr.Name,
        sr.Description,
        sr.Amount,
        sr.Currency,
        sr.Status,
        sr.Remark,
        sr.CreatedBy,
        sr.CreatedOn,
        sr.ModifiedBy,
        sr.ModifiedOn,
        sr.RelevanceScore,
        tc.Total AS TotalCount,
        CEILING(CAST(tc.Total AS FLOAT) / @RowCountPerPage) AS MaxPage,
        ROW_NUMBER() OVER (ORDER BY sr.RelevanceScore DESC, sr.CreatedOn DESC) AS RowNumber
    FROM SearchResults sr
    CROSS JOIN TotalCount tc
    ORDER BY sr.RelevanceScore DESC, sr.CreatedOn DESC
    OFFSET (@Page - 1) * @RowCountPerPage ROWS
    FETCH NEXT @RowCountPerPage ROWS ONLY;

    -- Return success
    SELECT 0 AS ErrorCode, 'Success' AS ErrorMessage;
END
GO

-- Grant execute permission
GRANT EXECUTE ON [dbo].[Coloris_{Feature}_Search_1.0.0] TO [ColorisUser]
GO

/**
 * InsertData.sql Registration:
 *
 * INSERT [dbo].[SimpleSettings]
 *   ([Website], [IsUAT], [Type], [Id], [Value], [Remark], [LastUpdatedOn], [LastUpdatedBy])
 * VALUES
 *   (N'Coloris', 0, N'sp_lookup', N'Coloris_{Feature}_Search',
 *    N'[dbo].[Coloris_{Feature}_Search_1.0.0]', N'{Feature} Search', GETUTCDATE(), N'System')
 * GO
 */

/**
 * Usage Examples:
 *
 * -- Search all columns containing 'john'
 * EXEC [dbo].[Coloris_{Feature}_Search_1.0.0]
 *   @WebId = 1,
 *   @SearchTerm = 'john',
 *   @SearchType = 0
 *
 * -- Search Name starting with 'VIP'
 * EXEC [dbo].[Coloris_{Feature}_Search_1.0.0]
 *   @WebId = 1,
 *   @SearchTerm = 'VIP',
 *   @SearchType = 1,
 *   @SearchColumns = 'Name'
 *
 * -- Exact match in Name only
 * EXEC [dbo].[Coloris_{Feature}_Search_1.0.0]
 *   @WebId = 1,
 *   @SearchTerm = 'Premium User',
 *   @SearchType = 2,
 *   @SearchColumns = 'Name'
 */
