/**
 * FE API Calling Template for Kirby
 *
 * IMPORTANT: ALL API methods go in ONE file: src/libraries/apiCalling.ts
 * DO NOT create separate feature API files!
 *
 * Pattern: Use FakeAPI for development, api.post for production
 */

// =============================================================
// ADD TO: src/libraries/apiCalling.ts
// =============================================================

import FakeAPI from '@/libraries/FakeAPI';
import { api, useRealApi } from '@/api/api';
import type {
  IGetFeatureListRequest,
  IGetFeatureListResponse,
  ICreateFeatureRequest,
  ICreateFeatureResponse,
} from '@/models/{feature}/{Feature}';

// Add these methods to the default export object in apiCalling.ts:

/**
 * Get feature list
 * Endpoint: POST /back-office/v2/{feature}/get-list
 */
callGetFeatureList(request: IGetFeatureListRequest): IAxiosPromise<IGetFeatureListResponse> {
  if (useRealApi) {
    return api.post('/back-office/v2/{feature}/get-list', request);
  }
  return FakeAPI.getFeatureList(request);
},

/**
 * Create feature
 * Endpoint: POST /back-office/v2/{feature}/create
 */
callCreateFeature(request: ICreateFeatureRequest): IAxiosPromise<ICreateFeatureResponse> {
  if (useRealApi) {
    return api.post('/back-office/v2/{feature}/create', request);
  }
  return FakeAPI.createFeature(request);
},


// =============================================================
// ADD TO: src/libraries/apis.ts
// =============================================================

import apiCalling from '@/libraries/apiCalling';
import { getResponse, ApiResponse } from '@/libraries/apiHelper';
import type {
  IGetFeatureListRequest,
  IGetFeatureListResponse,
} from '@/models/{feature}/{Feature}';

// Add these methods to the default export object in apis.ts:

getFeatureList(request: IGetFeatureListRequest): Promise<ApiResponse<IGetFeatureListResponse>> {
  return getResponse(apiCalling.callGetFeatureList(request));
},

createFeature(request: ICreateFeatureRequest): Promise<ApiResponse<ICreateFeatureResponse>> {
  return getResponse(apiCalling.callCreateFeature(request));
},


// =============================================================
// ADD TO: src/libraries/FakeAPI.ts
// =============================================================

import { fakeFeatureList } from '@/libraries/fakeData/fake{Feature}';

// Add these methods to the FakeAPI object:

getFeatureList(request: IGetFeatureListRequest): IAxiosPromise<IGetFeatureListResponse> {
  return Promise.resolve({
    data: {
      ErrorCode: 0,
      ErrorMessage: 'Success',
      Data: {
        List: fakeFeatureList,
        TotalCount: fakeFeatureList.length,
      },
    },
  }) as IAxiosPromise<IGetFeatureListResponse>;
},


// =============================================================
// CREATE: src/libraries/fakeData/fake{Feature}.ts
// =============================================================

import type { IFeatureItem } from '@/models/{feature}/{Feature}';

export const fakeFeatureList: IFeatureItem[] = [
  {
    Id: 1,
    Name: 'Sample Item 1',
    Amount: 1000,
    Currency: 'USD',
    Status: 'Active',
    CreatedOn: '2024-01-15T10:30:00',
  },
  {
    Id: 2,
    Name: 'Sample Item 2',
    Amount: 2500,
    Currency: 'USD',
    Status: 'Inactive',
    CreatedOn: '2024-01-14T09:00:00',
  },
];

export default fakeFeatureList;


/**
 * USAGE NOTES:
 *
 * 1. Replace {feature} with actual feature name (e.g., customerRecord)
 * 2. Replace {Feature} with PascalCase name (e.g., CustomerRecord)
 *
 * 3. Method naming:
 *    - apiCalling: call{Action}{Feature} (e.g., callGetCustomerRecordReport)
 *    - apis: {action}{Feature} (e.g., getCustomerRecordReport)
 *    - FakeAPI: {action}{Feature} (e.g., getCustomerRecordReport)
 *
 * 4. API route convention: /back-office/v2/{feature}/{action}
 *
 * 5. Composables use apis.methodName() NOT apiCalling.callMethodName()
 */
