import _ from 'lodash-es'
import { ref, reactive, onMounted, watch, computed } from 'vue'
import { format } from 'date-fns'
import apis from '@/libraries/apis'
import store from '@/store'
import EnumApiErrorCode from '@/models/enums/enumsApiErrorCode'
import EnumMessageType from '@/models/enums/enumMessageType'
import notificationHelper from '@/libraries/elementUiHelpers/notificationHelper'
import textFormatHelper from '@/libraries/formatHelpers/textFormatHelper'
import { t } from '@/libraries/vue-i18n'
import { IColumns, IColumnsDynamic } from '@/models/Columns'
import { ITableRow, multipleExportExcel } from '@/libraries/excelHelper'
import useToggleTableHeaderColumn from '@/composables/useToggleTableHeader'
import { useVariable } from './useVariable'
import {
  IFeatureRequest,
  IFeatureResponse,
  FeatureModel,
} from '@/models/feature/featureModel'

export default function useFeatureName() {
  const {
    isLoading,
    isProcessing,
    afterLoading,
    isExportable,
  } = useVariable('module/feature_name')

  const startDate = ref(format(textFormatHelper.getCompanyTodayTimeZone().start, 'yyyy-MM-dd HH:mm:ss'))
  const endDate = ref(format(textFormatHelper.getCompanyTodayTimeZone().end, 'yyyy-MM-dd HH:mm:ss'))

  const filterFormModel = reactive({
    StartDate: startDate.value,
    EndDate: endDate.value,
    FilterStatus: 'All',
    FilterUsername: '',
    FilterCurrency: ['all'],
    RowCountPerPage: 25,
    Page: 1,
    TotalPage: 1,
    OrderColumn: '',
    OrderBy: 'DESC',
  })

  const tableData = ref<FeatureModel[]>([])
  const exportData = ref<FeatureModel[]>([])
  const dataLoading = ref(false)
  const isExportLoading = ref(false)

  const columns = ref<IColumnsDynamic[]>([
    { label: 'no', value: 1, show: true, columnIndex: 1, isSortable: false, prop: 'RowNumber', width: 60, align: 'center' },
    { label: 'username', value: 2, show: true, columnIndex: 2, isSortable: false, prop: 'Username', width: 120, align: 'left' },
    { label: 'status', value: 3, show: true, columnIndex: 3, isSortable: false, prop: 'StatusForDisplay', width: 100, align: 'center' },
    { label: 'amount', value: 4, show: true, columnIndex: 4, isSortable: true, prop: 'AmountForDisplay', width: 120, align: 'right' },
    { label: 'currency', value: 5, show: true, columnIndex: 5, isSortable: false, prop: 'Currency', width: 80, align: 'center' },
    { label: 'created_on', value: 6, show: true, columnIndex: 6, isSortable: true, prop: 'CreatedOnForDisplay', width: 160, align: 'center' },
    { label: 'action', value: 7, show: true, columnIndex: 7, isSortable: false, prop: 'Action', width: 100, align: 'center', custom: 'custom' },
  ])

  const {
    storeTableColumnsToLocalStorage,
    setTableColumnsSettingFromStore,
    checkIsDefaultColumnsHidden,
  } = useToggleTableHeaderColumn()

  const createRequest = (): IFeatureRequest => ({
    CustomerId: Number(store.state.customerId),
    StartDate: filterFormModel.StartDate,
    EndDate: filterFormModel.EndDate,
    Status: filterFormModel.FilterStatus,
    Username: filterFormModel.FilterUsername,
    Currencies: filterFormModel.FilterCurrency.includes('all') ? [] : filterFormModel.FilterCurrency,
    RowCountPerPage: filterFormModel.RowCountPerPage,
    Page: filterFormModel.Page,
    OrderColumn: filterFormModel.OrderColumn,
    OrderBy: filterFormModel.OrderBy,
  })

  const getData = async () => {
    dataLoading.value = true
    try {
      const request = createRequest()
      const response = await apis.getFeatureData(request)

      if (response.ErrorCode === EnumApiErrorCode.Success) {
        tableData.value = response.Data.Items.map((item: IFeatureResponse) => new FeatureModel(item))
        filterFormModel.TotalPage = response.Data.MaxPage ?? 1
      } else {
        tableData.value = []
        notificationHelper.notification(response.Message, EnumMessageType.Error)
      }
    } catch (error) {
      console.error('getData error:', error)
      tableData.value = []
    } finally {
      dataLoading.value = false
    }
  }

  const onSearch = () => {
    filterFormModel.Page = 1
    getData()
  }

  const handleCurrentChange = () => {
    getData()
  }

  const handleSizeChange = () => {
    filterFormModel.Page = 1
    getData()
  }

  const handleSortChange = (column: string, order: string) => {
    filterFormModel.OrderColumn = column
    filterFormModel.OrderBy = order === 'ascending' ? 'ASC' : 'DESC'
    getData()
  }

  const onExport = async () => {
    isExportLoading.value = true
    try {
      const request = {
        ...createRequest(),
        RowCountPerPage: 65000,
        Page: 1,
      }
      const response = await apis.getFeatureData(request)

      if (response.ErrorCode === EnumApiErrorCode.Success) {
        exportData.value = response.Data.Items.map((item: IFeatureResponse) => new FeatureModel(item))
        exportToExcel()
      } else {
        notificationHelper.notification(response.Message, EnumMessageType.Error)
      }
    } catch (error) {
      console.error('onExport error:', error)
    } finally {
      isExportLoading.value = false
    }
  }

  const exportToExcel = () => {
    const visibleColumns = columns.value.filter((col) => col.show && col.prop !== 'Action')

    const headers: ITableRow = visibleColumns.map((col) => ({
      v: t(col.label),
    }))

    const rows: ITableRow[] = exportData.value.map((item) =>
      visibleColumns.map((col) => ({
        v: item[col.prop as keyof FeatureModel] ?? '',
        t: typeof item[col.prop as keyof FeatureModel] === 'number' ? 'n' : 's',
      }))
    )

    const tables = [[headers, ...rows]]

    multipleExportExcel({
      tables,
      title: `${t('feature_name')}_${format(new Date(), 'yyyyMMdd_HHmmss')}`,
    })
  }

  const initColumns = () => {
    setTableColumnsSettingFromStore({
      columns: columns.value,
      pageIndex: '1.1',
      isNestedPage: false,
    })
  }

  watch(
    () => columns,
    () => {
      storeTableColumnsToLocalStorage({
        columns: columns.value,
        pageIndex: '1.1',
      })
    },
    { deep: true }
  )

  onMounted(() => {
    initColumns()
    getData()
  })

  return {
    isLoading,
    isProcessing,
    isExportable,
    dataLoading,
    isExportLoading,
    filterFormModel,
    tableData,
    columns,
    onSearch,
    getData,
    handleCurrentChange,
    handleSizeChange,
    handleSortChange,
    onExport,
  }
}
