/**
 * Form Validation Template for Kirby
 *
 * Usage in composable:
 *   import { createFormRules, validators } from './fe-form-validation'
 *   const formRules = createFormRules()
 *
 * Usage in component:
 *   <el-form :rules="formRules" ...>
 */

import { t } from '@/libraries/vue-i18n'
import type { FormRules, FormItemRule } from 'element-plus'

// Custom validator functions
export const validators = {
  // Required field
  required: (message?: string): FormItemRule => ({
    required: true,
    message: message || t('field_required'),
    trigger: 'blur'
  }),

  // Required for select/dropdown
  requiredSelect: (message?: string): FormItemRule => ({
    required: true,
    message: message || t('please_select'),
    trigger: 'change'
  }),

  // String length range
  length: (min: number, max: number, message?: string): FormItemRule => ({
    min,
    max,
    message: message || t('length_between', { min, max }),
    trigger: 'blur'
  }),

  // Minimum number value
  minNumber: (min: number, message?: string): FormItemRule => ({
    type: 'number',
    min,
    message: message || t('min_value', { min }),
    trigger: 'blur'
  }),

  // Maximum number value
  maxNumber: (max: number, message?: string): FormItemRule => ({
    type: 'number',
    max,
    message: message || t('max_value', { max }),
    trigger: 'blur'
  }),

  // Number range
  numberRange: (min: number, max: number, message?: string): FormItemRule => ({
    type: 'number',
    min,
    max,
    message: message || t('value_between', { min, max }),
    trigger: 'blur'
  }),

  // Email format
  email: (message?: string): FormItemRule => ({
    type: 'email',
    message: message || t('invalid_email'),
    trigger: 'blur'
  }),

  // URL format
  url: (message?: string): FormItemRule => ({
    type: 'url',
    message: message || t('invalid_url'),
    trigger: 'blur'
  }),

  // Pattern match (regex)
  pattern: (regex: RegExp, message: string): FormItemRule => ({
    pattern: regex,
    message,
    trigger: 'blur'
  }),

  // Username format (alphanumeric, underscore)
  username: (message?: string): FormItemRule => ({
    pattern: /^[a-zA-Z0-9_]+$/,
    message: message || t('invalid_username_format'),
    trigger: 'blur'
  }),

  // Phone number format
  phone: (message?: string): FormItemRule => ({
    pattern: /^\+?[0-9]{8,15}$/,
    message: message || t('invalid_phone_format'),
    trigger: 'blur'
  }),

  // IP address format
  ipAddress: (message?: string): FormItemRule => ({
    pattern: /^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/,
    message: message || t('invalid_ip_format'),
    trigger: 'blur'
  }),

  // Positive integer
  positiveInteger: (message?: string): FormItemRule => ({
    pattern: /^[1-9]\d*$/,
    message: message || t('must_be_positive_integer'),
    trigger: 'blur'
  }),

  // Decimal number with max precision
  decimal: (precision: number, message?: string): FormItemRule => ({
    pattern: new RegExp(`^\\d+(\\.\\d{1,${precision}})?$`),
    message: message || t('max_decimal_places', { precision }),
    trigger: 'blur'
  }),

  // Custom validator function
  custom: (
    validator: (rule: any, value: any, callback: (error?: Error) => void) => void
  ): FormItemRule => ({
    validator,
    trigger: 'blur'
  }),

  // Async validator (e.g., check username availability)
  async: (
    asyncValidator: (value: any) => Promise<boolean>,
    message: string
  ): FormItemRule => ({
    asyncValidator: async (rule, value, callback) => {
      try {
        const isValid = await asyncValidator(value)
        if (isValid) {
          callback()
        } else {
          callback(new Error(message))
        }
      } catch {
        callback(new Error(message))
      }
    },
    trigger: 'blur'
  }),

  // Date range validation
  dateAfter: (targetField: string, message?: string): FormItemRule => ({
    validator: (rule, value, callback, source) => {
      const targetDate = source[targetField]
      if (value && targetDate && new Date(value) <= new Date(targetDate)) {
        callback(new Error(message || t('end_date_must_be_after_start')))
      } else {
        callback()
      }
    },
    trigger: 'change'
  }),

  // Confirm password match
  confirmPassword: (passwordField: string, message?: string): FormItemRule => ({
    validator: (rule, value, callback, source) => {
      if (value !== source[passwordField]) {
        callback(new Error(message || t('passwords_do_not_match')))
      } else {
        callback()
      }
    },
    trigger: 'blur'
  }),

  // Array min length
  arrayMinLength: (min: number, message?: string): FormItemRule => ({
    type: 'array',
    min,
    message: message || t('select_at_least', { min }),
    trigger: 'change'
  })
}

// Common field rule presets
export const fieldRules = {
  // Name field (required, 2-100 chars)
  name: [
    validators.required(t('name_required')),
    validators.length(2, 100, t('name_length_2_100'))
  ],

  // Username field
  username: [
    validators.required(t('username_required')),
    validators.length(4, 20, t('username_length_4_20')),
    validators.username()
  ],

  // Email field
  email: [
    validators.required(t('email_required')),
    validators.email()
  ],

  // Password field
  password: [
    validators.required(t('password_required')),
    validators.length(8, 32, t('password_length_8_32'))
  ],

  // Amount field (required, >= 0)
  amount: [
    validators.required(t('amount_required')),
    validators.minNumber(0, t('amount_min_0'))
  ],

  // Status select (required)
  status: [
    validators.requiredSelect(t('status_required'))
  ],

  // Remark/description (optional, max 500)
  remark: [
    validators.length(0, 500, t('remark_max_500'))
  ],

  // Phone field
  phone: [
    validators.required(t('phone_required')),
    validators.phone()
  ],

  // IP address field
  ipAddress: [
    validators.required(t('ip_required')),
    validators.ipAddress()
  ]
}

// Create form rules for {Feature}
export function create{Feature}FormRules(): FormRules {
  return {
    Name: fieldRules.name,
    Amount: fieldRules.amount,
    Status: fieldRules.status,
    Remark: fieldRules.remark
  }
}

// Create form rules for any feature (dynamic)
export function createFormRules(fields: Record<string, FormItemRule[]>): FormRules {
  return fields
}

// Example usage in composable:
//
// import { create{Feature}FormRules } from './fe-form-validation'
//
// export default function use{Feature}Form() {
//   const formRules = create{Feature}FormRules()
//   return { formRules }
// }
