/**
 * FE Model Template for Kirby
 * Following CustomerRecordReport pattern
 *
 * File: src/models/{feature}/{Feature}.ts
 */

import { getValueForDisplay, ValueForDisplay } from '@/models/valueForDisplay';

// =============================================================
// REQUEST INTERFACES
// =============================================================

export interface IGetFeatureListRequest {
  WebId: number;
  StartDate: string;
  EndDate: string;
  Page: number;
  RowCountPerPage: number;
  FilterUsername?: string;
  FilterStatus?: string;
  // Add other filter fields as needed
}

export interface ICreateFeatureRequest {
  WebId: number;
  OperatorId: number;
  Name: string;
  Status?: string;
  // Add other create fields
}

export interface IUpdateFeatureRequest {
  WebId: number;
  OperatorId: number;
  Id: number;
  Name: string;
  Status: string;
  // Add other update fields
}

export interface IDeleteFeatureRequest {
  WebId: number;
  OperatorId: number;
  Id: number;
}


// =============================================================
// RESPONSE INTERFACES
// =============================================================

export interface IGetFeatureListResponse {
  List: IFeatureItem[];
  TotalCount: number;
}

export interface ICreateFeatureResponse {
  Id: number;
}

export interface IUpdateFeatureResponse {
  // Usually empty for update
}

export interface IDeleteFeatureResponse {
  // Usually empty for delete
}


// =============================================================
// DATA CLASS WITH FORDISPLAY GETTERS
// =============================================================

/**
 * Feature item class with ForDisplay getters for formatted values
 * Use class (not interface) when you need computed/formatted display values
 */
export class IFeatureItem {
  Id: number = 0;
  Name: string = '';
  Amount: number = 0;
  Currency: string = '';
  Status: string = '';
  CreatedOn: string = '';
  CreatedBy: string = '';
  ModifiedOn: string = '';
  ModifiedBy: string = '';

  /**
   * ForDisplay getter for Amount with currency formatting
   * Returns { value: number, display: string, color: string }
   */
  get AmountForDisplay(): ValueForDisplay {
    return getValueForDisplay(this.Amount, this.Currency);
  }

  /**
   * ForDisplay getter for Status with readable text
   */
  get StatusForDisplay(): string {
    switch (this.Status) {
      case 'Active':
        return 'Active';
      case 'Inactive':
        return 'Inactive';
      case 'Pending':
        return 'Pending';
      default:
        return this.Status;
    }
  }

  /**
   * ForDisplay getter for date formatting
   */
  get CreatedOnForDisplay(): string {
    if (!this.CreatedOn) return '-';
    return new Date(this.CreatedOn).toLocaleString();
  }
}


// =============================================================
// SIMPLE INTERFACE (when no ForDisplay needed)
// =============================================================

/**
 * Use interface when no computed values needed
 */
export interface IFeatureSimple {
  Id: number;
  Name: string;
  Status: string;
}


// =============================================================
// ENUM DEFINITIONS (if needed)
// =============================================================

export enum EnumFeatureStatus {
  Active = 'Active',
  Inactive = 'Inactive',
  Pending = 'Pending',
}


/**
 * USAGE NOTES:
 *
 * 1. Use CLASS when you need ForDisplay getters (computed display values)
 * 2. Use INTERFACE when you just need type checking
 *
 * 3. Common ForDisplay patterns:
 *    - AmountForDisplay: Currency formatting with color
 *    - DateForDisplay: Date/time formatting
 *    - StatusForDisplay: Status code to readable text
 *
 * 4. Import ValueForDisplay from '@/models/valueForDisplay'
 *    for amount/currency display formatting
 */
