---
name: figma-provider
description: Download provider images from Figma and integrate into Picasso. Use this skill when the user wants to "download Figma provider images", "export provider assets from Figma", "integrate Figma design to Picasso", or "get provider images from Figma URL".
---

# Figma Provider Image Download Skill

This skill downloads game provider images from Figma designs and integrates them into the Picasso asset repository.

## Prerequisites

- Picasso project at `/Users/rithytep_1/projects/Nuxt/picasso`

## Figma API Token

```
FIGMA_ACCESS_TOKEN=figd_le6WsXSui6od8aRUP83H7K3Tw3r0UyIi3DIXb5nU
```

User: rithy.tep@techbodia.com (Rithy Tep)

## Workflow

### Step 1: Parse Figma URL

Extract file key and node ID from the Figma URL:

```
URL Format: https://www.figma.com/design/{FILE_KEY}/{TITLE}?node-id={NODE_ID}
Example: https://www.figma.com/design/A9PJuphIHcmReo08IThxO3/New-Provider?node-id=13507-59894

Extracted:
- fileKey: A9PJuphIHcmReo08IThxO3
- nodeId: 13507-59894 (URL format) → 13507:59894 (API format, replace - with :)
```

### Step 2: Validate Figma Token

Check if FIGMA_ACCESS_TOKEN is valid:

```bash
curl -s -H "X-Figma-Token: $FIGMA_ACCESS_TOKEN" \
  "https://api.figma.com/v1/me"
```

If response contains `{"status":403,"err":"Token expired"}`:
- Prompt user: "Your Figma token has expired. Please generate a new one:"
- Instructions: Go to Figma > Settings > Account > Personal access tokens > Generate new token
- Ask user to provide the new token
- Update token in MCP configuration

### Step 3: Get Node Structure

Fetch the artboard structure to find all child image nodes:

```bash
curl -s -H "X-Figma-Token: $FIGMA_ACCESS_TOKEN" \
  "https://api.figma.com/v1/files/{FILE_KEY}/nodes?ids={NODE_ID}"
```

Response contains `nodes` with children. Extract all child node IDs.

### Step 4: Export Images

Export all child nodes as PNG images:

```bash
curl -s -H "X-Figma-Token: $FIGMA_ACCESS_TOKEN" \
  "https://api.figma.com/v1/images/{FILE_KEY}?ids={NODE_IDS}&format=png&scale=2"
```

Response contains `images` map: `{ "nodeId": "https://..." }`

### Step 5: Download Images

Download each image URL and save with correct naming:

```bash
curl -o "{OUTPUT_FILE}" "{IMAGE_URL}"
```

### Step 6: Map to Picasso Structure

Map downloaded images to Picasso file names based on node names:

| Figma Node Name | Picasso File Name |
|-----------------|-------------------|
| `star`, `icon`, `main` | `star.png` |
| `logo`, `provider_logo` | `provider_logo_{provider_name}.png` |
| `label_1`, `template_1` | `label_product_list_feature_template_1.png` |
| `label_2` - `label_6` | `label_product_list_feature_template_2-6.png` |
| `normal`, `default` | `{provider_name}_normal.png` |
| `active`, `hover` | `{provider_name}_active.png` |
| `mobile`, `entry`, `china` | `china_{provider_name}.png` |
| `hover_logo`, `productHover` | `productHover_{provider_no_underscore}logo.png` |
| `gameicon` | `gameicons_{provider_no_underscore}.png` |

### Step 7: Integrate to Picasso

Create provider folder and copy images:

```bash
PROVIDER_NAME="new_provider"
PICASSO_DIR="/Users/rithytep_1/projects/Nuxt/picasso"

# Create source folder
mkdir -p "$PICASSO_DIR/$PROVIDER_NAME"

# Copy downloaded images to source folder with correct names
# Then run mapper script
cd "$PICASSO_DIR" && ./mapper_games_provider.sh
```

Or manually copy to all theme locations per the integration template.

## Image Naming Reference

Provider name conventions:
- `provider_name` = lowercase with underscores (e.g., `hacksaw_gaming`)
- `provider_no_underscore` = remove underscores (e.g., `hacksawgaming`)

## Destination Paths

See `references/picasso-template.md` for complete list of destination paths.

Key locations:
- `assets/theme/common/provider_icon/games/{provider_name}/`
- `assets/theme/common/provider_logos/{provider_name}.png`
- `assets/theme/light_flexible_1/mobile/images/entry/provider/`
- `assets/theme/light_flexible_1/desktop/images/productHover/`
- `assets/theme/black_modern_1/images/gameProviders/`
- `assets/theme/cool_88/desktop/logos/providers/`
- `assets/theme/bk8/desktop/logos/providers/`

## Fallback: Playwright Browser

If API fails, use Playwright MCP to:
1. Navigate to Figma URL: `mcp__playwright__browser_navigate`
2. Wait for canvas load: `mcp__playwright__browser_wait_for`
3. Take screenshots: `mcp__playwright__browser_take_screenshot`
4. Save with naming convention

## Example Usage

```
User: Download provider images from Figma for super_slots
      https://www.figma.com/design/ABC123/Provider-Design?node-id=100-200

Claude:
1. Parses URL → fileKey: ABC123, nodeId: 100:200
2. Validates token
3. Fetches node structure
4. Exports all child images
5. Downloads with correct names
6. Copies to Picasso structure
```
