# Integrating New Game Provider Images in Picasso

## Overview

This guide documents how to integrate new game provider images into the Picasso asset repository. Each provider requires multiple image assets placed in specific locations for different themes.

## Provider Naming Convention

- **Provider Name**: Use lowercase with underscores (e.g., `hacksaw_gaming`, `lucky_monaco`)
- **Provider Name No Underscore**: Remove underscores for certain paths (e.g., `hacksawgaming`, `luckymonaco`)

## Required Source Images

When integrating a new provider, prepare the following source images in a folder named after the provider:

```
{provider_name}/
├── china_{provider_name}.png                              # Mobile entry icon
├── star.png                                               # Main provider star/logo icon
├── provider_logo_{provider_name}.png                      # Provider logo
├── productHover_{provider_no_underscore}logo.png          # Desktop hover logo
├── productHover_activelco_{provider_no_underscore}logo.png # Desktop active hover logo
├── gameicons_{provider_no_underscore}.png                 # Game icons
├── {provider_name}_normal.png                             # Dropdown normal state
├── {provider_name}_active.png                             # Dropdown active state
└── label_product_list_feature_template_1.png              # Label template (1-6)
```

## Destination Paths

### 1. Provider Icon Games (Main folder)
```
assets/theme/common/provider_icon/games/{provider_name}/
├── star.png
├── float_bg.png                                    # Copy from existing provider
├── label.png                                       # Alias of template_1
├── label_product_list_feature_template_1.png
├── label_product_list_feature_template_2.png
├── label_product_list_feature_template_3.png
├── label_product_list_feature_template_4.png
├── label_product_list_feature_template_5.png
└── label_product_list_feature_template_6.png
```

### 2. Provider Logos
```
assets/theme/common/provider_logos/{provider_name}.png
assets/common/providers/logo/{provider_no_underscore}.png
```

### 3. Light Flexible Theme
```
assets/theme/light_flexible_1/mobile/images/entry/provider/{provider_no_underscore}.png
assets/theme/light_flexible_1/desktop/images/productHover/{provider_no_underscore}.png
assets/theme/light_flexible_1/desktop/images/productHover/{provider_no_underscore}logo.png
assets/theme/light_flexible_1/desktop/images/productHover/activeIco/{provider_no_underscore}logo.png
```

### 4. Black Modern Theme
```
assets/theme/black_modern_1/images/gameProviders/{provider_name}.png
assets/theme/black_modern_1/images/gamelcons/{provider_no_underscore}.png
assets/theme/black_modern_1/images/gameIcons/{provider_no_underscore}.png
```

### 5. Cool 88 & BK8 Themes
```
assets/theme/cool_88/desktop/logos/providers/provider_logo_{provider_name}.png
assets/theme/bk8/desktop/logos/providers/provider_logo_{provider_name}.png
```

### 6. Game Tab Dropdown
```
assets/theme/common/desktop/game_tab/dropdown/{provider_name}/
├── normal.png
└── active.png
```

### 7. Magic Style & Entry Center
```
assets/theme/common/provider_icon/magic-style/{provider_name}.png
assets/theme/common/entry/center/all/logo-center-{provider_name}.png
```

## Using the Mapper Script

The `mapper_games_provider.sh` script automates the file copying:

```bash
# 1. Create provider folder with source images
mkdir {provider_name}
# Add your images following the naming convention above

# 2. Run the mapper script
./mapper_games_provider.sh

# The script will:
# - Auto-detect the provider folder
# - Copy files to all required destinations
# - Create float_bg.png from reference
# - Create label.png alias
# - Clean up source folder
```

## Manual Integration Steps

If not using the script, follow these steps:

### Step 1: Create Provider Icon Folder
```bash
mkdir -p assets/theme/common/provider_icon/games/{provider_name}
```

### Step 2: Copy Label Templates
Copy `label_product_list_feature_template_1.png` through `template_6.png` to the provider folder.

### Step 3: Copy float_bg.png
```bash
cp assets/theme/common/provider_icon/games/lucky_monaco/float_bg.png \
   assets/theme/common/provider_icon/games/{provider_name}/float_bg.png
```

### Step 4: Create label.png Alias
```bash
cp assets/theme/common/provider_icon/games/{provider_name}/label_product_list_feature_template_1.png \
   assets/theme/common/provider_icon/games/{provider_name}/label.png
```

### Step 5: Copy to All Theme Locations
Copy the appropriate images to all destination paths listed above.

## Example: Integrating "super_slots" Provider

```bash
PROVIDER="super_slots"
PROVIDER_NO_UNDERSCORE="superslots"

# Create directories
mkdir -p assets/theme/common/provider_icon/games/$PROVIDER
mkdir -p assets/theme/common/desktop/game_tab/dropdown/$PROVIDER

# Copy main assets (assuming source images are ready)
cp source/star.png assets/theme/common/provider_icon/games/$PROVIDER/
cp source/provider_logo.png assets/theme/common/provider_logos/$PROVIDER.png
cp source/provider_logo.png assets/common/providers/logo/$PROVIDER_NO_UNDERSCORE.png

# Copy float_bg from reference
cp assets/theme/common/provider_icon/games/lucky_monaco/float_bg.png \
   assets/theme/common/provider_icon/games/$PROVIDER/

# Copy to theme-specific locations
cp source/mobile_icon.png assets/theme/light_flexible_1/mobile/images/entry/provider/$PROVIDER_NO_UNDERSCORE.png
cp source/star.png assets/theme/black_modern_1/images/gameProviders/$PROVIDER.png
# ... continue for all locations
```

## Verification Checklist

After integration, verify these files exist:

- [ ] `assets/theme/common/provider_icon/games/{provider_name}/star.png`
- [ ] `assets/theme/common/provider_icon/games/{provider_name}/float_bg.png`
- [ ] `assets/theme/common/provider_icon/games/{provider_name}/label.png`
- [ ] `assets/theme/common/provider_icon/games/{provider_name}/label_product_list_feature_template_1-6.png`
- [ ] `assets/theme/common/provider_logos/{provider_name}.png`
- [ ] `assets/theme/light_flexible_1/mobile/images/entry/provider/{provider_no_underscore}.png`
- [ ] `assets/theme/light_flexible_1/desktop/images/productHover/{provider_no_underscore}.png`
- [ ] `assets/theme/light_flexible_1/desktop/images/productHover/{provider_no_underscore}logo.png`
- [ ] `assets/theme/light_flexible_1/desktop/images/productHover/activeIco/{provider_no_underscore}logo.png`
- [ ] `assets/theme/black_modern_1/images/gameProviders/{provider_name}.png`
- [ ] `assets/theme/black_modern_1/images/gamelcons/{provider_no_underscore}.png`
- [ ] `assets/theme/cool_88/desktop/logos/providers/provider_logo_{provider_name}.png`
- [ ] `assets/theme/bk8/desktop/logos/providers/provider_logo_{provider_name}.png`
- [ ] `assets/theme/common/desktop/game_tab/dropdown/{provider_name}/normal.png`
- [ ] `assets/theme/common/desktop/game_tab/dropdown/{provider_name}/active.png`

## Image Specifications

| Image Type | Recommended Size | Format |
|------------|------------------|--------|
| star.png | 200x200 or as designed | PNG with transparency |
| provider_logo | 300x100 or as designed | PNG with transparency |
| mobile entry icon | 100x100 | PNG with transparency |
| productHover | 150x50 | PNG with transparency |
| label templates | As per template | PNG with transparency |
| dropdown icons | 24x24 or 32x32 | PNG with transparency |

## Notes

- All images should be PNG format with transparency
- Use consistent naming (lowercase, underscores)
- The `float_bg.png` is typically copied from an existing provider as reference
- Label templates follow a standard design - adapt the provider logo to fit each template style
