#!/bin/bash
# Build eclair Docker image from source
#
# Usage:
#   docker-build.sh [--source /path/to/eclair] [--tag tagname] [--arm64]
#   docker-build.sh --branch <branch>       # Build from a specific branch
#   docker-build.sh --pr <number>           # Build from a GitHub PR
#   docker-build.sh --commit <sha>          # Build from a specific commit
#
# Options:
#   --source PATH   Path to eclair source directory (default: attempts to find)
#   --tag NAME      Docker image tag (default: acinq/eclair:local)
#   --arm64         Build for ARM64 (Apple Silicon)
#   --pull          Pull pre-built image instead of building
#   --branch NAME   Checkout and build from a specific branch
#   --commit SHA    Checkout and build from a specific commit
#   --pr NUMBER     Checkout and build from a GitHub PR (uses gh CLI)
#   --remote NAME   Git remote to fetch from (default: origin)

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
ECLAIR_SOURCE=""
IMAGE_TAG="acinq/eclair:local"
ARM64=false
PULL=false
BRANCH=""
COMMIT=""
PR_NUMBER=""
REMOTE="origin"

# Parse arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --source)
            ECLAIR_SOURCE="$2"
            shift 2
            ;;
        --tag)
            IMAGE_TAG="$2"
            shift 2
            ;;
        --arm64)
            ARM64=true
            shift
            ;;
        --pull)
            PULL=true
            shift
            ;;
        --branch)
            BRANCH="$2"
            shift 2
            ;;
        --commit)
            COMMIT="$2"
            shift 2
            ;;
        --pr)
            PR_NUMBER="$2"
            shift 2
            ;;
        --remote)
            REMOTE="$2"
            shift 2
            ;;
        -h|--help)
            echo "Usage: docker-build.sh [options]"
            echo ""
            echo "Build eclair Docker image from source with optional branch/commit/PR checkout."
            echo ""
            echo "Options:"
            echo "  --source PATH   Path to eclair source directory"
            echo "  --tag NAME      Docker image tag (default: acinq/eclair:local)"
            echo "  --arm64         Build for ARM64 (Apple Silicon)"
            echo "  --pull          Pull pre-built image instead of building"
            echo ""
            echo "Git checkout options (mutually exclusive):"
            echo "  --branch NAME   Checkout and build from a specific branch"
            echo "  --commit SHA    Checkout and build from a specific commit"
            echo "  --pr NUMBER     Checkout and build from a GitHub PR (requires gh CLI)"
            echo "  --remote NAME   Git remote to fetch from (default: origin)"
            echo ""
            echo "Examples:"
            echo "  # Build from current source state"
            echo "  docker-build.sh"
            echo ""
            echo "  # Build from a specific branch"
            echo "  docker-build.sh --branch taproot-feature-bit"
            echo ""
            echo "  # Build from a GitHub PR"
            echo "  docker-build.sh --pr 3144"
            echo ""
            echo "  # Build from a specific commit"
            echo "  docker-build.sh --commit ea9c4ca8dc1403bca6c6dcbe9bc4f3bd81d76513"
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            exit 1
            ;;
    esac
done

if [ "$PULL" = true ]; then
    echo "Pulling pre-built eclair image..."
    docker pull acinq/eclair:latest
    echo "Done! Image available as acinq/eclair:latest"
    exit 0
fi

# Try to find eclair source
if [ -z "$ECLAIR_SOURCE" ]; then
    # Check common locations
    if [ -d "$HOME/codez/eclair" ]; then
        ECLAIR_SOURCE="$HOME/codez/eclair"
    elif [ -d "$HOME/src/eclair" ]; then
        ECLAIR_SOURCE="$HOME/src/eclair"
    elif [ -d "/Users/roasbeef/codez/eclair" ]; then
        ECLAIR_SOURCE="/Users/roasbeef/codez/eclair"
    else
        echo "Error: Could not find eclair source directory."
        echo "Please specify with --source /path/to/eclair"
        exit 1
    fi
fi

if [ ! -f "$ECLAIR_SOURCE/Dockerfile" ]; then
    echo "Error: Dockerfile not found in $ECLAIR_SOURCE"
    exit 1
fi

cd "$ECLAIR_SOURCE"

# Check for mutually exclusive git options
GIT_OPTIONS=0
[ -n "$BRANCH" ] && GIT_OPTIONS=$((GIT_OPTIONS + 1))
[ -n "$COMMIT" ] && GIT_OPTIONS=$((GIT_OPTIONS + 1))
[ -n "$PR_NUMBER" ] && GIT_OPTIONS=$((GIT_OPTIONS + 1))

if [ $GIT_OPTIONS -gt 1 ]; then
    echo "Error: --branch, --commit, and --pr are mutually exclusive"
    exit 1
fi

# Handle git checkout if requested
if [ $GIT_OPTIONS -gt 0 ]; then
    # Check for uncommitted changes
    if ! git diff --quiet || ! git diff --cached --quiet; then
        echo "Error: Uncommitted changes in $ECLAIR_SOURCE"
        echo "Please commit or stash your changes before switching branches."
        git status --short
        exit 1
    fi

    if [ -n "$PR_NUMBER" ]; then
        # Check if gh CLI is available
        if ! command -v gh &> /dev/null; then
            echo "Error: gh CLI is required for --pr option"
            echo "Install with: brew install gh"
            exit 1
        fi
        echo "Checking out PR #$PR_NUMBER..."
        gh pr checkout "$PR_NUMBER" --repo ACINQ/eclair
    elif [ -n "$BRANCH" ]; then
        echo "Fetching from $REMOTE and checking out branch: $BRANCH"
        git fetch "$REMOTE"
        # Try local branch first, then remote
        if git show-ref --verify --quiet "refs/heads/$BRANCH"; then
            git checkout "$BRANCH"
            git pull "$REMOTE" "$BRANCH" 2>/dev/null || true
        elif git show-ref --verify --quiet "refs/remotes/$REMOTE/$BRANCH"; then
            git checkout -B "$BRANCH" "$REMOTE/$BRANCH"
        else
            echo "Error: Branch '$BRANCH' not found locally or on $REMOTE"
            exit 1
        fi
    elif [ -n "$COMMIT" ]; then
        echo "Checking out commit: $COMMIT"
        git fetch "$REMOTE"
        git checkout "$COMMIT"
    fi
fi

# Show current git state
echo ""
echo "Building eclair Docker image"
echo "  Source: $ECLAIR_SOURCE"
echo "  Branch: $(git branch --show-current 2>/dev/null || echo 'detached HEAD')"
echo "  Commit: $(git rev-parse --short HEAD) - $(git log -1 --format='%s')"
echo "  Tag:    $IMAGE_TAG"
echo ""

if [ "$ARM64" = true ]; then
    # Check for ARM64 Dockerfile
    if [ -f "contrib/arm64v8.Dockerfile" ]; then
        echo "Building for ARM64 using contrib/arm64v8.Dockerfile..."
        docker build -f contrib/arm64v8.Dockerfile -t "$IMAGE_TAG" .
    else
        echo "Building for ARM64 using standard Dockerfile with platform flag..."
        docker build --platform linux/arm64 -t "$IMAGE_TAG" .
    fi
else
    docker build -t "$IMAGE_TAG" .
fi

echo ""
echo "Build complete! Image available as: $IMAGE_TAG"
echo ""
echo "To use this image with docker-compose, update the image field:"
echo "  image: $IMAGE_TAG"
