#!/bin/bash
# Start eclair development environment
#
# Usage:
#   docker-start.sh [options]
#
# Options:
#   --compose         Use docker-compose (default, includes bitcoind)
#   --standalone      Run eclair only (requires external bitcoind)
#   --network NET     Bitcoin network: regtest (default), testnet, mainnet
#   --api-password P  API password (default: devpassword)
#   --data-dir DIR    Data directory for eclair
#   --detach/-d       Run in background

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
TEMPLATES_DIR="$SCRIPT_DIR/../templates"

MODE="compose"
NETWORK="regtest"
API_PASSWORD="devpassword"
DATA_DIR=""
DETACH="-d"

# Parse arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --compose)
            MODE="compose"
            shift
            ;;
        --standalone)
            MODE="standalone"
            shift
            ;;
        --network)
            NETWORK="$2"
            shift 2
            ;;
        --api-password)
            API_PASSWORD="$2"
            shift 2
            ;;
        --data-dir)
            DATA_DIR="$2"
            shift 2
            ;;
        -d|--detach)
            DETACH="-d"
            shift
            ;;
        --foreground)
            DETACH=""
            shift
            ;;
        -h|--help)
            echo "Usage: docker-start.sh [options]"
            echo ""
            echo "Options:"
            echo "  --compose         Use docker-compose (default, includes bitcoind)"
            echo "  --standalone      Run eclair only (requires external bitcoind)"
            echo "  --network NET     Bitcoin network: regtest, testnet, mainnet (default: regtest)"
            echo "  --api-password P  API password (default: devpassword)"
            echo "  --data-dir DIR    Data directory for eclair"
            echo "  --detach/-d       Run in background (default)"
            echo "  --foreground      Run in foreground"
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            exit 1
            ;;
    esac
done

if [ "$MODE" = "compose" ]; then
    echo "Starting eclair + bitcoind stack with docker-compose..."
    cd "$TEMPLATES_DIR"

    # Check if eclair image exists, if not trigger build
    if ! docker images eclair:local --format "{{.Repository}}" | grep -q eclair; then
        echo "Building eclair from source (first time, ~5-10 minutes)..."
        docker-compose up $DETACH --build
    else
        docker-compose up $DETACH
    fi

    if [ -n "$DETACH" ]; then
        echo ""
        echo "Stack started in background."
        echo ""
        echo "Services:"
        echo "  bitcoind: localhost:18443 (RPC), localhost:18444 (P2P)"
        echo "  eclair:   localhost:8080 (API), localhost:9735 (Lightning)"
        echo ""
        echo "Check status:"
        echo "  docker exec eclair eclair-cli -p $API_PASSWORD getinfo"
        echo "  docker exec bitcoind bitcoin-cli -regtest -rpcuser=bitcoin -rpcpassword=bitcoin getblockchaininfo"
        echo ""
        echo "View logs:"
        echo "  docker logs -f eclair"
        echo "  docker logs -f bitcoind"
    fi
else
    echo "Starting standalone eclair container..."

    VOLUME_ARGS=""
    if [ -n "$DATA_DIR" ]; then
        mkdir -p "$DATA_DIR"
        VOLUME_ARGS="-v $DATA_DIR:/data"
    fi

    # Map network to bitcoind port
    case $NETWORK in
        mainnet)
            BITCOIND_PORT=8332
            ;;
        testnet)
            BITCOIND_PORT=18332
            ;;
        regtest)
            BITCOIND_PORT=18443
            ;;
        *)
            echo "Unknown network: $NETWORK"
            exit 1
            ;;
    esac

    docker run $DETACH --rm \
        --name eclair \
        -p 9735:9735 \
        -p 8080:8080 \
        $VOLUME_ARGS \
        -e "JAVA_OPTS=-Xmx512m \
            -Declair.chain=$NETWORK \
            -Declair.api.enabled=true \
            -Declair.api.password=$API_PASSWORD \
            -Declair.api.binding-ip=0.0.0.0 \
            -Declair.printToConsole" \
        acinq/eclair:latest

    if [ -n "$DETACH" ]; then
        echo ""
        echo "Eclair started in background."
        echo ""
        echo "Note: You need to configure bitcoind connection via eclair.conf"
        echo "Mount your config: -v /path/to/eclair.conf:/data/eclair.conf"
    fi
fi
