#!/bin/bash
# Wrapper for eclair API calls
#
# Usage:
#   eclair-cli.sh <command> [--param=value]...
#
# Examples:
#   eclair-cli.sh getinfo
#   eclair-cli.sh channels
#   eclair-cli.sh createinvoice --amountMsat=100000000 --description="test"
#   eclair-cli.sh payinvoice --invoice=lntb...
#
# Environment variables:
#   ECLAIR_API_URL      API URL (default: http://localhost:8080)
#   ECLAIR_API_PASSWORD API password (default: devpassword)
#   ECLAIR_CONTAINER    Container name (default: eclair)

set -e

# Configuration
API_URL="${ECLAIR_API_URL:-http://localhost:8080}"
API_PASSWORD="${ECLAIR_API_PASSWORD:-devpassword}"
CONTAINER="${ECLAIR_CONTAINER:-eclair}"
USE_DOCKER=false
SHORT_OUTPUT=false

# Check if we should use docker exec
if ! curl -s --connect-timeout 1 "$API_URL/getinfo" -u ":$API_PASSWORD" > /dev/null 2>&1; then
    # API not directly accessible, try via docker
    if docker ps --format '{{.Names}}' | grep -q "^${CONTAINER}$"; then
        USE_DOCKER=true
    fi
fi

# Parse options
while [[ $# -gt 0 ]]; do
    case $1 in
        -p)
            API_PASSWORD="$2"
            shift 2
            ;;
        -a)
            API_URL="$2"
            shift 2
            ;;
        -s)
            SHORT_OUTPUT=true
            shift
            ;;
        -h|--help)
            echo "Usage: eclair-cli.sh [options] <command> [--param=value]..."
            echo ""
            echo "Options:"
            echo "  -p PASSWORD   API password (default: devpassword)"
            echo "  -a URL        API URL (default: http://localhost:8080)"
            echo "  -s            Short output format (for channels)"
            echo ""
            echo "Commands:"
            echo "  getinfo, peers, connect, disconnect, open, close, channels,"
            echo "  createinvoice, payinvoice, usablebalances, globalbalance, etc."
            echo ""
            echo "Examples:"
            echo "  eclair-cli.sh getinfo"
            echo "  eclair-cli.sh createinvoice --amountMsat=100000000 --description=\"test\""
            echo "  eclair-cli.sh -s channels"
            exit 0
            ;;
        -*)
            # Check if it's a command parameter (--param=value)
            if [[ $1 == --* ]]; then
                break
            fi
            echo "Unknown option: $1"
            exit 1
            ;;
        *)
            break
            ;;
    esac
done

if [ $# -eq 0 ]; then
    echo "Error: No command specified"
    echo "Usage: eclair-cli.sh <command> [--param=value]..."
    exit 1
fi

COMMAND="$1"
shift

if [ "$USE_DOCKER" = true ]; then
    # Build command for docker exec
    ARGS="-p $API_PASSWORD"
    if [ "$SHORT_OUTPUT" = true ]; then
        ARGS="$ARGS -s"
    fi

    docker exec "$CONTAINER" eclair-cli $ARGS "$COMMAND" "$@"
else
    # Direct API call
    # Build payload from arguments
    PAYLOAD=""
    for arg in "$@"; do
        if [[ $arg == --* ]]; then
            PAYLOAD="$PAYLOAD --data-urlencode \"${arg:2}\""
        fi
    done

    # Add jq filter
    JQ_FILTER='if type=="object" and .error != null then .error else . end'

    if [ "$SHORT_OUTPUT" = true ]; then
        case $COMMAND in
            channels)
                JQ_FILTER="$JQ_FILTER | map({ nodeId, shortChannelId: .data.lastAnnouncement_opt.shortChannelId, channelId, state })"
                ;;
        esac
    fi

    eval curl --user ":$API_PASSWORD" --silent --show-error \
        -X POST \
        -H "Content-Type: application/x-www-form-urlencoded" \
        $PAYLOAD \
        "$API_URL/$COMMAND" | jq -r "$JQ_FILTER"
fi
